/*
  File: wff2smalltalk.c
  Authors: John Maloney,
           K.R. Sloan
  Last Modified: 2 December 1988
  Purpose: Converts a binary .wff image file into a Smalltalk "Form" file.
  Note: The .wff file must be of type "I" with one bit per pixel. You can
	use various .wff utilities (such as rgb2i, Thresh, Bits and DiffuseN)
        to covert into a suitable binary image file.

  A Smalltalk Form object is stored as follows:
    <ID><width><height><horizonalOffset><verticalOffset>[bits...]
  where:
    <ID> is the sixteen bit integer constant 1
    <width> and <height> are sixteen bit unsigned integers
    <horizonalOffset> and <verticalOffset> are sixteen bit signed integers
    [bits...] is the bitmap, packed as sixteen bit words and stored in
       row-major order. The width is rounded up to the nearest multiple
       of sixteen; the extra bits are ignored. The first row is the top
       scanline of the image.

  Because forms are stored top-to-bottom whereas .wff files are stored
  bottom-to-top, we buffer the entire bitmap.
*/
#include <stdio.h>
#include <strings.h>
#include "wff.h"

/* Smalltalk Form header */
unsigned char Header[10] =
 {                    /* Header for a Smalltalk Form file */
  0, 1,               /* [0,1] ID field = 1               */
  0, 0,               /* [2,3] unsigned width             */
  0, 0,               /* [4,5] unsigned height            */
  0, 0,               /* [6,7] signed horizontal offset   */
  0, 0                /* [8,9] signed vertical offset     */
 };

static char *RoutineName;

static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s <wffFile >formFile\n", RoutineName);
 }

static void
WriteSmalltalk(smalltalkFile, width, height, bitmap)
 FILE *smalltalkFile;
 int width, height;
 unsigned short *bitmap;
 {
  int columnCount = (width + 15) / 16;
  int column, row;
  unsigned short word;
  unsigned char outWord[2];

  /* write Smalltalk Form header */
  Header[2] = (unsigned char) (width >> 8);
  Header[3] = (unsigned char) (width & 0xff);
  Header[4] = (unsigned char) (height >> 8);
  Header[5] = (unsigned char) (height & 0xff);
  /* horizonal and vertical offsets are zero */
  Header[6] = (unsigned char) 0;
  Header[7] = (unsigned char) 0;
  Header[8] = (unsigned char) 0;
  Header[9] = (unsigned char) 0;

  if (sizeof(Header) != fwrite(&Header[0], 1, sizeof(Header), smalltalkFile))
   {
    fprintf(stderr, "%s: Error writing Smalltalk header\n",
                     RoutineName);
    exit(1);
   }

  /* write the bitmap, top row first */
  for (row = (height - 1); row >= 0; row--)
   {
    for (column = 0; column < columnCount; column++)
     {
      word = bitmap[(row * columnCount) + column];
      outWord[0] = (unsigned char) (word >> 8);
      outWord[1] = (unsigned char) (word & 0xff);
      fwrite(&outWord[0], 1, 2, smalltalkFile);
     }
   }
 }

static void
ReadBitMap(FBin, width, height, bitmap)
 FrameBufferType *FBin;
 int width, height;
 unsigned short *bitmap;
 {
  int columnCount = (width + 15) / 16;
  int row, column, bitCount, bit;
  unsigned short pixel;
  unsigned short inWord;

  /* read the bitmap */
  for (row = 0; row < height; row++)
   {
    for (column = 0; column < columnCount; column++)
    {
     /* read the next word in the scan line (row) by reading the next
      * N pixels, where N is 16 except at the end of a scan line */
     bitCount = width - (column * 16);
     bitCount = (bitCount > 16) ? 16 : bitCount;
     inWord = 0;
     for (bit = 0; bit < bitCount; bit++)
      {
       /* read the next pixel from the WFF file */
       if (FAILURE == NextPixelIn(FBin, &pixel))
        {
         fprintf(stderr, "Error in NextPixelIn\n" );
         (void) CloseFB(&FBin);
         exit(1);
        }
        pixel = (pixel == 0) ? 1 : 0;
        inWord = (inWord << 1) + pixel;
      }
      /* adjust the last word in the scanline */
      if (bitCount < 16)
       { inWord = inWord << (16 - bitCount); }
      bitmap[(row * columnCount) + column] = inWord;
     }
   }
 }

static void
Pass(fdIn, fdOut)
 FILE *fdIn, *fdOut;
 {
  FrameBufferType *FBin;
  char WhatBands[ValueLength];
  int  BitsPerBand;
  int Bottom, Left, Top, Right;
  int width, height, columnCount;
  int bitmapSize;
  unsigned short *bitmap;  /* points to a dynamically allocated bitmap */

  /* open the WFF file and start reading it */
  FBin = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                       return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void) CloseFB(&FBin); return; }

  /* make sure we know how to handle it */
  GetColorSystem(FBin, WhatBands, &BitsPerBand);
  if ((0 != strcmp(WhatBands, "I")) ||
      (FBin->BandsPerPixel != 1))
   {
    fprintf(stderr,
            "%s: can't deal with %s images with %d bits per band\n",
            RoutineName, WhatBands, FBin->BandsPerPixel);
     (void) CloseFB(&FBin);
     exit(1);
   }
  if (BitsPerBand != 1)
   {
    fprintf(stderr,
            "%s: can't deal with %d bits per band\n",
            RoutineName, BitsPerBand);
    (void) CloseFB(&FBin);
    exit(1);
   }

  /* find out how big the image is */
  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   {
    (void) CloseFB(&FBin);
    exit(1);
   }
  width = Right - Left + 1;
  height = Top - Bottom + 1;
  columnCount = (width + 15) / 16;

  /* allocate the bitmap */
  bitmapSize = columnCount * height * sizeof(unsigned short);
  bitmap = (unsigned short *) malloc(bitmapSize);
  if (NULL == bitmap)
   {
    fprintf(stderr, "Insufficient memory for scratch bitmap\n" );
    (void) CloseFB(&FBin);
    exit(1);
   }

  /* read the bitmap from the WFF file */
  fprintf(stderr, "reading in bitmap from WFF file...\n");
  ReadBitMap(FBin, width, height, bitmap);
  
  /* write the Smalltalk Form file */
  fprintf(stderr, "writing Smalltalk file...\n");
  WriteSmalltalk(fdOut, width, height, bitmap);

  /* free bitmap storage and close WFF file */       
  fprintf(stderr, "Done\n");
  free(bitmap);
  (void)CloseFB(&FBin);
 }

int main(argc,argv)
 int argc;
 char *argv[];
 {
  RoutineName = argv[0];
  if (argc != 1)  { usage(); exit (-1); }
  Pass(stdin,stdout);
  exit(0);
 }
