/*
  File: wff2sun.c
  Authors: B.G. Mahesh,
           K.R. Sloan
  Last Modified: 7 February 1991
  Purpose: convert a .wff file to Sun raster format

  NOTE: the Sun files have 8 bits per sample, and are either 8 or 24 bits
        per pixel.  

        the .wff files must be I or RGB.  If the BitsPerSample is less than
        8, then samples are extended to 8 bits (correctly!).  If the 
        BitsPerSample is greater than 8, the samples are truncated.   
 */

#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <sys/types.h>
#include "wff.h"
#include "rasterfile.h"

unsigned char *malloc();

int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is:\n\t%s [-h][-v]\n",RoutineName);
 } 

static int SunHeader[8];
CreateSunHeader(width, height, BandsPerPixel, depth)
 int width, height, BandsPerPixel, depth;
 {
  if (VERBOSE) fprintf(stderr,"CreateSunHeader(%d,%d,%d,%d)\n",
                                 width,height,BandsPerPixel,depth);
  SunHeader [0] = RAS_MAGIC;
  SunHeader [1] = width;
  SunHeader [2] = height;
  switch (BandsPerPixel)
   {
    case 1:     /* Monochrome */
            SunHeader [3] = 8;                          /* depth in bits */
            SunHeader [4] = width * height;             /* length in bytes */
            SunHeader [5] = RT_STANDARD;                /* ras_type */
            SunHeader [6] = RMT_NONE;                   /* ras_maptype */
            SunHeader [7] = 0;                          /* ras_maplength */
            break;
    case 3:    /* RGB */
            if (1 == depth)
             { /* produce an 8-bit colormapped image */
              SunHeader [3] = 8;                        /* depth in bits */
              SunHeader [4] = width * height;           /* length in bytes */
              SunHeader [5] = RT_STANDARD;              /* ras_type */
              SunHeader [6] = RMT_EQUAL_RGB;            /* ras_maptype */
              SunHeader [7] = 768;                      /* ras_maplength */
             }
            else if (3 == depth)
             { /* produce a 24-bit unmapped image */
              SunHeader [3] = 24;                       /* depth in bits */
              SunHeader [4] = width * height * 3;       /* length in bytes */
              SunHeader [5] = RT_STANDARD;              /* ras_type */
              SunHeader [6] = RMT_NONE;                 /* ras_maptype */
              SunHeader [7] = 0;                        /* ras_maplength */
             }
            else
             {
              fprintf(stderr,"%s: Sorry, I don't understand depth %d\n",
                      RoutineName, depth);
               exit(-1);
             } 

            break;
    default:  /* unknown */
            fprintf(stderr,"%s: Sorry, I don't understand BandsPerPixel %d\n",
                     RoutineName, BandsPerPixel);
            exit(-1);
   }

  if (VERBOSE)
   {
    fprintf(stderr,"MAGIC    %8x\n",SunHeader[0]);
    fprintf(stderr,"Width    %8d\n",SunHeader[1]);
    fprintf(stderr,"Height   %8d\n",SunHeader[2]);
    fprintf(stderr,"Depth    %8d\n",SunHeader[3]);
    fprintf(stderr,"Length   %8d\n",SunHeader[4]);
    fprintf(stderr,"RasType  %8d\n",SunHeader[5]);
    fprintf(stderr,"MapType  %8d\n",SunHeader[6]);
    fprintf(stderr,"MapLength%8d\n",SunHeader[7]);
   }
 }

static int SwapBytes=0;
static int SwapShorts=0;
static void WhichEndUp()
 {
  int magic;
  char *c;

  magic = 0x59a66a95; c = (char *)&magic;
  switch (*c)
    {
     case 0x59: SwapBytes =  0; SwapShorts =  0; break;
     case 0xa6: SwapBytes = -1; SwapShorts =  0; break;
     case 0x6a: SwapBytes =  0; SwapShorts = -1; break;
     case 0x95: SwapBytes = -1; SwapShorts = -1; break;
     default: fprintf(stderr,"%s: see Key Operator\n",RoutineName); exit(-1);
    }
 }

static void fput16(s,fd)
 unsigned short s;
 FILE *fd;
 {
  unsigned char c0,c1;

  c0 = s>>8; c1 = s & 0x00ff;
  fputc(c0,fd); fputc(c1,fd);
 } 
static void fput32(w,fd)
 unsigned int w;
 FILE *fd;
 {
  unsigned short s0,s1;

  s0 = w>>16; s1 = w & 0xffff;
  fput16(s0,fd); fput16(s1,fd);
 }

static void WriteSunHeader(fd)
 FILE *fd;
 {
  int i;

  for (i=0;i<8;i++) fput32(SunHeader[i],fd);
 }

static void WriteSunScanLine(SunScanLine, SunScanLineLength, fd)
 unsigned char *SunScanLine;
 int SunScanLineLength;
 FILE *fd;
 {
  unsigned short s;
  int i,j;
  
  if (SwapBytes)
   {
    for (i=0,j=1;j < SunScanLineLength; i +=2, j += 2)
     {
      fputc(SunScanLine[j], fd);
      fputc(SunScanLine[i], fd); 
     } 
   }
  else (void)fwrite(SunScanLine, 1, SunScanLineLength, fd);
 }

static void WriteSunColorMap(BitsPerBand,fd)
 int BitsPerBand;
 {
  int slot,red,green,blue,mask,sample,t,j;
  unsigned char RedCM[256], GreenCM[256], BlueCM[256];

  mask = (1 << BitsPerBand) - 1;
  for(slot=0;slot<256;slot++)
   {
    /* extract red, green, blue bits */
    sample = slot;
    blue  = sample & mask; sample = sample >> BitsPerBand;
    green = sample & mask; sample = sample >> BitsPerBand;
    red   = sample & mask; 
    /* expand read, green, blue to 8 bits */

    t = red << (8-BitsPerBand); red = 0;
    for(j=0;j<8;j++) red = (red >> BitsPerBand) | t;
    t = green << (8-BitsPerBand); green = 0;
    for(j=0;j<8;j++) green = (green >> BitsPerBand) | t;
    t = blue << (8-BitsPerBand); blue = 0;
    for(j=0;j<8;j++) blue = (blue >> BitsPerBand) | t;

    RedCM[slot] = red;
    GreenCM[slot] = green;
    BlueCM[slot] = blue;
   } 
  (void)fwrite(RedCM  , 1, 256, fd);
  (void)fwrite(GreenCM, 1, 256, fd);
  (void)fwrite(BlueCM , 1, 256, fd);
 }

static void Pass(inFD, outFD)
 FILE *inFD;
 FILE *outFD;
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right, BitsPerBand;
  int KnownColorSystem, BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  int x,y,b,width,height,i,c;
  unsigned short *WFFScanLine;
  int WFFScanLineLength;
  unsigned char *SunScanLine;
  int SunScanLineLength, depth;

  FrameBuffer = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer)) 
   {
    fprintf("%s: OpenFB failed\n", RoutineName);
    exit(-1);
   }
  
  if (FAILURE == ReadImage(inFD, FrameBuffer)) 
   {
    fprintf(stderr,"%s: ReadImage failed\n", RoutineName);
    (void)CloseFB(&FrameBuffer);
    exit(-1);
   }

  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right)) 
   { 
    fprintf(stderr,"%s: GetBounds failed\n", RoutineName);
     (void)CloseFB(&FrameBuffer); exit(-1);
   }

  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   { 
    fprintf(stderr,"%s: GetColorsystem failed\n", RoutineName);
    (void)CloseFB(&FrameBuffer); exit(-1);
   }

  if      (0 == strcmp(WhatBands,"I"))   BandsPerPixel = 1;
  else if (0 == strcmp(WhatBands,"RGB")) BandsPerPixel = 3; 
  else 
   {
    fprintf(stderr,"%s: Sorry, we only do I and RGB format\n",RoutineName);
    (void)CloseFB(&FrameBuffer); exit(-1);
   }

  width = Right - Left + 1;
  height = Top - Bottom + 1;

  switch (BandsPerPixel)
   {
    case 1: depth = 1; break;
    case 3: if (8 < (BitsPerBand*3)) depth = 3; else depth = 1;
   }
  if (VERBOSE)
   fprintf(stderr,"%s: BandsPerPixel = %d; BitsPerBand = %d; depth = %d\n",
     RoutineName, BandsPerPixel, BitsPerBand, depth);


  WFFScanLineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanLine = (unsigned short *) malloc(WFFScanLineLength);
  if ((unsigned short *)0 == WFFScanLine)
   {
    fprintf (stderr,"%s: no memory for WFFScanLine\n", RoutineName);
    (void)CloseFB(&FrameBuffer); exit(-1);
   }

  SunScanLineLength = width*depth*(sizeof (unsigned char));
  SunScanLineLength = SunScanLineLength + (SunScanLineLength % 2);
  SunScanLine = (unsigned char *) malloc(SunScanLineLength);
  if ((unsigned char *)0 == SunScanLine)
   {
    fprintf (stderr,"%s: no memory for SunScanLine\n",RoutineName);
    free(WFFScanLine); (void)CloseFB(&FrameBuffer); exit(-1);
   }

  if (VERBOSE) fprintf(stderr,"%s: Writing SunHeader\n", RoutineName);

  CreateSunHeader(width, height, BandsPerPixel, depth); 
  WriteSunHeader(outFD);

  if ((3 == BandsPerPixel) && (1 == depth))
   {
    if (VERBOSE) fprintf(stderr,"%s: Writing Sun ColorMap\n",RoutineName);
    WriteSunColorMap(BitsPerBand,outFD);
   }
  
  if (VERBOSE) fprintf(stderr,"%s: Started to write the image\n", RoutineName);

  for (y=Top;y>=Bottom;y--)
   {
    if (FAILURE == GetBlock(FrameBuffer,y,Left,y,Right, WFFScanLine))
     { 
      fprintf("%s: GetBlock Failed",RoutineName);
      free(SunScanLine);free(WFFScanLine);(void)CloseFB(&FrameBuffer);exit(-1);
     }

    i = 0;  c = 0;
    if (depth == BandsPerPixel)
     {
      for(x=Left;x<=Right;x++)
       {
        for(b=0;b<BandsPerPixel;b++)
         {  /* convert a sample from BitsPerBand to 8 bits */
          register unsigned int j, sample, all;
          sample = ((int)WFFScanLine[c++]) << (16-BitsPerBand) >> 8;
          for(all=0, j=0; j<8; j += BitsPerBand)
           all = (all >> BitsPerBand) | sample;
          SunScanLine[i++] = (unsigned char) (all & 0x000000ff);
         }
       }
     }
    else
     { /* pack RGB into 8 bits  - we know they will fit */
      for(x=Left;x<=Right;x++)
       {
        unsigned int red, green, blue, all;
        red   = (int)WFFScanLine[c++];
        green = (int)WFFScanLine[c++];
        blue  = (int)WFFScanLine[c++];
        all = red;
        all = (all << BitsPerBand) | green;
        all = (all << BitsPerBand) | blue;
        SunScanLine[i++] = (unsigned char) (all & 0x000000ff);
       }
     }
    WriteSunScanLine(SunScanLine, SunScanLineLength, outFD);
   }
  
   if (ferror(outFD))
    fprintf(stderr,"%s: Error writing image\n", RoutineName);

   if (VERBOSE)
    fprintf(stderr,"%s: finished writing the image\n", RoutineName);
   fflush(outFD);
   free(SunScanLine);free(WFFScanLine);(void)CloseFB(&FrameBuffer);
 }

int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;

  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1); }
    switch (argv[ArgsParsed++][1])
     {
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': { usage(); exit(-1); }
     }
   }
 
  WhichEndUp();
  if (VERBOSE)
   {
    if (SwapBytes) fprintf(stderr,"%s: LittleEndian Bytes\n",RoutineName);
    else           fprintf(stderr,"%s: BigEndian Bytes\n",RoutineName);
    if (SwapShorts)fprintf(stderr,"%s: LittleEndian Shorts\n",RoutineName);
    else           fprintf(stderr,"%s: BigEndian Shorts\n",RoutineName);
   }
  Pass(stdin, stdout);

  exit(0); 
 }
