/*
   File: DiffuseN.c
   Authors: K.R. Sloan,
           James Painter
   Last Modified: 25 March 1991
   Purpose: Convert a .wff file to n bits per band, using the Floyd-Steinberg
            error propagation scheme.

            Parameters:

            -b n        - bits per band in output image
                          [default = 1]

            -s screen u - filename containing the "screen" pattern -
                          an 8x8 array of integers in the range [0,63]
                          and a modulation factor for the screen:
                          0.0 => (constant threshold = 0.5)
                          1.0 => (threshold varying between 0.0 and 1.0)
                          [default = 0.0]

            -e v        - error propagation modulation
                          0.0 => (no error propagation - straight dither)
                          1.0 => (all error propagated)
                          [default is 1.0]

            -h          - help

            Both of the modulation parameters can be set to values
            greater than 1.0, with "interesting" results 
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>
double atof();

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-b n] [-s screen u] [-e v][-h] \n",
           RoutineName);
 }

static void
Pass(fdIn,fdOut,T,u,v,NewBitsPerBand)
 FILE *fdIn, *fdOut;
 int T[8][8];
 double u,v;
 int NewBitsPerBand;
 {
  FrameBufferType   *FBin, *FBout;
  int Bottom, Left, Top, Right;
  char Name[NameLength], Value[ValueLength];
  char WhatBands[ValueLength];
  int  BitsPerBand, BandsPerPixel;
  int x,y,n, band, ex;
  unsigned short Pixel[99];  /* that ought to be enough... */
  int passed = 0;
  int *Error[99];
  int L[99];
  int i;
  int N[8][8];
  int White, Black, WhiteClip, BlackClip, MidGray, Half;
  double ErrorFactor;
  int Shift, DupShift, Dups;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }
  if (FAILURE == OpenFB(&FBout))          { (void)CloseFB(&FBin);  return; }
 

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  GetColorSystem (FBin, WhatBands, &BitsPerBand );
  BandsPerPixel = strlen(WhatBands);
  Shift = BitsPerBand - NewBitsPerBand;
  if (Shift<0)
   {
    fprintf(stderr,"%s: Can't INCREASE the BitsPerBand!\n", RoutineName);
    (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; 
   }

  /* Set the output color system */
  SetColorSystem( FBout, WhatBands, NewBitsPerBand);
  
  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */
  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }


  Dups = (BitsPerBand+NewBitsPerBand-1) / NewBitsPerBand;
 
  White = (1 << BitsPerBand) - 1;
  Black = 0;
  MidGray = (White+Black) >> 1;
  WhiteClip = (int)((double)White*1.0); /* 1.1 */
  BlackClip = Black + White - WhiteClip;
  Half = (1 << Shift) >> 2;

  for(band=0; band<BandsPerPixel; band++)
   Error[band] = (int *) calloc( (unsigned) (Right-Left+3),
                                 sizeof(int) );
  ErrorFactor = v / 8.0;

  for(i=0;i<NewBitsPerBand;i++) u = u/2.0;
  for (y=0;y<8;y++)
   for (x=0;x<8;x++)
    N[x][y] = (int)(u* ((double)(T[x][y]-32)) /64.0 * (double)(White-Black) );

 
  for (y=Bottom;y<=Top;y++)
   {
    for(band=0; band<BandsPerPixel; band++)
     {
      L[band] = 0;
      Error[band][0] = 0;
     }

    for (x=Left, ex = 0;x<=Right;x++, ex++)
     {
      if (FAILURE == NextPixelIn(FBin,Pixel))
       { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
      for(band=0; band<BandsPerPixel; band++) 
       {
        int value, out, e, eee;
        int Nv, p, dup;
        unsigned short ink;

        value = ((int)Pixel[band]) - (ErrorFactor * Error[band][ex+1]);

        if      (value<BlackClip) value = BlackClip;
        else if (value>WhiteClip) value = WhiteClip;

        Nv = value+N[x%8][y%8];

        if (Nv<0)
         {
          Pixel[band] = 0;
          e = Black - value;
         }
        else if (White<=Nv)
         {
          Pixel[band] = White >> Shift;
          e = White - value;
         }
        else
         {
          p = Nv >> Shift;
/*          p = (Nv + Half) >> Shift; */
          Pixel[band] = p;
          ink = p << Shift;
          for(dup=0;dup<Dups;dup++) ink = ink | (ink>>NewBitsPerBand);  
          e = ink - value;
         }
        eee = e+e+e;
        Error[band][ex+1]  = eee+L[band]; /* down from us, and from our left*/
        Error[band][ex  ] += e;     /* to the left and down               */
        Error[band][ex+2] += eee;   /* to the right                       */
        L[band]            = e;     /* to the right and (eventually) down */
	  /* that's 8 e distributed to our neighbors */
       }
      if (FAILURE == NextPixelOut(FBout,Pixel))
       { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }
     }
   }
  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);

 }

main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *ThresholdFileName = (char *)0;
  double u = 0.0;
  double v = 1.0;
  int NewBitsPerBand = 1;
  FILE *TF;
  int i,j;
  int T[8][8];
 
  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'b':
         if ((argc-ArgsParsed) < 1) { usage(); exit(-1);}
         NewBitsPerBand = atoi(argv[ArgsParsed++]);
         break;
        case 's':
         if ((argc-ArgsParsed) < 2) { usage(); exit(-1);}
         ThresholdFileName = argv[ArgsParsed++];
         u = atof(argv[ArgsParsed++]);
         break;
        case 'e':
         if ((argc-ArgsParsed) < 1) { usage(); exit(-1);}
         v = atof(argv[ArgsParsed++]);
         break;
        default:
        case 'h':
         usage(); exit(-1);
      }
     }
    else { usage(); exit (-1); }	
   }

  if ((char *)0 == ThresholdFileName)
   {
    for(j=0;j<8;j++)
     for (i=0;i<8;i++)
      T[i][j] = 32;
   }
  else
   {   
    if ((FILE *)0 == (TF = fopen(ThresholdFileName,"r")))
     {
      fprintf(stderr,"%s: Can't open (%s)\n",RoutineName, ThresholdFileName);
      perror(RoutineName);
      exit(1);
     }
    for(j=0;j<8;j++)
     for (i=0;i<8;i++)
      fscanf(TF," %d",&T[i][j]);
   }  
    
  Pass(stdin,stdout,T,u,v,NewBitsPerBand);

  exit (0);
}

