/*
  File: xy2rgb.c
  Author: K.R. Sloan
  Last Modified: 11 June 1991
  Purpose: produce rgb values as a function of wavelength,
           starting with a table of xy values
 */

#include <stdio.h>

int VERBOSE = 0;
static char *RoutineName;

static void usage()
 {
  fprintf(stderr,"usage is:\n\t%s [-h][-v]\n",RoutineName); 
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(-1);
 }

static int l0 = 400;
static int l1 = 700;
static double x[301]; /* 400-700 */
static double y[301];
static double R[301];
static double G[301];
static double B[301];

static void GamutMap(r,g,b)
 double *r, *g, *b;
 {
  double s,t;

  /*
     r,g,b > 0
   */ 
  if (*r < 0)
   {
    t = 1.0/(1.0-(*r*3.0));
    s = (1.0-t)/3.0;
    *r = *r*t + s;
    *g = *g*t + s;
    *b = *b*t + s;
   
   }
  if (*g < 0)
   {
    t = 1.0/(1.0-(*g*3.0));
    s = (1.0-t)/3.0;
    *r = *r*t + s;
    *g = *g*t + s;
    *b = *b*t + s;
   }
  if (*b < 0)
   {
    t = 1.0/(1.0-(*b*3.0));
    s = (1.0-t)/3.0;
    *r = *r*t + s;
    *g = *g*t + s;
    *b = *b*t + s;
   }

  /*
     r,g,b < 0.666
   */ 
  if (0.666 < *r)
   {
    t = 1.0/((*r*3.0)-1.0);
    s = (1.0-t)/3.0;
    *r = *r*t + s;
    *g = *g*t + s;
    *b = *b*t + s;
   
   }
  if (0.666 < *g)
   {
    t = 1.0/((*g*3.0)-1.0);
    s = (1.0-t)/3.0;
    *r = *r*t + s;
    *g = *g*t + s;
    *b = *b*t + s;
   }
  if (0.666 < *b)
   {
    t = 1.0/((*b*3.0)-1.0);
    s = (1.0-t)/3.0;
    *r = *r*t + s;
    *g = *g*t + s;
    *b = *b*t + s;
   }

  /*
     final clamping
   */

  if (*r > 1.0) { *g /= *r; *b /= *r; *r = 1.0;}
  if (*g > 1.0) { *b /= *g; *r /= *g; *g = 1.0;}
  if (*b > 1.0) { *r /= *b; *g /= *b; *b = 1.0;}

  /*
     should not happen??
   */

  if (*r < 0.0) *r = 0.0;
  if (*g < 0.0) *g = 0.0;
  if (*b < 0.0) *b = 0.0;

 }

static void xy2RGB(x,y,R,G,B)
 double x[], y[];
 double R[], G[], B[];
 {
  int l,k;
  double X,Y,Z;

  for(l=l0;l<=l1;l++)
   {
    k = l-l0;

    Y = 1.0; /* make this Y[l] later  ??? */

    X = x[k]*Y/y[k];   /* FvDFH p 581 */
    Y = Y;
    Z = (1.0 - x[k] - y[k])*Y/y[k];

    /* 
      this transform was derived by taking the xy coordinates
      for a short-persistence phosphor on page 583 of FvDF&H and
      using 0.3, 0.59, 0.11 for Yr,Yg,Yb.
      The computed X,Y,Z values were plugged
      into equation 13.23 (p.586) and the matrix was inverted.

      With luck - this should represent the XYZ->RGB transform for
      an "average" monitor...
      
     */

    R[k] =  2.6350*X - 1.2128*Y - 0.4051*Z;
    G[k] = -1.3496*X + 2.3441*Y - 0.0696*Z;
    B[k] =  0.0521*X - 0.1745*Y + 0.7318*Z;

    if (VERBOSE)
     fprintf(stderr,
        "%s: l = %5d, [%10.4f,%10.4f] -> [%10.4f,%10.4f,%10.4f]\n",
         RoutineName,l,x[k],y[k],R[k],G[k],B[k]);
   }
 }

static void ReadAndInterpolate(s,x,y)
 FILE *s;
 double x[],y[];
 {
  int ThisL, LastL, l;
  double ThisX, ThisY, LastX, LastY;
  

  ThisL = -1; ThisX = -1.0; ThisY = -1.0;
  for(;;)
   {
    LastL = ThisL; LastX = ThisX; LastY = ThisY;
    if (3 != fscanf(s," %d %lf %lf",&ThisL, &ThisX, &ThisY)) break;

    if (VERBOSE) fprintf(stderr,
                         "%s: ThisL = %5d, ThisX = %10.4f, ThisY = %10.4f\n",
                                 RoutineName,ThisL, ThisX, ThisY);
    if (-1 != LastL)
     for (l=LastL; l<=ThisL; l++)
      {
       if ((l < l0) || (l1 < l)) continue;
       x[l-l0] = LastX + (ThisX-LastX)*(l-LastL)/(ThisL-LastL);
       y[l-l0] = LastY + (ThisY-LastY)*(l-LastL)/(ThisL-LastL);
       if (VERBOSE) fprintf(stderr,"%s: l = %5d, x = %10.4f, y = %10.4f\n",
                                 RoutineName,l,x[l-l0],y[l-l0]);
      }
   }  
 }

static void WriteRGB(s,R,G,B)
 FILE *s;
 double R[],G[],B[];
 {
  double r,g,b;
  int l,k;
  for(l=l0;l<=l1;l++)
   {
    k = l-l0;
    r = R[k]; g = G[k]; b = B[k];
    GamutMap(&r,&g,&b);
    fprintf(s,"%5d   %10.4f %10.4f %10.4f   %10.4f %10.4f %10.4f\n",
               l,R[k],G[k],B[k],r,g,b);
   }
 }


int main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;

  RoutineName = argv[ArgsParsed++];

  while(ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) {usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': usage(); exit(-1);
     }
   }  

  if (VERBOSE) fprintf(stderr,"%s: Reading ...\n",RoutineName);
  ReadAndInterpolate(stdin,x,y);

  if (VERBOSE) fprintf(stderr,"%s: Converting ...\n",RoutineName);
  xy2RGB(x,y,R,G,B);

  if (VERBOSE) fprintf(stderr,"%s: Writing ...\n",RoutineName);
  WriteRGB(stdout,R,G,B);

  exit(0);
 }

