/*
 File: RateTournament.c
 Author: K.R. Sloan
 Last Modified: 23 December 1992
 Purpose: rate a tournament

 Input: PET - 0 == use Normal tables
              1 == use Logistic tables

        PLAYERS - number of players
        ROUNDS  - number of rounds
        PairingCards - see "PairingCards.h"

 */

#include <stdio.h>
#include <malloc.h>
#include "PairingCards.h"

extern int VERBOSE;

static int *LowerBound;

int LowerBoundNormal[51]
 = {  0,  4, 11, 18, 26, 33, 40, 47, 54, 62, 69, 77, 84, 92, 99,107,114,
    122,130,138,146,154,163,171,180,189,198,207,216,226,236,246,257,268,
    279,291,303,316,329,345,358,375,392,412,433,457,485,518,560,620,735};

int LowerBoundLogistic[51]
 = {  0,  4, 11, 18, 25, 32, 39, 46, 53, 60, 67, 75, 82, 89, 97,104,112,
    120,128,136,144,152,160,169,178,187,196,206,215,225,236,247,258,270,
    282,295,309,324,339,355,373,392,413,437,464,495,531,577,637,727,920};   

double P(D)
 int D;
 {
  int d,i,j;
  double p;

  if (0 > D) d = -D; else d = D;
  j = 50;
  for(i=1;i<51;i++) if (d < LowerBound[i]) {j = i-1; break;}
  p = 0.01*(50.0+(double)j);
  if (0 > D) return(1.0-p); else return(p);
 }

int D(P)
 double P;
 {
  double p;
  int j,d;

  if (0.50 <= P) p = P-0.50; else p = 0.50-P;
  j = (p*100.0);
  if (j<0) j = 0; else if (50<j) j = 50;
  d = LowerBound[j];
  if (0.50 <= P) return(d); else return(-d); 
 }

int RateTournament(PET,PLAYERS,ROUNDS,PairingCards)
 int PET;
 int PLAYERS;
 int ROUNDS;
 PlayerType PairingCards[];
 {
  int *R0;
  int p; 
  int notConverged;

  switch (PET)
   {
    default:
    case 0: LowerBound = &LowerBoundNormal[0]; break;
    case 1: LowerBound = &LowerBoundLogistic[0]; break;
   }


  if (!(R0 = (int *)malloc(PLAYERS*sizeof(int))))
   {
    fprintf(stderr,"RateTournament: no memory for R0");
    exit(-1);
   }

  for(p=0;p<PLAYERS;p++)
   R0[p] = PairingCards[p].Ro;  /* start with pre-tournament rating */

  notConverged = -1;
  while(notConverged)
   {
    notConverged = 0; /* assume no unrateds */
    for(p=0;p<PLAYERS;p++)
     {
      PlayerType *PC;
      unsigned long int Rc;
      int r,GamesPlayed;
      double W, We;
    
      PC = &(PairingCards[p]);
      W = 0.0; We = 0.0; GamesPlayed = 0; Rc = 0;    
      for(r=0;r<ROUNDS;r++)
       {
        GameType *G;
        int opponent, Ropp;
      

        G = &(PC->Games[r]);
        opponent = G->Opponent;
        if (0 <= opponent)
         {
          Ropp = R0[opponent];
          Rc += Ropp;
          switch(G->Result)
           {
            case 'W': W += 1.0; break;
            case 'D': W += 0.5; break;
	    default:            break;
           }
          We += P(R0[p]-Ropp);
          if (VERBOSE) fprintf(stderr,
           "Results[%d][%d] = %c(%d)(%4d), W = %f, We = %f\n",
            p+1,r,G->Result,opponent+1,Ropp,W,We);
          GamesPlayed++;
         }
       }
         
      if (0 == GamesPlayed) 
       {
        PC->Rp = PC->Ro; /* why not? */
        PC->Rn = PC->Ro;
       }
      else
       {
        /* the linear approximation - best for small samples */
        PC->Rp = Rc/GamesPlayed
                 + 400.0*((2.0*W-(double)GamesPlayed) / (double)GamesPlayed);
        if (VERBOSE)
         {
          fprintf(stderr,
           "%d: Rc = %d, GP = %d, W = %f, We = %f\nP = %f, D = %d, Rp = %d\n",
                    p+1,(int)(Rc/GamesPlayed),GamesPlayed,W,We,
                    W/(double)GamesPlayed, D(W/(double)GamesPlayed),
                    PC->Rp);
         } 
        if (PC->Previous < 0)
         { /* established rating */
          int K;
 
          if      (PC->Ro < 2100) K = 32;
          else if (PC->Ro < 2400) K = 24;
          else                    K = 16;
          PC->Rn = PC->Ro + K*(W-We);
         }
        else if (PC->Previous > 0)
         { /* provisional */
          int prev,total;

          prev = PC->Previous; total = prev+GamesPlayed;
          PC->Rn = ((prev*(PC->Ro)) + (GamesPlayed*(PC->Rp))) / total;
         }
        else
         { /* unrated */
          PC->Rn = PC->Rp;
          notConverged = -1;
          if (VERBOSE)
           fprintf(stderr,"Unrated: set notConverged to %d\n",notConverged);
         }
       }
     }
     /* if we have unrated players, we may have to converge */
    if (notConverged)
     {
      int delta, MaxDelta;

      if (VERBOSE) fprintf(stderr,"Checking for convergence\n");
      MaxDelta = 0;
      for(p=0;p<PLAYERS;p++)
       {
        PlayerType *PC;

        PC = &(PairingCards[p]);
        if (0 == PC->Previous)
         {
          delta = (PC->Rn) - R0[p];
          if (0 > delta) delta = -delta;
          if (delta > MaxDelta) MaxDelta = delta;
          R0[p] = PC->Rn;  /* for the next iteration */
         }  
       }
      if (MaxDelta <= 0) notConverged = 0; /* close enough */
     }
   }
  /* Rn has the post-tournament ratings */
  /* Rp has the performance rating */
  free(R0);
  return(0);  /* all OK */
 }
