/*
  File: devDQ/DEVICE.c
  Authors: K.R. Sloan,
           David Meyers
  Last Modified: 10 February 1991
  Purpose: definitions for the generic graphics device used by SGP
           this version calls dqRect for each call to devRect.
           (modelling the case where painting a colored rectangle
           is the only hardware operation)

  See also: devLine.c, devTriangle.c, devCircle.c, devClip.c.
           Some of the functions promised in DEVICE.h are implemented there.

           devPick always fails
 */
#include <stdio.h>
#include <local/dq.h>
#include "SGP.h"
#include "DEVICE.h"
extern devClipLine();
extern devClipRectangle();

/*  Device state  - made global for convenience ... */
int devLeft, devBottom, devRight, devTop;
int devClipL, devClipB, devClipR, devClipT;
int devCurrentColor;
FILE *devOutputStream = stdout;
double devAspectRatio = 1.0;

/* local flag indicating color or monochrome display */
static int DisplayType;

/* current color map */
byte ColorMap[256][3];

static void RGBwash(start, stop, r1, g1, b1, r2, g2, b2, cm)
 int start, stop, r1, g1, b1, r2, g2, b2;
 byte cm[256][3];
 {
  double r, g, b, dr, dg, db, steps;
  int slot;
 
  steps = stop - start;
  dr = (r2-r1)/steps;
  dg = (g2-g1)/steps;
  db = (b2-b1)/steps;
  r = r1; g = g1; b = b1;
  for (slot = start; slot <= stop; slot++)
   {
    cm[slot][0] = r; cm[slot][1] = g; cm[slot][2] = b;
    r = r + dr; g = g + dg; b = b + db;
   }
 }

/*
   devInit initializes the display, loading color maps if necessary
   There are two modes: color and mono.  Here, we assume a full-color
   display, and the two modes are identical

   devInit may also need to fill in the "physical" size of the display
   here, we output the "physical" limits to help your interpreter 
 */
int devInit(color, OutputStream)
 int color;
 FILE *OutputStream;
 {
  int slot;

  devOutputStream = OutputStream;
  devLeft = 0;  devBottom = 481; devRight = 639; devTop = 2;
  devSetClipWin(devLeft,devBottom,devRight,devTop);
  (void) dqInit();
  DisplayType = color;

  switch (color)
   {
    case (int) ColorDisplay:
     for (slot = 0; slot < 256; slot++)
      {
       ColorMap[slot][0] = (slot & 0xe0);      
       ColorMap[slot][1] = (slot & 0x1c) << 3; 
       ColorMap[slot][2] = (slot & 0x03) << 6; 
      }
     (void)dqSetOLUT(7);
     (void)dqWriteCM(0, 255, ColorMap);
    break;

    case (int) MonochromeDisplay:
      RGBwash(0, 255, 0, 0, 0, 255, 255, 255, ColorMap);
      (void)dqSetOLUT(0); (void)dqWriteCM(0, 255, ColorMap);     
    break;
   }

  devSetColor(devColorCode(1.0, 1.0, 1.0));
  return(0); /* success */ 
 }

/*
  devClose insures that all operations are complete and closes up
 */
int devClose()
 {
  return(0); /* success */ 
 }

/*
  devSetClipWin establishes a clipping window
  restrictions such as (L<R) must be enforced here!
 */
int devSetClipWin(L, B, R, T)
 int L, B, R, T;
 {
  if (L<R) { devClipL=L; devClipR=R; } else { devClipL=R; devClipR=L;} 
  if (B<T) { devClipB=B; devClipT=T; } else { devClipB=T; devClipT=B;} 
  if (devLeft < devRight)
   {
    if (devClipL < devLeft)   devClipL = devLeft;
    if (devClipR > devRight)  devClipR = devRight;
   }
  else
   {
    if (devClipL < devRight)  devClipL = devRight;
    if (devClipR > devLeft)   devClipR = devLeft;
   }

  if (devBottom < devTop)
   {
  /*  if (devClipB < devBottom) devClipB = devBottom;   kludge alert */
  /*  if (devClipT > devTop)    devClipT = devTop;                   */
   }
  else 
   {
  /*  if (devClipB < devTop)    devClipB = devTop;                */
  /*  if (devClipT > devBottom) devClipT = devBottom;   and again */
   }

  return(0); /* success */
 }

/*
 devClearScreen fills the entire screen with a particular color
 */
int devClearScreen(color)
 int color;
 {
  int SavedColorCode;
  int SavedClipL, SavedClipB, SavedClipR, SavedClipT; 

  SavedColorCode = devCurrentColor;
  SavedClipL = devClipL;  SavedClipB = devClipB;
  SavedClipR = devClipR;  SavedClipT = devClipT;
  /* devSetClipWin(devLeft,devBottom,devRight,devTop); */
  devSetClipWin(devLeft,devBottom+10,devRight,devTop-2); /* kludge alert */
  (void) devSetColor(color);
  devRect(devLeft, devBottom+10, devRight, devTop-2);    /* and again */    

  (void) devSetColor(SavedColorCode);
  devSetClipWin(SavedClipL,SavedClipB,SavedClipR,SavedClipT);

  return(0); /* success */
 }

/*
   the following routines draw primitive shapes on the physical device
   All of them clip against the device's clipping window.
   All of them paint in the device's current color
 */

/*
  devLine draws a line - see devLine.c
 */

/*
 devRect fills a rectangle - we assume this is a primitive
 */
int devRect(L, B, R, T)
 int L, B, R, T;
 {
  if (devClipRectangle(&L, &B, &R, &T)) return(0); /* success */

  return(dqRect(L, T, R, B)); /* status returned by dqRect */
 }

/*
 devDisc fills a disc - see devCircle.c
 */

/*
 devCircle draws a circle - see devCircle.c
 */

/*
 devSolidTriangle fills a triangle - see devTriangle.c
 */

/*
 devShadeTriangle shades a triangle
   with color codes specified at each vertex - see devTriangle.c
 */

/*
 devSetColor establishes the current color
 */
int devSetColor(color)
 int color;
 {
  devCurrentColor = color;
  return(dqSetPenColor(devCurrentColor)); /* return value returned by driver */
 }

/*
   devColorCode returns a framebuffer value - here we assume a full-color
   display, with 3 bits of r, 3 bits of g, 2 bits of b packed as:
            [rrrgggbb]
  OR an 8-bit Monochrome display, with 8 bits of i = [iiiiiiii].
  If a color is specified as RGB we convert to Intensity using CIE weights
  of (0.30, 0.59, 0.11).
 */ 
int devColorCode(R, G, B)
 double R,G,B; 
 {
  int devR, devG, devB;

  if ((int) ColorDisplay == DisplayType)
   {
    devR = (8.0 * R); if(8 == devR) devR = 7;
    devG = (8.0 * G); if(8 == devG) devG = 7;
    devB = (4.0 * B); if(4 == devB) devB = 3;
    return (  ((devR & 0x07) << 5)
	    | ((devG & 0x07) << 2)
	    | ((devB & 0x03)     ) );
   }
  else /* MonochromeDisplay so convert to Intensity using CIE weights */
   {
    devR = ((0.3 * R) + (0.59 * G) + (0.11 * B)) * 255.0;
    return (devR & 0xff);
   }
 }
/*
  devColorDecode generates r,g,b values in [0.0, 1.0] 
 */
int devColorDecode(code, R, G, B)
 int code;
 double *R, *G, *B;
 {
  int devR, devG, devB;

  if ((int) ColorDisplay == DisplayType)
   {
    devR = (code >> 5) & 0x07;
    devG = (code >> 2) & 0x07;
    devB = (code     ) & 0x03;
    *R = (double)devR / 7.0;
    *G = (double)devG / 7.0;
    *B = (double)devB / 3.0;
   }
  else /* MonochromeDisplay */
   {
    devR = code & 0xff;
    *R = (double) devR / 255.0;
    *G = *R;
    *B = *R;
   }
  return (0);
 }

int devPick(x,y,button)
 int *x,*y,*button;
 {
  *x = 0; *y = 0; *button = 0; return(-1); /* failure */
 }

/*       -30-         */
