/*
  File: Triangles.c
  Author: K.R. Sloan
  Last Modified: 7 March 1991
  Purpose: hidden-surface rendering of triangles

           RenderTriangles displays the entire collection of triangles,
           with hidden surfaces removed. 

           The user program should allocate an array of TriangleType3D,
           and fill in the geometry and color information for each vertex.

           The Triangles array is used for scratch storage, may be
           extended, and is meaningless after RenderTriangles returns.
          
           If the user program doesn't cull back-facing triangles, we
           do it here (and that ought to provide enough extra space).
           If not, the user program must allocate an array large enough
           to handle some expansion.  10-20% should be enough...

           Unless...the global TwoSided indicates that we should
           reverse all normals on back-facing triangles.  This may
           not do what you want - tough.   It's purely kludge-control.
           The tasteful user will not find it necessary.

           Similarly, the global LeftHanded indicates that triangles
           follow the left-hand rule for cross-product calculations.
           Adroit users won't need this feature.

           Yet another global, Facets, indicate that the normal to the
           plane containing the vertices should be used INSTEAD OF the
           vertex normals.

           A "hidden" function:
             RenderMode(Preview, TensorProduct, InteriorColor, OutlineColor)
           allows the serious hacker to specify line-drawing output,
           rather than shaded surfaces.

           If Preview == 0, each triangle is Gouraud shaded; the colors
           specified in the call to RenderMode are ignored.  

           If Preview == 1, each triangle is painted "InteriorColor",
           and the outline drawn in "OutlineColor".

           if Preview == 2, no hidden surface calculation is done, and
           triangles are rendered as outlines, AFTER backface culling.

           The global VERBOSE controls output. 

 */
#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"
#include "Triangles.h"

extern int VERBOSE;
static int DEBUG = 0;
double fabs();

int Facets = 0;
int TwoSided = 0;
int LeftHanded = 0;

#define EPSILON (0.0000000000001)


/* this is PROBABLY not necessary - see the Panic trap... */
#define MAXLOOPING (10)


static Preview = 0;
static TensorProduct = 0;
static sgpColorType InteriorColor = { 1.0, 1.0, 1.0 };
static sgpColorType OutlineColor  = { 0.0, 0.0, 0.0 };

void RenderMode(NewPreview, NewTensorProduct, 
                NewInteriorColor, NewOutlineColor)
 int NewPreview, NewTensorProduct;
 sgpColorType NewInteriorColor, NewOutlineColor;
 {
  Preview = NewPreview;
  TensorProduct = NewTensorProduct;
  InteriorColor = NewInteriorColor;
  OutlineColor = NewOutlineColor;
 }

/*
   Is the point P inside triangle [A,B,C]?
 */
static int InsideP(P,A,B,C)
 sgpPointType P,A,B,C;
 {
  double abx, aby, acx, acy, abCROSSac;
  double pax, pay, pbx, pby, pcx, pcy;
  double paCROSSpb, pbCROSSpc, pcCROSSpa; 

  /* check for tiny triangles - nothing is inside them... */
  abx = B.x-A.x; aby  = B.y-A.y;
  acx = C.x-A.x; acy  = C.y-A.y;
  abCROSSac = abx*acy - aby*acx;
  if ((-EPSILON < abCROSSac) && (abCROSSac < EPSILON)) return(0);

  pax = A.x-P.x; pay = A.y-P.y;
  pbx = B.x-P.x; pby = B.y-P.y;
  pcx = C.x-P.x; pcy = C.y-P.y;

  paCROSSpb = pax*pby - pay*pbx;
  pbCROSSpc = pbx*pcy - pby*pcx;
  pcCROSSpa = pcx*pay - pcy*pax;

  /* note - anything close to 0 loses */
  return(
    ((paCROSSpb<-EPSILON) && (pbCROSSpc<-EPSILON) && (pcCROSSpa<-EPSILON))
  ||((paCROSSpb> EPSILON) && (pbCROSSpc> EPSILON) && (pcCROSSpa> EPSILON)) 
        );
 }

/*
   Does the line segment between A1 and A2 intersect
        the line segment between B1 and B2?
   Reference: Newman&Sproull p. 494-5
 */

static int LineSegmentIntersectionP(A1,A2,B1,B2)
 sgpPointType A1,A2,B1,B2;
 {
  double u1,u2,u3, v1,v2,v3;
  double w1,w2,w3;

  /* represent the line between A1 and A2 as u = A1 x A2  */
  /* A1 and A2 specify x and y - augment to add w = 1.0, and optimize */
  u1 = A1.y - A2.y;
  u2 = A2.x - A1.x;
  u3 = A1.x*A2.y - A1.y*A2.x;
  /* represent the line between B1 and B2 as v = B1 x B2*/
  /* B1 and B2 specify x and y - augment to add w = 1.0, and optimize */
  v1 = B1.y - B2.y;
  v2 = B2.x - B1.x;
  v3 = B1.x*B2.y - B1.y*B2.x;
  /* the intersection of the lines is  w = u x v */
  w1 = u2*v3 - u3*v2;
  w2 = u3*v1 - u1*v3;
  w3 = u1*v2 - u2*v1;
  /* check for degeneracy, and convert back to 2D */
  if ( (-EPSILON <= w3) && (w3 <= EPSILON) ) return(0);  /* can't tell ... */
  w1 /= w3; w2 /= w3;
  /*
    [w1,w2] is the point of intersection of the lines.
    check BOTH line segments to see that the point is strictly INSIDE
    the segment.
   */
  if ( (w1<=A1.x) && (w1<=A2.x) ) return(0);
  if ( (w1>=A1.x) && (w1>=A2.x) ) return(0);
  if ( (w2<=A1.y) && (w2<=A2.y) ) return(0);
  if ( (w2>=A1.y) && (w2>=A2.y) ) return(0);
  if ( (w1<=B1.x) && (w1<=B2.x) ) return(0);
  if ( (w1>=B1.x) && (w1>=B2.x) ) return(0);
  if ( (w2<=B1.y) && (w2<=B2.y) ) return(0);
  if ( (w2>=B1.y) && (w2>=B2.y) ) return(0);
  return(1);
 }

static int BoundingBox(t)
 TriangleType3D *t;
 {
  PointType3D VB;

    /* calculate Bounding Box min/max(x,y,z) in StandardViewBox */
  WorldToViewBox3D(t->Vertex[0], &VB);
  t->MinX = VB.x;  t->MaxX = VB.x;
  t->MinY = VB.y;  t->MaxY = VB.y;
  t->MinZ = VB.z;  t->MaxZ = VB.z;

  WorldToViewBox3D(t->Vertex[1], &VB);
  if      (VB.x < t->MinX) t->MinX = VB.x;
  else if (VB.x > t->MaxX) t->MaxX = VB.x;
  if      (VB.y < t->MinY) t->MinY = VB.y;
  else if (VB.y > t->MaxY) t->MaxY = VB.y;
  if      (VB.z < t->MinZ) t->MinZ = VB.z;
  else if (VB.z > t->MaxZ) t->MaxZ = VB.z;

  WorldToViewBox3D(t->Vertex[2], &VB);
  if      (VB.x < t->MinX) t->MinX = VB.x;
  else if (VB.x > t->MaxX) t->MaxX = VB.x;
  if      (VB.y < t->MinY) t->MinY = VB.y;
  else if (VB.y > t->MaxY) t->MaxY = VB.y;
  if      (VB.z < t->MinZ) t->MinZ = VB.z;
  else if (VB.z > t->MaxZ) t->MaxZ = VB.z;

  /* and, might as well initialize this, too */
  t->Looping=0;
 }

/*
   Blend two values, in the ratio (1.0-t) / t
   perhaps this ought to be a macro?
 */
static double Blend(A, B, t)
 double A, B, t;
 {
  return(A + t*(B-A));
 }

/*
  Subdivide the triangle (t), at a plane (N,d), producing one, two, or three 
  output triangles.

  Vertex normals for new vertices are linearly interpolated.  All triangle 
  normals match the original.

  Surface properties for new vertices are linearly interpolated.

 */
static int SubdivideTriangle(t, N, d, T)
 TriangleType3D *t;
 VectorType3D N;
 double d;
 TriangleType3D T[3];
 {
  int a,b,c;
  double tab, tac;
  double q[3];
  int i;

  /* calculate distances from the plane, for each vertex */
  q[0] = t->Vertex[0].x*N.dx+t->Vertex[0].y*N.dy+t->Vertex[0].z*N.dz+d; 
  q[1] = t->Vertex[1].x*N.dx+t->Vertex[1].y*N.dy+t->Vertex[1].z*N.dz+d; 
  q[2] = t->Vertex[2].x*N.dx+t->Vertex[2].y*N.dy+t->Vertex[2].z*N.dz+d; 

  /* case analysis ... */
  /* note that we use real "zeros" here, rather than EPSILON */
  if ( (q[0]<=0.0) && (q[1]<=0.0) && (q[2]<=0.0) ) { T[0] = *t; return(1); }
  if ( (q[0]<=0.0) && (q[1]<=0.0) && (q[2]> 0.0) ) a = 2;
  if ( (q[0]<=0.0) && (q[1]> 0.0) && (q[2]<=0.0) ) a = 1;
  if ( (q[0]< 0.0) && (q[1]>=0.0) && (q[2]>=0.0) ) a = 0;
  if ( (q[0]> 0.0) && (q[1]<=0.0) && (q[2]<=0.0) ) a = 0;
  if ( (q[0]>=0.0) && (q[1]< 0.0) && (q[2]>=0.0) ) a = 1;
  if ( (q[0]>=0.0) && (q[1]>=0.0) && (q[2]< 0.0) ) a = 2;
  if ( (q[0]>=0.0) && (q[1]>=0.0) && (q[2]>=0.0) ) { T[0] = *t; return(1); }

  if(2 == a) b = 0; else b = a+1;
  if(2 == b) c = 0; else c = b+1;

  /*
     vertex a is strictly on one side of the plane,
     b and c are on the other side. 
     One of b or c (but not both) may be ON the plane.

     So, the lines ab and ac cross (or at least reach) the plane.
     Consider these lines as :
                  ab = a + tab*(b-a)
                  ac = a + tac*(c-a)
     The blending values, tab and tac, can be found by finding the blend
     of q's which yields Q = 0 (on the plane)

   */

  if (fabs(q[a]-q[b])<EPSILON) tab=1.0;
  else                         tab=q[a]/(q[a]-q[b]);

  if (fabs(q[a]-q[c])<EPSILON) tac=1.0;
  else                         tac=q[a]/(q[a]-q[c]);

  /* construct the triangle (a, ab, ac) */
  /* copy plane equation */
  T[0].N = t->N;
  T[0].d = t->d;

  /* vertex 0 is vertex a */
  T[0].Vertex[0] = t->Vertex[a];
  T[0].Normal[0] = t->Normal[a];
  T[0].Diffuse[0] = t->Diffuse[a];
  T[0].Specular[0] = t->Specular[a];
  T[0].Specularity[0] = t->Specularity[a];
 
  /* vertex 1 is the blend between a and b */

  T[0].Vertex[1].x = Blend(t->Vertex[a].x,t->Vertex[b].x, tab);
  T[0].Vertex[1].y = Blend(t->Vertex[a].y,t->Vertex[b].y, tab);
  T[0].Vertex[1].z = Blend(t->Vertex[a].z,t->Vertex[b].z, tab);

  T[0].Normal[1].dx = Blend(t->Normal[a].dx,t->Normal[b].dx, tab);
  T[0].Normal[1].dy = Blend(t->Normal[a].dy,t->Normal[b].dy, tab);
  T[0].Normal[1].dz = Blend(t->Normal[a].dz,t->Normal[b].dz, tab);
  Normalize3D(&T[0].Normal[1]);

  T[0].Diffuse[1].r = Blend(t->Diffuse[a].r,t->Diffuse[b].r, tab);
  T[0].Diffuse[1].g = Blend(t->Diffuse[a].g,t->Diffuse[b].g, tab);
  T[0].Diffuse[1].b = Blend(t->Diffuse[a].b,t->Diffuse[b].b, tab);

  T[0].Specular[1].r =Blend(t->Specular[a].r,t->Specular[b].r, tab);
  T[0].Specular[1].g =Blend(t->Specular[a].g,t->Specular[b].g, tab);
  T[0].Specular[1].b =Blend(t->Specular[a].b,t->Specular[b].b, tab);

  T[0].Specularity[1] = Blend(t->Specularity[a],t->Specularity[b], tab);

  /* vertex 2 is the blend between a and c */

  T[0].Vertex[2].x = Blend(t->Vertex[a].x,t->Vertex[c].x, tac);
  T[0].Vertex[2].y = Blend(t->Vertex[a].y,t->Vertex[c].y, tac);
  T[0].Vertex[2].z = Blend(t->Vertex[a].z,t->Vertex[c].z, tac);

  T[0].Normal[2].dx = Blend(t->Normal[a].dx,t->Normal[c].dx, tac);
  T[0].Normal[2].dy = Blend(t->Normal[a].dy,t->Normal[c].dy, tac);
  T[0].Normal[2].dz = Blend(t->Normal[a].dz,t->Normal[c].dz, tac);
  Normalize3D(&T[0].Normal[2]);

  T[0].Diffuse[2].r = Blend(t->Diffuse[a].r,t->Diffuse[c].r, tac);
  T[0].Diffuse[2].g = Blend(t->Diffuse[a].g,t->Diffuse[c].g, tac);
  T[0].Diffuse[2].b = Blend(t->Diffuse[a].b,t->Diffuse[c].b, tac);

  T[0].Specular[2].r =Blend(t->Specular[a].r,t->Specular[c].r, tac);
  T[0].Specular[2].g =Blend(t->Specular[a].g,t->Specular[c].g, tac);
  T[0].Specular[2].b =Blend(t->Specular[a].b,t->Specular[c].b, tac);

  T[0].Specularity[2] = Blend(t->Specularity[a],t->Specularity[c], tac);

  /* fill in the Bounding Box information */
  BoundingBox(&T[0]);

  /* but keep the looping number */
  T[0].Looping = t->Looping;

  /* count the number of triangles produced */
  i = 1;

  /*
     next the triangle (b, ac, ab).
     but, if tab = 1.0, then b and ab are the same point, so no triangle
   */

  if ((1.0-EPSILON) > tab)
   {
    /* copy the plane equation */
    T[i].N = t->N;
    T[i].d = t->d;

    /* vertex 0 is vertex b */
    T[i].Vertex[0] = t->Vertex[b];
    T[i].Normal[0] = t->Normal[b];
    T[i].Diffuse[0] = t->Diffuse[b];
    T[i].Specular[0] = t->Specular[b];
    T[i].Specularity[0] = t->Specularity[b];
 
    /* vertex 1 is the blend between a and c we calculated above */
    T[i].Vertex[1] = T[0].Vertex[2];
    T[i].Normal[1] = T[0].Normal[2];
    T[i].Diffuse[1] = T[0].Diffuse[2];
    T[i].Specular[1] = T[0].Specular[2];
    T[i].Specularity[1] = T[0].Specularity[2];

    /* vertex 2 is the blend between a and b we calculated above */
    T[i].Vertex[2] = T[0].Vertex[1];
    T[i].Normal[2] = T[0].Normal[1];
    T[i].Diffuse[2] = T[0].Diffuse[1];
    T[i].Specular[2] = T[0].Specular[1];
    T[i].Specularity[2] = T[0].Specularity[1];

    BoundingBox(&T[i]);

    /* but keep the looping number */
    T[i].Looping = t->Looping;

    i++;
   }
  /*
     next the triangle (b, c, ac)
     but, if tac = 1.0, then b and ac are the same point, so no triangle
   */

  if ((1.0-EPSILON) > tac)
   {
    /* copy the plane equation */
    T[i].N = t->N;
    T[i].d = t->d;

    /* vertex 0 is vertex b */
    T[i].Vertex[0] = t->Vertex[b];
    T[i].Normal[0] = t->Normal[b];
    T[i].Diffuse[0] = t->Diffuse[b];
    T[i].Specular[0] = t->Specular[b];
    T[i].Specularity[0] = t->Specularity[b];
 
    /* vertex 1 is vertex c */
    T[i].Vertex[1] = t->Vertex[c];
    T[i].Normal[1] = t->Normal[c];
    T[i].Diffuse[1] = t->Diffuse[c];
    T[i].Specular[1] = t->Specular[c];
    T[i].Specularity[1] = t->Specularity[c];

    /* vertex 2 is the blend between a and c we calculated above */
    T[i].Vertex[2] = T[0].Vertex[2];
    T[i].Normal[2] = T[0].Normal[2];
    T[i].Diffuse[2] = T[0].Diffuse[2];
    T[i].Specular[2] = T[0].Specular[2];
    T[i].Specularity[2] = T[0].Specularity[2];

    BoundingBox(&T[i]);

    /* but keep the looping number */
    T[i].Looping = t->Looping;

    i++;
   }

  /*
     it appears that sometimes a triangle just doesn't subdivide
     The following is strictly kludge control, but it is (VERY VERY RARELY)
     necessary 
   */

  if (i>1)
   {
    int j,k,l,nomatch;
    double Closest,dx,dy,d2;

    for(j=0;j<i;j++)
     {
      for(k=0;k<2;k++)
       {
        Closest = -1;
        for(l=0;l<2;l++)
         {
          dx=t->Vertex[l].x - T[j].Vertex[k].x;
          dy=t->Vertex[l].y - T[j].Vertex[k].y;
          d2 = dx*dx + dy*dy;
          if((-1 == Closest) || (d2<Closest)) Closest=d2;
         }
        if (nomatch = (EPSILON < Closest)) break;
       }
      if (!nomatch)
       {
        if (DEBUG) fprintf(stderr,"Subdivide can't\n");
        if (j != 0) T[0] = T[j];
        i = 1;
        break;
       }
     } 
   }

  if (DEBUG) fprintf(stderr,"Sudivide returning %d triangles\n",i);
  if (DEBUG)
   {
    int j;
    fprintf(stderr,"t was {[%f,%f,%f],[%f,%f,%f],[%f,%f,%f]}\n",
                      t->Vertex[0].x, t->Vertex[0].y, t->Vertex[0].z,  
                      t->Vertex[1].x, t->Vertex[1].y, t->Vertex[1].z,  
                      t->Vertex[2].x, t->Vertex[2].y, t->Vertex[2].z);
    for(j=0;j<i;j++)
     fprintf(stderr,"T[%d] was {[%f,%f,%f],[%f,%f,%f],[%f,%f,%f]}\n",
                      j,
                      T[j].Vertex[0].x, T[j].Vertex[0].y, T[j].Vertex[0].z,  
                      T[j].Vertex[1].x, T[j].Vertex[1].y, T[j].Vertex[1].z,  
                      T[j].Vertex[2].x, T[j].Vertex[2].y, T[j].Vertex[2].z);
   }

  return(i);
 }

static int PreprocessTriangles(n, T, EyePoint)
 int n;
 TriangleType3D T[];
 PointType3D EyePoint;
 {
  int s,t;
  VectorType3D A, B, C;
  int v;
  double back;

  /* t consumes triangles, s produces them */
  for(s=0,t=0;t<n;t++)
   {
    /*
      cull triangles which have any vertex on wrong side of hither 
      this ought to be fixed to clip the triangle against the hither
      plane - producing either 0, 1, or 2 triangles

      but then, we wouldn't be able to do this in place...
     */

    if (0 == ClipLineHither(&T[t].Vertex[0], &T[t].Vertex[0])) continue;
    if (0 == ClipLineHither(&T[t].Vertex[1], &T[t].Vertex[1])) continue;
    if (0 == ClipLineHither(&T[t].Vertex[2], &T[t].Vertex[2])) continue;

    /* calculate normal */
    
    A.dx = (T[t].Vertex[1].x) - (T[t].Vertex[0].x);
    A.dy = (T[t].Vertex[1].y) - (T[t].Vertex[0].y);
    A.dz = (T[t].Vertex[1].z) - (T[t].Vertex[0].z);
    Normalize3D(&A);

    B.dx = (T[t].Vertex[2].x) - (T[t].Vertex[0].x);
    B.dy = (T[t].Vertex[2].y) - (T[t].Vertex[0].y);
    B.dz = (T[t].Vertex[2].z) - (T[t].Vertex[0].z);
    Normalize3D(&B);

    if (LeftHanded) T[t].N = Cross3D(B,A); else T[t].N = Cross3D(A,B);
    Normalize3D(&T[t].N);

    /*
      cull back-facing (and on-edge) triangles 
      by comparing the face normal with the vector from the EyePoint
      to any vertex

      or, perhaps flip the normals

     */

    C.dx = T[t].Vertex[0].x - EyePoint.x;
    C.dy = T[t].Vertex[0].y - EyePoint.y;
    C.dz = T[t].Vertex[0].z - EyePoint.z;
    Normalize3D(&C);

    if (-EPSILON <= Dot3D(T[t].N, C))
     if (!TwoSided) continue;
     else
      {
       T[t].N.dx = -T[t].N.dx;
       T[t].N.dy = -T[t].N.dy;
       T[t].N.dz = -T[t].N.dz;
       for(v=0;v<3;v++)
        {
         T[t].Normal[v].dx = -T[t].Normal[v].dx;
         T[t].Normal[v].dy = -T[t].Normal[v].dy;
         T[t].Normal[v].dz = -T[t].Normal[v].dz;
	}
      }     

    /* solve the rest of the plane equation */
    A.dx = T[t].Vertex[0].x; A.dy = T[t].Vertex[0].y; A.dz = T[t].Vertex[0].z;
    T[t].d = -Dot3D(A,T[t].N);

    /* does ANY vertex normal face away from us? */
    /* if so, make it perpendicular to the line of sight to that vertex */
    for(v=0;v<3;v++)
     {
      C.dx = T[t].Vertex[v].x - EyePoint.x;
      C.dy = T[t].Vertex[v].x - EyePoint.y;
      C.dz = T[t].Vertex[v].x - EyePoint.z;
      Normalize3D(&C);
      back = Dot3D(C,T[t].Normal[v]);
      if (back>0.0)
       {
        /* it does!, subtract out the away-facing component */
        T[t].Normal[v].dx -= back*C.dx;
        T[t].Normal[v].dy -= back*C.dy;
        T[t].Normal[v].dz -= back*C.dz;
       } 
     }

    /* If we want facets, replace the Vertex normals with the face normal */
    if (Facets) 
     for(v=0;v<3;v++)
      T[t].Normal[v] = T[t].N;

    BoundingBox(&T[t]);

    /* copy forward */
    T[s++] = T[t];
   } 
  return(s); /* number of surviving triangles */
 }

int ByMaxZ(s,t)
 TriangleType3D *s, *t;
 {
  double dz;

  dz = (s->MaxZ) - (t->MaxZ);
  if      (dz<0.0) return(-1);
  else if (0.0<dz) return(1);
  return(0);
 }

static void DisplayT(t)
 TriangleType3D *t;
 {
  int v;  
  sgpColorType Colors[3];

  if (Preview)
   {
    sgpColor(InteriorColor);
    SolidTriangle3D(t->Vertex);
    sgpColor(OutlineColor);
    if (TensorProduct)
     {
      double dx,dy,dz,d1,d2,d3;

      dx = (t->Vertex[0].x)-(t->Vertex[1].x);
      dy = (t->Vertex[0].y)-(t->Vertex[1].y);
      dz = (t->Vertex[0].z)-(t->Vertex[1].z);
      d1 = dx*dx + dy*dy + dz*dz;

      dx = (t->Vertex[1].x)-(t->Vertex[2].x);
      dy = (t->Vertex[1].y)-(t->Vertex[2].y);
      dz = (t->Vertex[1].z)-(t->Vertex[2].z);
      d2 = dx*dx + dy*dy + dz*dz;

      dx = (t->Vertex[2].x)-(t->Vertex[0].x);
      dy = (t->Vertex[2].y)-(t->Vertex[0].y);
      dz = (t->Vertex[2].z)-(t->Vertex[0].z);
      d3 = dx*dx + dy*dy + dz*dz;

      if ((d1 <= d2) || (d1 <= d3)) Line3D(t->Vertex[0], t->Vertex[1]);    
      if ((d2 <= d3) || (d2 <= d1)) Line3D(t->Vertex[1], t->Vertex[2]);    
      if ((d3 <= d1) || (d3 <= d2)) Line3D(t->Vertex[2], t->Vertex[0]);    
     }
    else
     {
      Line3D(t->Vertex[0], t->Vertex[1]);    
      Line3D(t->Vertex[1], t->Vertex[2]);    
      Line3D(t->Vertex[2], t->Vertex[0]);    
     }
   }
  else
   {
    for(v=0;v<3;v++)
     {
      SetSurfaceReflectance(t->Diffuse[v], 
                            t->Specular[v],
                            t->Specularity[v]); 
      Colors[v] = ColorAtPoint(t->Vertex[v], t->Normal[v]);
     } 
    ShadeTriangle3D(t->Vertex, Colors);
   } 
 }

void RenderTriangles(n, T, Tmax)
 int n;
 TriangleType3D T[];
 int Tmax;
 {
  PointType3D EyePoint;
  VectorType3D OpticalAxis, Up;
  int m;
  int OK;
  int P, Q, s;
  VectorType3D A;
  double q0, q1, q2, qEye;
  double p0, p1, p2, pEye;
  sgpPointType A0,A1,A2,B0,B1,B2;
  int i,j;
  TriangleType3D MoreT[3];
  int Zoverlap, Xoverlap, Yoverlap;
  int Qinfront, Pbehind;
  int XYoverlap, Subdivide;
  int Retry, Panic, MaxLoop;
  int HighWater;

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  /* we need a unit-length OpticalAxis later, so normalize now */
  Normalize3D(&OpticalAxis);

  m = PreprocessTriangles(n, T, EyePoint);

  if (m <= 0) return; /* nothing to see */

  if (2 == Preview)
   {
    for(P=0; P<m; P++) DisplayT(&T[P]); 
    return;
   }
  qsort(T, m, sizeof(TriangleType3D), ByMaxZ);
  
  /* These are meters */
  Zoverlap = 0; Xoverlap = 0; Yoverlap = 0;
  Qinfront = 0; Pbehind = 0; 
  XYoverlap = 0;  Subdivide = 0; 
  Retry = 0; Panic = 0; MaxLoop = 0;
  HighWater = m-1;

  for(P=m-1; P>=0;)
   {
    if (DEBUG) fprintf(stderr,"P = %d\n",P);
    /* try to prove that P does not obscure any other triangle Q */
    OK = -1;  /* assume the best */
    for(Q= P-1; Q >=0; Q--)
     {  
       /*
         check z-overlap.
         this could probably be optimized, but it's fast enough

         if P is completely behind Q, then P does not obscure Q
        */
      Zoverlap++;
      if (T[Q].MaxZ <= T[P].MinZ) continue;
       /*
         check x-overlap
         if P and Q are separated in X, or Y, P does not obscure Q
        */
      Xoverlap++;
      if (T[Q].MaxX <= T[P].MinX) continue;
      if (T[P].MaxX <= T[Q].MinX) continue;
       /* check y-overlap */
      Yoverlap++;
      if (T[Q].MaxY <= T[P].MinY) continue;
      if (T[P].MaxY <= T[Q].MinY) continue;
       /*
          check P behind Q
          if all vertices of P are on the opposite side of Q  from
          the EyePoint, then P does not obscure Q
        */ 
      Pbehind++;
      qEye = EyePoint.x*T[Q].N.dx
            +EyePoint.y*T[Q].N.dy
            +EyePoint.z*T[Q].N.dz
            +           T[Q].d;
      q0 = T[P].Vertex[0].x*T[Q].N.dx
          +T[P].Vertex[0].y*T[Q].N.dy
          +T[P].Vertex[0].z*T[Q].N.dz
          +                 T[Q].d;
      q1 = T[P].Vertex[1].x*T[Q].N.dx
          +T[P].Vertex[1].y*T[Q].N.dy
          +T[P].Vertex[1].z*T[Q].N.dz
          +                 T[Q].d;
      q2 = T[P].Vertex[2].x*T[Q].N.dx
          +T[P].Vertex[2].y*T[Q].N.dy
          +T[P].Vertex[2].z*T[Q].N.dz
          +                 T[Q].d;
      if (DEBUG)
       fprintf(stderr,"qEye = %g, q0 = %g, q1 = %g, q2 = %g\n",
                        qEye,q0,q1,q2);   

      if      (  (qEye>=-EPSILON)
               &&(q0<=EPSILON)
               &&(q1<=EPSILON)
               &&(q2<=EPSILON)    ) continue;
      else if (  (qEye<=EPSILON)
               &&(q0>=-EPSILON)
               &&(q1>=-EPSILON)
               &&(q2>=-EPSILON)   ) continue;
       /*
         check Q in front of P
         if all the vertices of Q are on the same side of P as
         the EyePoint, then P does not obscure Q 
        */
      Qinfront++;
      pEye = EyePoint.x*T[P].N.dx
            +EyePoint.y*T[P].N.dy
            +EyePoint.z*T[P].N.dz
            +           T[P].d;  
      p0 = T[Q].Vertex[0].x*T[P].N.dx
          +T[Q].Vertex[0].y*T[P].N.dy
          +T[Q].Vertex[0].z*T[P].N.dz
          +                 T[P].d;
      p1 = T[Q].Vertex[1].x*T[P].N.dx
          +T[Q].Vertex[1].y*T[P].N.dy 
          +T[Q].Vertex[1].z*T[P].N.dz
          +                 T[P].d;
      p2 = T[Q].Vertex[2].x*T[P].N.dx
          +T[Q].Vertex[2].y*T[P].N.dy
          +T[Q].Vertex[2].z*T[P].N.dz
          +                 T[P].d;
      if (DEBUG)
       fprintf(stderr,"pEye = %g, p0 = %g, p1 = %g, p2 = %g\n",
                      pEye,p0,p1,p2);   
      if      (  (pEye<=EPSILON)
               &&(p0<=EPSILON)
               &&(p1<=EPSILON)
               &&(p2<=EPSILON)   ) continue;
      else if (  (pEye>=-EPSILON)
               &&(p0>=-EPSILON)
               &&(p1>=-EPSILON)
               &&(p2>=-EPSILON)  ) continue;

       /*
         check xy-overlap
            if any point of P is inside Q,
         or if any point of Q is inside P,
         or if any edge of P intersects any edge of Q
         then these triangles overlap in XY, and P may obscure Q.

         On the other hand, if NONE of these is true, then P
         and Q project to disjoint parts of the screen, and P can't
         obscure Q
        */
      XYoverlap++;
      World3DTosgpWorld(T[Q].Vertex[0],&A0);
      World3DTosgpWorld(T[Q].Vertex[1],&A1);
      World3DTosgpWorld(T[Q].Vertex[2],&A2);
      World3DTosgpWorld(T[P].Vertex[0],&B0);
      World3DTosgpWorld(T[P].Vertex[1],&B1);
      World3DTosgpWorld(T[P].Vertex[2],&B2);

      if      (InsideP(A0, B0, B1, B2)) OK = 0;
      else if (InsideP(A1, B0, B1, B2)) OK = 0;
      else if (InsideP(A2, B0, B1, B2)) OK = 0;
      else if (InsideP(B0, A0, A1, A2)) OK = 0;
      else if (InsideP(B1, A0, A1, A2)) OK = 0;
      else if (InsideP(B2, A0, A1, A2)) OK = 0;
      else if (LineSegmentIntersectionP(A0,A1,B0,B1)) OK = 0;
      else if (LineSegmentIntersectionP(A0,A1,B1,B2)) OK = 0;
      else if (LineSegmentIntersectionP(A0,A1,B2,B0)) OK = 0;
      else if (LineSegmentIntersectionP(A1,A2,B0,B1)) OK = 0;
      else if (LineSegmentIntersectionP(A1,A2,B1,B2)) OK = 0;
      else if (LineSegmentIntersectionP(A1,A2,B2,B0)) OK = 0;
      else if (LineSegmentIntersectionP(A2,A0,B0,B1)) OK = 0;
      else if (LineSegmentIntersectionP(A2,A0,B1,B2)) OK = 0;
      else if (LineSegmentIntersectionP(A2,A0,B2,B0)) OK = 0;

      if (OK) continue;

      /*
        It appears that P obscures Q. We should swap P and Q 
        and try to prove that Q doesn't obscure any other triangle.

        But, this swapping process can loop.  So...first check to
        see if either triangle has been moved already.  If not,
        then swap, mark, and continue
       */
      if (  (0 == T[P].Looping)
          &&(0 == T[Q].Looping) )
       {
        MoreT[0] = T[Q]; T[Q] = T[P]; T[P] = MoreT[0];
        T[P].Looping++;
        if (T[P].Looping > MaxLoop) MaxLoop = T[P].Looping;
        break;
       }

      /*
        Oh well, it looks like we may have a loop here.
        Let's try to subdivide the triangles.

        First, try to subdivide P against the plane of Q 
       */
      
      Subdivide++;
      s = SubdivideTriangle(&T[P--], T[Q].N, T[Q].d, MoreT);

      /*
        But...what happens if Subdivide yields only 1 triangle?
        P must be completely on the near side of Q.
        Swap P and Q and try to subdivide the other way.

        Remember that the remains of T[P] are in MoreT[0].

        It may have been "trimmed", so we prefer to use it rather
        than the original (still in T[P]).

        On the other hand, we want to keep the "Looping" count rising,
        so salvage that from T[P]. 
       */

      if (1 == s)
       {
        if (DEBUG)
         fprintf(stderr,"Subdivision yielded one triangle\n");
        P++;
        MoreT[0].Looping = T[P].Looping;
        T[P] = T[Q]; T[Q] = MoreT[0];
        T[P].Looping++;
        if (T[P].Looping > MaxLoop) MaxLoop = T[P].Looping;
        if (T[P].Looping > MAXLOOPING) { Panic++; OK = -1; continue; }
        else                           { Retry++;          break;    }
        s = SubdivideTriangle(&T[P--], T[Q].N, T[Q].d, MoreT);
       }

      /*
        And now, suppose we only get 1 triangle again?
        Then, our new P must be completely on the FAR side of Q.  
       
        But, we think we're in a loop.

        Maybe the source of the loop is somewhere else in the list.
        Maybe we're losing to numerical instability.

        We really ought to test P against ALL of the triangles,
        but this may continue to loop.

        If a triangle gets this far too many times (MAXLOOOPING),
        we try to break the loop by assuming that the new P doesn't 
        obscure any triangle that we've already tested Q against.  

        This may lose, but then again, we won't loop forever.  Do you
        want the right answer at t=infinity, or would you prefer the
        wrong answer right now?

        It SHOULD be true that we don't need this panic-trap.  But,
        better to have it and not need it, than to need it and not have it.

        Again, the triangle may have been trimmed, so we prefer the
        data from MoreT[0], and the Looping from T[P].
       */

      if (1 == s)
       {
        if (DEBUG)
         fprintf(stderr,"Subdivision yielded one triangle AGAIN\n");
        P++; 
        MoreT[0].Looping = T[P].Looping;
        T[P] = MoreT[0];
        T[P].Looping++;
        if (T[P].Looping > MaxLoop) MaxLoop = T[P].Looping;
        if (T[P].Looping > MAXLOOPING) { Panic++; OK = -1; continue; }
        else                           { Retry++;          break;    }
       }

      /*
        If we get more than one new triangle, insert
        them into the list, using MaxZ
        as a hint about where to put each one.

        Perhaps we ought to just put them on the end of the list,
        and let the rest of this process sort them out?

        Perhaps we ought to just sort THESE triangles by MaxZ?

        The code is written as an insertion sort that assumes the
        current lt is already sorted.  That's not true, or course, 
        but the Z sort is only a hint, anyway.
       */
      
      for(i=0;i<s;i++)
       {
        for(j=P++; T[j].MaxZ>MoreT[i].MaxZ;j--) T[j+1] = T[j];
        T[j+1] = MoreT[i];
       }
      if (P>HighWater) 
       {
        HighWater = P;
        if (VERBOSE) fprintf(stderr,"RenderTriangles: new HighWater = %d\n",
                                HighWater);
       }
      if (P>(Tmax-5)) /* a tad conservative */
       {
        fprintf(stderr,"RenderTriangles: out of space\n");
        return;
       }
      break;  /* We have a new P, start over, from the top */
     }
    if (OK) DisplayT(&T[P--]);
   }
  /*
    finally, report on our good deeds
   */ 
  if (VERBOSE)
   {
    fprintf(stderr,"RenderTriangles:\n");
    fprintf(stderr,"                 N         = %10d\n",n);
    fprintf(stderr,"                 M         = %10d\n",m);
    fprintf(stderr,"                 HighWater = %10d\n",HighWater);
    fprintf(stderr,"                 Zoverlap  = %10d\n",Zoverlap);
    fprintf(stderr,"                 Xoverlap  = %10d\n",Xoverlap);
    fprintf(stderr,"                 Yoverlap  = %10d\n",Yoverlap);
    fprintf(stderr,"                 Pbehind   = %10d\n",Pbehind);
    fprintf(stderr,"                 Qinfront  = %10d\n",Qinfront);
    fprintf(stderr,"                 XYoverlap = %10d\n",XYoverlap);
    fprintf(stderr,"                 Subdivide = %10d\n",Subdivide);
    fprintf(stderr,"                 Retry     = %10d\n",Retry);
    fprintf(stderr,"                 Panic     = %10d\n",Panic);
    fprintf(stderr,"                 MaxLoop   = %10d\n",MaxLoop);
   }
 }
