/*
  File: Mickey.c
  Authors: Jinling Lee (Bill),
           K.R. Sloan   
  Last Modified:  26 April 1991
  Purpose: Draw a crude approximation to Mickey Mouse, using
           4 intersecting spheres
           Note - all measurements are in meters.  
 */

#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"
#include "Triangles.h"
#include "Render.h"
#include "Scanline.h"

double atof();

#define MAXTRIANGLES (16376)
#define PRECISION (10)

int VERBOSE = 0;    /* exported for the real renderers to see */
static int DEBUG = 0;
static Preview = 0;
static Hidden = 1;
static Quality = 0;
static sgpColorType InteriorColor = { 1.0, 1.0, 1.0 };
static sgpColorType OutlineColor  = { 0.0, 0.0, 0.0 };

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,
"Usage is\n\t%s [-v][-p][-P][-D][-Q][-e x y z][-o x y z][-u x y z][-f f][-x x]\n",
                 RoutineName);
 }

static int t=0;
static TriangleType3D T[MAXTRIANGLES];  /* that should be enough... */  
void DrawSphere(Center, Radius, Precision,
                Diffuse, Specular, Specularity,
                OpticalAxis)
 PointType3D Center;
 double Radius;
 int Precision;
 sgpColorType Diffuse, Specular;
 double Specularity;
 VectorType3D OpticalAxis;
 {
  int i, j;
  PointType3D Pa, Pb, Pc, Pd;
  VectorType3D Na, Nb, Nc, Nd;
  double alpha, beta, e;

  fprintf(stderr,"DrawSphere: Precision = %d\n",Precision);

  /* 
    Divide the sphere into triangles.
    The sphere is centered at the origin, so position on the sphere
    is "the same as" the surface normal.

   */
  e = M_PI/Precision;
  for(i=0; i<Precision; i++)
   {
    for(j=0; j<2*Precision; j++)
     {
      alpha = i*e;
      beta  = j*e;

      /* normals */

      Na.dx = sin(alpha)*cos(beta); 
      Na.dy = sin(alpha)*sin(beta);
      Na.dz = cos(alpha);

      Nb.dx = sin(alpha+e)*cos(beta);
      Nb.dy = sin(alpha+e)*sin(beta);
      Nb.dz = cos(alpha+e);

      Nc.dx = sin(alpha+e)*cos(beta+e);
      Nc.dy = sin(alpha+e)*sin(beta+e);
      Nc.dz = cos(alpha+e);

      Nd.dx = sin(alpha)*cos(beta+e);
      Nd.dy = sin(alpha)*sin(beta+e);
      Nd.dz = cos(alpha);

      /* cull backfacing triangles */
      if (  (0.0 <= Dot3D(Na,OpticalAxis)) 
          &&(0.0 <= Dot3D(Nb,OpticalAxis))
          &&(0.0 <= Dot3D(Nc,OpticalAxis)) ) continue;
          
      /* points */

      Pa.x = Center.x + Radius*Na.dx;
      Pa.y = Center.y + Radius*Na.dy;
      Pa.z = Center.z + Radius*Na.dz;
     
      Pb.x = Center.x + Radius*Nb.dx;
      Pb.y = Center.y + Radius*Nb.dy;
      Pb.z = Center.z + Radius*Nb.dz;

      Pc.x = Center.x + Radius*Nc.dx;
      Pc.y = Center.y + Radius*Nc.dy;
      Pc.z = Center.z + Radius*Nc.dz;

      Pd.x = Center.x + Radius*Nd.dx;
      Pd.y = Center.y + Radius*Nd.dy;
      Pd.z = Center.z + Radius*Nd.dz;

      /* draw the triangles */

      T[t].Vertex[0] = Pa; T[t].Vertex[1] = Pb; T[t].Vertex[2] = Pc;
      T[t].Normal[0] = Na; T[t].Normal[1] = Nb; T[t].Normal[2] = Nc;
      T[t].Diffuse[0] = Diffuse;
      T[t].Diffuse[1] = Diffuse;
      T[t].Diffuse[2] = Diffuse;
      T[t].Specular[0] = Specular;
      T[t].Specular[1] = Specular;
      T[t].Specular[2] = Specular;
      T[t].Specularity[0] = Specularity;
      T[t].Specularity[1] = Specularity;
      T[t].Specularity[2] = Specularity;
      t++;
      T[t].Vertex[0] = Pa; T[t].Vertex[1] = Pc; T[t].Vertex[2] = Pd;
      T[t].Normal[0] = Na; T[t].Normal[1] = Nc; T[t].Normal[2] = Nd;
      T[t].Diffuse[0] = Diffuse;
      T[t].Diffuse[1] = Diffuse;
      T[t].Diffuse[2] = Diffuse;
      T[t].Specular[0] = Specular;
      T[t].Specular[1] = Specular;
      T[t].Specular[2] = Specular;
      T[t].Specularity[0] = Specularity;
      T[t].Specularity[1] = Specularity;
      T[t].Specularity[2] = Specularity;

      if (Hidden) t++;
      else
       {
        sgpColor(OutlineColor);
        Line3D(T[t].Vertex[0], T[t].Vertex[1]);
        Line3D(T[t].Vertex[1], T[t].Vertex[2]);
        Line3D(T[t].Vertex[2], T[t].Vertex[0]);
       }
     }
   }
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  static int ArgsParsed = 0;
  static PointType3D  EyePoint    = {  3.0,  3.0,  3.0 };
  static VectorType3D OpticalAxis = { -1.0, -1.0, -1.0 };
  static VectorType3D Up          = {  0.0,  1.0,  0.0 };
  static double FocalLength = 0.0040;
  static sgpRectangleType Window   = {-0.0015, -0.0012, 0.0015, 0.0012};
  static sgpRectangleType Viewport = {-0.125,   0.00,   1.125,  1.00};  
  static double L = 100.0;      /* half width of light cube */
  double Exposure = 0.5;
  PointType3D LightPosition;
  sgpColorType Color;
  sgpColorType Ambient, Diffuse, Specular;
  double Specularity;
  PointType3D Center;
  static double Radius;
  int Precision;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'e':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         EyePoint.x = atof(argv[ArgsParsed++]);
         EyePoint.y = atof(argv[ArgsParsed++]);
         EyePoint.z = atof(argv[ArgsParsed++]);
         break;
        case 'o':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         OpticalAxis.dx = atof(argv[ArgsParsed++]);
         OpticalAxis.dy = atof(argv[ArgsParsed++]);
         OpticalAxis.dz = atof(argv[ArgsParsed++]);
         break;
        case 'u':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Up.dx = atof(argv[ArgsParsed++]);
         Up.dy = atof(argv[ArgsParsed++]);
         Up.dz = atof(argv[ArgsParsed++]);
         break;
        case 'f':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         FocalLength = atof(argv[ArgsParsed++]);
         break;
        case 'x':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         Exposure = atof(argv[ArgsParsed++]);
         break;
        case 'p':
         Preview = 1;
         break;
        case 'P':
         Hidden = 0;
         Preview = 1;
         break;
        case 'Q':
         Quality = 1;
         Hidden = 1;
         Preview = 0;
         break;
        case 'D':
         DEBUG = 1;
         break;
        case 'v':
         VERBOSE = 1;
         break;
        case 'h': 
        default:
         usage(); exit(-1);
       }
     }
    else { usage(); exit (-1); }
   }   

  while (ArgsParsed < argc) { usage(); exit (-1); }

  sgpInit(ColorDisplay);
  sgpSetWindow(Window);
  sgpSetViewport(Viewport);

  sgpGrayLevel(0.5); sgpClearScreen();

  Camera3D(EyePoint, OpticalAxis, Up);
  Lens3D(FocalLength);  SetHither(FocalLength);

  GetCamera3D(&EyePoint, &OpticalAxis, &Up);

  if (VERBOSE)
   {
    fprintf(stderr,"%s: EyePoint = { %f, %f, %f }\n", RoutineName,
                        EyePoint.x,
                        EyePoint.y,
                        EyePoint.z);
    fprintf(stderr,"%s: OpticalAxis = { %f, %f, %f }\n", RoutineName,
                        OpticalAxis.dx,
                        OpticalAxis.dy,
                        OpticalAxis.dz);
    fprintf(stderr,"%s: Up = { %f, %f, %f }\n", RoutineName,
                        Up.dx,
                        Up.dy,
                        Up.dz);
    fprintf(stderr,"%s: FocalLength = %f\n", RoutineName, FocalLength);
   }
  if (Preview)
   {
    if (VERBOSE)
     {
      fprintf(stderr,"%s: Previewing\n", RoutineName);
      fprintf(stderr,"%s: InteriorColor = { %f, %f, %f }\n", RoutineName,
                          InteriorColor.r, InteriorColor.g, InteriorColor.b);
      fprintf(stderr,"%s: OutlineColor = { %f, %f, %f }\n", RoutineName,
                          OutlineColor.r, OutlineColor.g, OutlineColor.b);
     }
    RenderMode(Preview, InteriorColor, OutlineColor);
    sgpColor(InteriorColor); sgpClearScreen();
   }
  else
   {
    Ambient.r = 0.2; Ambient.g = 0.2; Ambient.b = 0.5;
    SetAmbient(Ambient);

    LightPosition.x=  L; LightPosition.y= -L; LightPosition.z= -L;
    Color.r = 1.0; Color.g = 0.0; Color.b = 0.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x= -L; LightPosition.y=  L; LightPosition.z= -L;
    Color.r = 0.0; Color.g = 1.0; Color.b = 0.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x= -L; LightPosition.y= -L; LightPosition.z=  L;
    Color.r = 0.0; Color.g = 0.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x=  L; LightPosition.y=  L; LightPosition.z= -L;
    Color.r = 1.0; Color.g = 1.0; Color.b = 0.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x=  L; LightPosition.y= -L; LightPosition.z=  L;
    Color.r = 1.0; Color.g = 0.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);
    LightPosition.x= -L; LightPosition.y=  L; LightPosition.z=  L;
    Color.r = 0.0; Color.g = 1.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    LightPosition.x = L; LightPosition.y = L; LightPosition.z=  L;
    Color.r = 1.0; Color.g = 1.0; Color.b = 1.0;
    AddLightSource(LightPosition, Color);
    if (VERBOSE)
     fprintf(stderr,"%s: Light source is at { %f, %f, %f }\n", RoutineName,
                         LightPosition.x,
                         LightPosition.y,
                         LightPosition.z);

    SetExposure(Exposure);
    if (VERBOSE)
     fprintf(stderr,"%s: Exposure = %f\n", RoutineName, Exposure);

   }
  Diffuse.r  = 0.5;  Diffuse.g  = 0.5;  Diffuse.b  = 0.5;
  Specular.r = 0.5;  Specular.g = 0.5;  Specular.b = 0.5;
  Specularity = 6.0;

  Precision = PRECISION;
  while ((Precision*Precision*5)>MAXTRIANGLES) Precision--;

  /* head */
  Center.x = 0.0; Center.y = 0.0; Center.z = 0.0;
  Radius = 1.0;
  DrawSphere(Center, Radius, Precision, 
             Diffuse, Specular, Specularity,
             OpticalAxis);
  Center.x = 0.25; Center.y = 0.25; Center.z = 0.25;

  /* ears */
  Center.x = 0.7; Center.y = 0.7; Center.z = 0.0;
  Radius = 0.5;
  DrawSphere(Center, Radius, Precision, 
             Diffuse, Specular, Specularity,
             OpticalAxis);
  Center.x = -0.7; Center.y = 0.7; Center.z = 0.0;
  Radius = 0.5;
  DrawSphere(Center, Radius, Precision, 
             Diffuse, Specular, Specularity,
             OpticalAxis);

  /* nose */
  Center.x = 0.0; Center.y = 0.0; Center.z = 1.0;
  Radius = 0.25;
  DrawSphere(Center, Radius, Precision, 
             Diffuse, Specular, Specularity,
             OpticalAxis);

  if (Hidden)
   if (Quality)
    {
     if (VERBOSE)
      fprintf(stderr,"%s: calling ScanTriangles(%d, T, %d)\n",
                     RoutineName,t,MAXTRIANGLES);
     ScanTriangles(t,T,MAXTRIANGLES);
     if (VERBOSE)
      fprintf(stderr,"%s: ScanTriangles returns\n",RoutineName);
    }
   else
    {
     if (VERBOSE)
      fprintf(stderr,"%s: calling RenderTriangles(%d, T, %d)\n",
                    RoutineName,t,MAXTRIANGLES);
     RenderTriangles(t,T,MAXTRIANGLES);

     if (VERBOSE) fprintf(stderr,"%s: RenderTriangles returns\n",RoutineName);
    }

  sgpQuit();
  if (VERBOSE) fprintf(stderr,"-30-\n");
  exit(0);
 }
