/*
 *  This file is part of ixemul.library for the Amiga.
 *  Copyright (C) 1991, 1992  Markus M. Wild
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define KERNEL
#include "ixemul.h"
#include <sys/wait.h>
#include <ctype.h>

/* 2.0 support */
#include <utility/tagitem.h>
#include <dos/dostags.h>

#define alloca __builtin_alloca

int
system (const char *cmd)
{
  int rc = -1, err = 0;
  int stack_size;
  struct CommandLineInterface *CLI;
  struct Process *me;
  int omask;

  /* I retry with our new signal mechanism ;-) */
  omask = syscall (SYS_sigsetmask, ~0);

  me = (struct Process *)FindTask(0);
  CLI = BTOCPTR (me->pr_CLI);
  stack_size = CLI ? CLI->cli_DefaultStack * 4 : me->pr_StackSize;
  if (stack_size <= 4096) stack_size = 250000;

  /* limited support to allow starting of files in the current directory
   * with `./foo'. The better approach would use the __plock() trick to
   * parse the command, LoadSeg it. But then this approach would have to
   * do the whole redirection stuff on its own.. */
  while (isspace (*cmd)) cmd++;
  while (cmd[0] == '.' && cmd[1] == '/') cmd += 2;

    {
      struct TagItem tags [] = {
	/* a stack of 4K is generally ways too small.. */
	{ NP_StackSize, stack_size, },
	{ TAG_END, 0, }
      };

      rc = SystemTagList ((UBYTE *)cmd, tags);
      err = __ioerr_to_errno (IoErr ());
    }

  syscall (SYS_sigsetmask, omask);

  if (rc > 128)
    errno = EINTR;
  else 
    errno = err;

  /* according to the BSD manual, system() should return the `exit status'
   * of the shell, the implementation returns the wait-status. So I return
   * an artificial wait status for now ... */
  return (rc >= 128) ? W_EXITCODE (0, rc & 0x7f) : W_EXITCODE (rc, 0);
}
