/*
 *  This file is part of ixemul.library for the Amiga.
 *  Copyright (C) 1991, 1992  Markus M. Wild
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  ix_init.c,v 1.1.1.1 1994/04/04 04:30:27 amiga Exp
 *
 *  ix_init.c,v
 * Revision 1.1.1.1  1994/04/04  04:30:27  amiga
 * Initial CVS check in.
 *
 *  Revision 1.5  1993/11/05  21:55:17  mwild
 *  allocate much more space in filetable, due to introduction of sockets
 *  change default to ignore global environment
 *
 *  Revision 1.4  1992/08/09  20:52:38  amiga
 *  change to new way of remembering open libraries
 *  get prepared to deal with ttys
 *
 *  Revision 1.3  1992/05/22  01:43:46  mwild
 *  initialize buddy allocator
 *
 * Revision 1.2  1992/05/18  00:49:07  mwild
 * changed async mp to be global
 *
 * Revision 1.1  1992/05/14  19:55:40  mwild
 * Initial revision
 *
 */

#define KERNEL
#include "ixemul.h"

#include <exec/memory.h>

#ifdef DEBUG
#define DP(a) kprintf a
#else
#define DP(a)
#endif

/* not changed after the library is initialized */
struct ixemul_base *ixemulbase = 0;

/* used by gnulib.. I was too lazy to convert that library as well ;-) */
void *MathIeeeDoubBasBase, *MathIeeeDoubTransBase,
     *MathIeeeSingBasBase;

struct ExecBase *SysBase;

/* global port for asynchronous packet notification */
struct MsgPort *ix_async_mp;
extern int mp_interrupt ();

/* global port for tty handling */
struct MsgPort *ix_tty_mp;
extern int tty_interrupt ();

/* since gcc can now statically initialize unions, they're quite useful;-) */
union lib {
  char *name;
  struct Library *base;
};

void
open_libraries (union lib *libs)
{
  SysBase = *(struct ExecBase **)4;

  /* on input `libs' contains a pointer to the library name, on output it
     contains the library base pointer */
  do
    libs->base = (struct Library *) OpenLibrary (libs->name, 0) ? : (struct Library *) -1;
  while ((++libs)->name);
}

void
close_libraries (union lib *libs)
{
  do
    if (libs->base != (struct Library *) -1)
      CloseLibrary (libs->base);
  while ((++libs)->base);
}

int
compatible_hardware ()
{
  /* There is no guarantee that we would not already have tried to execute an
     instruction not supported by this processor unless we are running a variant
     compiled for the 68000, of course.

     Thus, this source file MUST be compiled with "-m68000".  */

  int err = 0;

#if defined (CPU68000)
  /* AFF_68020 is set for 68020 and all better CPUs.  */
  if (SysBase->AttnFlags & AFF_68020)
    err = 1;
#endif
#if defined (CPU68020)
  if (!(SysBase->AttnFlags & AFF_68020))
    err = 1;
#endif
#if defined (CPU68030)
  if (!(SysBase->AttnFlags & AFF_68030))
    err = 1;
#endif
#if defined (CPU68040)
  if (!(SysBase->AttnFlags & AFF_68040))
    err = 1;
#endif
#if defined (CPU68060)
  You might want to fix this.  The definition of AFF_68060 has been seen on the net.
#endif
#if !(defined (CPU68000) || defined (CPU68020) || defined (CPU68030) || defined (CPU68040) || defined (CPU68060))
  You loose.
#endif

#if 0
/* The library self-configures to appropriate math circumstances.  */
#if defined (FPU68881)
  if (!(SysBase->AttnFlags & AFF_68881))
    err = 1;
#else /* "FPUsoft-float"; but we can't have a dash in a symbol name... */
  if (SysBase->AttnFlags & AFF_68881)
    err = 1;
#endif
#endif

  if (err)
    {
      ix_panic ("Install the correct variant of ixemul.library for your CPU and FPU type.");
      return 0;
    }
  return 1;
}

/* these are the libraries we are potentially interested in. Not finding
   a certain library doesn't mean we can't continue. */
static union lib ix_libs[] = {
  "dos.library",
  "intuition.library",
  "graphics.library",
  "mathieeesingbas.library",
  "mathieeedoubbas.library",
  "mathieeedoubtrans.library",
  0,
};
enum { DOS_LIB=0, INTUI_LIB, GFX_LIB, SINGB_LIB, DOUBB_LIB, DOUBT_LIB };


struct ixemul_base *
ix_init (struct ixemul_base *ixbase)
{
  int i;

  DP (("ix_init1, ix_init @$%lx\n", ix_init));
  ixemulbase = ixbase;
  open_libraries (ix_libs);

  /* these ones are necessary and are very (!) unlikely to be missing */
  if (   ! (ixbase->ix_dos_base = ix_libs[DOS_LIB].base)
      || ! (ixbase->ix_intui_base = (struct IntuitionBase *) ix_libs[INTUI_LIB].base)
      || ! (ixbase->ix_gfx_base = (struct GfxBase *) ix_libs[GFX_LIB].base))
    {
      close_libraries (ix_libs);
      return 0;
    }

  DP(("ix_init2\n"));
  if (ixbase->ix_dos_base->lib_Version < 36)
    {
      ix_panic ("ixemul.library needs at least dos.library 36");
      close_libraries (ix_libs);
      return 0;
    }

  if (!compatible_hardware ())
    {
      /* User informed already.  */
      close_libraries (ix_libs);
      return 0;
    }

  DP(("ix_init3\n"));
  /* those are more or less optional, and if not available cause 
     ix_patch_functions() to replace functions that need them by aborting
     functions */
  ixbase->ix_ms_base =
    MathIeeeSingBasBase = ix_libs[SINGB_LIB].base;
  ixbase->ix_md_base =
    MathIeeeDoubBasBase = ix_libs[DOUBB_LIB].base;
  ixbase->ix_mdt_base =
    MathIeeeDoubTransBase = ix_libs[DOUBT_LIB].base;

  ixbase->ix_file_tab = (struct file *)
    AllocMem (NOFILE * sizeof(struct file), MEMF_PUBLIC | MEMF_CLEAR);
  ixbase->ix_fileNFILE = ixbase->ix_file_tab + NOFILE;
  ixbase->ix_lastf = ixbase->ix_file_tab;
  ixbase->ix_red_zone_size = 0; /* not enabled by default */
  ixbase->ix_membuf_limit = 0;
  ixbase->ix_fs_buf_factor = 64;
  ixbase->ix_watch_stack = 0;
  ixbase->ix_translate_dots = 1;
  ixbase->ix_translate_slash = 1;
  ixbase->ix_translate_symlinks = 0;
  ixbase->ix_force_translation = 0;
  ixbase->ix_ignore_global_env = 0;

  DP(("ix_init4\n"));
  /* initialize the list structures for the allocator */
  init_buddy ();

  DP(("ix_init5\n"));
  /* patch our library to optimally adapt to the given hardware */
  ix_patch_functions (ixbase);

  DP(("ix_init6\n"));
  ix_async_mp = (struct MsgPort *) CreateInterruptPort (0, 0, mp_interrupt, 0);
  ix_tty_mp = 0; /* (struct MsgPort *) CreateInterruptPort (0, 0, tty_interrupt, 0);*/

  if (ixbase->ix_file_tab && ix_async_mp /* && ix_tty_mp */)
    {
      InitSemaphore (& ixbase->ix_semaph);

      configure_context_switch ();

      NewList ((struct List *) &ixbase->ix_socket_list);

      for (i = 0; i < IX_NUM_SLEEP_QUEUES; i++)
        NewList ((struct List *) &ixbase->ix_sleep_queues[i]);

      /* pass the array over to ix_expunge () */
      ixbase->ix_libs = ix_libs;

      return ixbase;
    }
        
  if (ix_async_mp)
    DeleteInterruptPort (ix_async_mp);

  if (ix_tty_mp)
    DeleteInterruptPort (ix_tty_mp);

  if (ixbase->ix_file_tab)
    FreeMem (ixbase->ix_file_tab, NOFILE * sizeof(struct file));
  else
    ix_panic ("out of memory");

  close_libraries (ix_libs);
  
  return 0;
}      


void
ix_lock_base ()
{
  u_save.u_oldmask = u_save.p_sigmask;
  u_save.p_sigmask = ~0;
  ObtainSemaphore (& ix.ix_semaph);
}

void
ix_unlock_base ()
{
  ReleaseSemaphore (& ix.ix_semaph);
  u_save.p_sigmask = u_save.u_oldmask;
}
