/* 
Copyright (C) 1988 Free Software Foundation
    written by Doug Lea (dl@rocky.oswego.edu)

This file is part of GNU CC.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the GNU CC General Public
License for full details.

Everyone is granted permission to copy, modify and redistribute
GNU CC, but only under the conditions described in the
GNU CC General Public License.   A copy of this license is
supposed to have been given to you along with GNU CC so you
can know your rights and responsibilities.  It should be in a
file named COPYING.  Among other things, the copyright notice
and this notice must be preserved on all copies.  
*/

#include <streambuf.h>
#include <std.h>
#include <stdarg.h>
#include <values.h>

#include <sys/file.h>           // needed to determine values of O_RDONLY...

#ifndef _bufsiz
#define _bufsiz(p) ((p)->_bufsiz)
#endif

streambuf::streambuf()
     :base(0), gptr(0), pptr(0), eptr(0), alloc(0)
{}

streambuf::streambuf(char* buf, int buflen)
     : base(buf), gptr(buf), pptr(buf), eptr(buf+buflen-1), alloc(0)
{}

streambuf::~streambuf()
{
  if (alloc && (base != 0)) delete base;
}

int streambuf::doallocate()
{
  if (alloc && base != 0) delete base;
  base = new char[BUFSIZ];
  gptr = pptr = base;
  eptr = base + BUFSIZ - 1;
  alloc = 1;
  return BUFSIZ;
}

streambuf* streambuf::setbuf(char* buf, int buflen, int preloaded_count)
{
  base = gptr = buf;
  pptr = base + preloaded_count;
  eptr = base + buflen - 1;
  return this;
}

const char* streambuf::name()
{
  return 0;
}

int streambuf::overflow(int c)
{
  if (base == 0) allocate();
  return (c == EOF)? c : ((pptr <= eptr)? (*pptr++ = (char)(c)) : EOF);
}

int streambuf::underflow()
{
  return EOF;
}

int streambuf::sputs(const char* s)
{
  if (s != 0)
  {
    for(; *s != 0; ++s)
    {
      if (must_overflow(*s)) { if (overflow(*s) == EOF) return EOF; }
      else *pptr++ = *s;
    }
  }
  return 0;
}

int streambuf::sputsn(const char* s, int len)
{
  for(; --len >= 0; ++s)
  {
    if (must_overflow(*s)) { if (overflow(*s) == EOF) return EOF; }
      else *pptr++ = *s;
  }
  return 0;
}


int streambuf::is_open()
{
  return 1;
}

int streambuf::close()
{
  return 1;
}

void streambuf::error()
{
  abort();
}

streambuf* streambuf::open(const char*, open_mode)
{
  return 0;
}

streambuf* streambuf::open(const char*, io_mode, access_mode)
{
  return 0;
}
streambuf* streambuf::open(const char*, const char*)
{
  return 0;
}
streambuf* streambuf::open(int, io_mode)
{
  return 0;
}
streambuf* streambuf::open(FILE*)
{
  return 0;
}

//--------------------------------------------------------------------

filebuf::filebuf() 
     :streambuf(), fd(-1), opened(0) {}

filebuf::filebuf(int newfd) 
     : streambuf(), fd(newfd), opened(1) {}

filebuf::filebuf(int newfd, char* buf, int buflen)
     : streambuf(buf, buflen), fd(newfd), opened(1) {}

int filebuf::is_open()
{
  return opened;
}

int filebuf::close()
{
  int was = opened;
  if (was) ::close(fd);
  opened = 0;
  return was;
}

streambuf* filebuf::open(const char* name, open_mode m)
{
  if (opened) return 0;
  int mode = -1; // any illegal value
  switch (m)
  {
  case input: mode = O_RDONLY; 
              break;
  case output: mode = O_WRONLY | O_CREAT | O_TRUNC;
              break;
  case append: mode = O_APPEND | O_CREAT;
              break;
  }
  fd = ::open(name, mode, 0666);
  if (opened = (fd >= 0))
  {
    allocate();
    return this;
  }
  else
    return 0;
}

int filebuf::underflow()
{
  if (!opened) return EOF;
  if (base == 0) allocate();
  int nwanted = eptr - base + 1;
  int nread = ::read(fd, base, nwanted);
  gptr = base;
  pptr = base + nread;
  return (nread == 0)? EOF : int(*gptr);
}

int filebuf::overflow(int ch)
{
  if (!opened) return EOF;
  if (base == 0) allocate();
  if (ch != EOF)             // overflow *must* be called before really full
    *pptr++ = (char)(ch);

  int n = pptr - base;
  if (n > 0) n -= ::write(fd, base, n);
  pptr = base;
  return (n ==0)? 0 : EOF;
}

filebuf::~filebuf()
{
  close();
}

//----------------------------------------------------------------

void Filebuf::init_streambuf_ptrs()
{
  if (Fp->fp == 0 || Fp->fp->_cnt == 0)
  {
    base = gptr = pptr = eptr = 0; // let first over/under flow deal with it
  }
  else
  {
    base = Fp->fp->_base;
    eptr = base - 1 + _bufsiz(Fp->fp);
    gptr = pptr = base;
  }
}


int Filebuf::is_open()
{
  return (Fp != 0 && Fp->is_open());
}

streambuf* Filebuf::open(const char* name, io_mode m, access_mode a)
{
  if (Fp == 0)
    Fp = new File(name, m, a);
  else
    Fp->open(name, m, a);
  if (base != 0) Fp->setbuf(eptr-base+1, base);
  init_streambuf_ptrs();
  return this;
}

streambuf* Filebuf::open(const char* name, const char* m)
{
  if (Fp == 0)
    Fp = new File(name, m);
  else
    Fp->open(name, m);
  if (base != 0) Fp->setbuf(eptr-base+1, base);
  init_streambuf_ptrs();
  return this;
}

streambuf* Filebuf::open(int filedesc, io_mode m)
{
  if (Fp == 0)
    Fp = new File(filedesc, m);
  else
    Fp->open(filedesc, m);
  if (base != 0) Fp->setbuf(eptr-base+1, base);
  init_streambuf_ptrs();
  return this;
}

streambuf* Filebuf::open(FILE* fileptr)
{
  if (Fp == 0)
    Fp = new File(fileptr);
  else
    Fp->open(fileptr);
  if (base != 0) Fp->setbuf(eptr-base+1, base);
  init_streambuf_ptrs();
  return this;
}

Filebuf::Filebuf() : streambuf(), Fp(0) {}

Filebuf::Filebuf(const char* filename, io_mode m, access_mode a)
     : streambuf()
{
  Fp = new File(filename, m, a);
  init_streambuf_ptrs();
}

Filebuf::Filebuf(const char* filename, const char* m)
     : streambuf()
{
  Fp = new File(filename, m);
  init_streambuf_ptrs();
}

Filebuf::Filebuf(int filedesc, io_mode m)
     : streambuf()
{
  Fp = new File(filedesc, m);
  init_streambuf_ptrs();
}

Filebuf::Filebuf(FILE* fileptr)
     : streambuf()
{
  Fp = new File(fileptr);
  init_streambuf_ptrs();
}

int Filebuf::close()
{
  int was = Fp->is_open();
  if (was) { overflow(); Fp->close(); }
  return was;
}


Filebuf::~Filebuf()
{
  if (Fp != 0)
  {
    close();
    delete Fp;
  }
}

/*
  The underflow and overflow methods sync the streambuf with the _iobuf
  ptrs on the way in and out of the read. I believe that this is
  done in a portable way.
*/

int Filebuf::underflow()
{
  int ch;
  if (Fp == 0) return EOF;
  if (gptr == 0) // stdio _iobuf ptrs not initialized until after 1st read
  {
    ch = Fp->fill();
    base = Fp->fp->_base;
    eptr = base - 1 + _bufsiz(Fp->fp);
  }
  else
  {
    Fp->fp->_ptr = gptr;
    Fp->fp->_cnt = eptr - gptr + 1;
    ch = Fp->fill();
  }
  gptr = base;
  *gptr = ch;
  if (ch == EOF)
    pptr = base;
  else
    pptr = base + Fp->fp->_cnt + 1;
  if (Fp->good())
    return ch;
  else
  {
    Fp->clear();
    return EOF;
  }
}

int Filebuf::overflow(int ch)
{
  if (Fp == 0) return EOF;
  if (Fp->fp->_flag & _IONBF)  // handle unbuffered IO specially
  {
    if (pptr == 0) // 1st write
    {
      if (ch == EOF)
        return 0;
      else
      {
        Fp->flush(ch);
      }
    }
    else
    {
      if (ch == EOF)
        Fp->flush();		// Probably not necessary
      else
        Fp->flush(ch);
    }
  }
  else
  {
    if (pptr == 0) // 1st write
    {
      if (ch == EOF)
        return 0;
      else
      {
        Fp->flush(ch);
        base = Fp->fp->_base;
        eptr = base - 1 + _bufsiz(Fp->fp);
        gptr = base;
      }
    }
    else
    {
      if (ch != EOF) *pptr++ = ch;
      Fp->fp->_ptr = pptr;
      Fp->fp->_cnt = eptr - pptr + 1;
      Fp->flush();
    }
    pptr = Fp->fp->_ptr;
  }
  if (Fp->fail() || Fp->bad())
  {
    Fp->clear(); // this allows recovery from ostream level
    return EOF;
  }
  else
    return 0;
}

const char* Filebuf::name()
{
  return Fp->name();
}

streambuf* Filebuf::setbuf(char* buf, int buflen, int preload)
{
  if (preload != 0) return 0; // cannot preload, sorry
  if (Fp != 0) Fp = new File;
  Fp->setbuf(buflen, buf);
  init_streambuf_ptrs();
  return (Fp->good())? this : 0;
}

void Filebuf::error()
{
  Fp->error();
}
