// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include "driver.h"

#ifdef	WIN32
#define	strdup	ost::strdup
#endif

namespace sipdriver {
using namespace ost;
using namespace std;

static Keydata::Define driver[] = {
	{"type", "proto"},
	{"stack", "0"},
	{"events", "128"},
	{"priority", "0"},
	{"encoding", "mulaw"},
	{"framing", "20"},
	{"payload", "0"},
	{"dtmf", "101"},
	{"inband", "false"},
	{"silence", "500"},
	{"duration", "3600"},
	{"pickup", "2000"},
	{"hangup", "250"},
	{"accept", "320"},
	{"audio", "60"},
	{"jitter", "2"},
	{"filler", "false"},
	{"urlprefix", "sip:"},
#ifdef	WIN32		// use localhost default if not in conf for w32
	{"interface", "127.0.0.1:5070"},
#endif
        {NULL, NULL}};

#ifdef	WIN32
#define	KEYS	"/bayonne/sip"
#else
#define	KEYS	"/bayonne/driver/sip"
#endif

Driver Driver::sip;

Driver::Driver() :
BayonneDriver(driver, KEYS, "sip", false),
Thread(atoi(getLast("priority")), atoi(getLast("stack")) * 1024)
{
	const char *cp, *p;
	char buf[1024];
	AudioCodec *codec;

#ifdef	WIN32
	const char *env = Process::getEnv("DRIVERCONFIG");

	if(env)
		loadFile(env, "sip");
#else
	if(Bayonne::provision_system)
		load("/bayonne/provision/sip");
	else if(Bayonne::provision_user)
		load("~bayonne/sip");
#endif
	cp = getLast("ack");
	if(!cp)
		cp = getLast("acktimer");

	if(!cp)
		cp = getLast("pickup");

	if(cp)
		pickup_timer = atol(cp);

	if(pickup_timer < 10)
		pickup_timer *= 1000;

	cp = getLast("hangup");
	if(cp)
		hangup_timer = atol(cp);

	cp = getLast("bind");
	if(!cp)
		cp = getLast("interface");

	if(!cp)
	{
		gethostname(buf, sizeof(buf) - 1);
		InetAddress host(buf);
		snprintf(buf, sizeof(buf), "%s:5060", inet_ntoa(host.getAddress()));
		setValue("interface", buf);
		cp = getLast("interface");
	}
	else
		setValue("interface", cp);

	p = strchr(cp, ':');
	if(p)
		sip_port = atoi(++p);
	else
		sip_port = atoi(cp);		

	rtp_port = sip_port + 2;

	cp = getLast("rtp");
	if(cp)
		rtp_port = atoi(cp);

	silence = atoi(getLast("silence"));
	cp = getLast("inband");

	dtmf_inband = true;
	switch(*cp)
	{
	case 'n':
	case 'N':
	case 'f':
	case 'F':
	case '0':
	case 0:
		dtmf_inband = false;
		break;
	}

        cp = getLast("filler");

        data_filler = true;
        switch(*cp)
        {
        case 'n':
        case 'N':
        case 'f':
        case 'F':
        case '0':
        case 0:
                data_filler = false;
                break;
        }


	audio_timer = atol(getLast("audio"));
	hangup_timer -= audio_timer;
	accept_timer = atol(getLast("accept"));
	jitter = atoi(getLast("jitter"));
	data_payload = atoi(getLast("payload"));
	dtmf_payload = 0;

	cp = getLast("dtmf");
	if(cp)
		dtmf_payload = atoi(cp);

	memset(&info, 0, sizeof(info));
	cp = getLast("encoding");
	info.encoding = getEncoding(cp);
	info.rate = rate8khz;
	cp = getLast("rate");
	if(cp)
		info.rate = atol(cp);
	if(peer_encoding != Audio::unknownEncoding)
	{
		if(peer_encoding != Audio::pcm16Mono)	
			info.encoding = peer_encoding;
		info.setFraming(peer_framing);
	}
	else
		info.setFraming(atol(getLast("framing")));
	registry = false;
	silent_frame = new Sample[info.framecount];
	memset(silent_frame, 0, info.framecount * 2);
	silent_encoded = new unsigned char[info.framesize];
	if(info.encoding == Audio::pcm16Mono)
		memset(silent_encoded, 0, info.framesize);
	else
	{
		codec = AudioCodec::getCodec(info);
		codec->encode(silent_frame, silent_encoded, info.framecount);
		AudioCodec::endCodec(codec);
	}	
}

void Driver::startDriver(void)
{
	char cbuf[65];
	unsigned tries = 0;
	char *p;
	const char *cp;
	unsigned max_count = ts_count;
	cp = server->getLast("slots.sip");

	exiting = false;
	timeslot = ts_used;
	msgport = new BayonneMsgport(this);
	
	if(cp)
		max_count = atoi(cp);

	while(ts_used < ts_count && max_count--)
		new Session(ts_used);

	count = ts_used - timeslot;
	if(!count)
		return;


	while(eXosip_init(NULL, NULL, sip_port))
	{
		slog.error("sip: failed to bind %d", sip_port);
		if(tries++ > 3)
			return;
		sip_port += 10;
		if(!getLast("rtp"))
			rtp_port += 10;
	}

	setString(cbuf, sizeof(cbuf) - 8, getLast("interface"));
	p = strrchr(cbuf, ':');
	snprintf(p + 1, 8, "%d", sip_port);
	setValue("interface", cbuf);
	
        osip_trace_initialize_syslog(TRACE_LEVEL0, "bayonne");
        osip_trace_enable_level(TRACE_LEVEL1);
        osip_trace_enable_level(TRACE_LEVEL2);
        osip_trace_enable_level(TRACE_LEVEL3);
        osip_trace_enable_level(TRACE_LEVEL4);
        osip_trace_enable_level(TRACE_LEVEL5);
        osip_trace_enable_level(TRACE_LEVEL6);
        osip_trace_enable_level(TRACE_LEVEL7);

	eXosip_sdp_negotiation_remove_audio_payloads();

	switch(info.encoding)
	{
	case mulawAudio:
		if(!data_payload)
			setValue("payload", "0");
		snprintf(cbuf, sizeof(cbuf), "%d PCMU/%d", 
			data_payload, info.rate);
		break;
	case alawAudio:
		if(!data_payload)
		{
			data_payload = 8;
			setValue("payload", "8");
		}
		snprintf(cbuf, sizeof(cbuf), "%d PCMA/%d",
			data_payload, info.rate);
		break;
	case pcm16Mono:
		snprintf(cbuf, sizeof(cbuf), "%d L16_1CH/%d",
			data_payload, info.rate);
		break;
	case pcm16Stereo:
		snprintf(cbuf, sizeof(cbuf), "%d L16_2CH/%d",
			data_payload, info.rate);
		break;
	case gsmVoice:
		if(!data_payload)
		{
			data_payload = 3;
			setValue("payload", "3");
		}
		snprintf(cbuf, sizeof(cbuf), "%d GSM/%d",
			data_payload, info.rate);
		break;
	default:
		slog.error("sip: unsupported sdp audio encoding");
		return;
	}

	eXosip_sdp_negotiation_add_codec(
		strdup(getLast("payload")), NULL, strdup("RTP/AVP"), 
		NULL, NULL, NULL, NULL, NULL, strdup(cbuf));
	slog.debug("sip: adding sdp encoding %s", cbuf);
	info.annotation = strdup(cbuf);

	if(dtmf_payload)
	{
		snprintf(cbuf, sizeof(cbuf), "%d telephony-event/%d", 
			dtmf_payload, info.rate);
		slog.debug("sip: adding sdp encoding %s", cbuf);
		eXosip_sdp_negotiation_add_codec(
			strdup(getLast("dtmf")), NULL, strdup("RTP/AVP"), 
			NULL, NULL, NULL, NULL, NULL, strdup(cbuf));
	}

	eXosip_set_user_agent("GNU Bayonne");

	cp = getLast("local");
	if(cp)
	{
		InetAddress la = cp;
		eXosip_force_localip(la.getHostname());
	}
		
	cp = getLast("firewall");
	if(cp)	
	{
		InetAddress fa = cp;
		eXosip_set_firewallip(fa.getHostname());
	}

	slog.debug("sip: bound to port %d", sip_port);
	msgport->start();
	Thread::start();
	Thread::sleep(100);
	BayonneDriver::startDriver();	
}

void Driver::stopDriver(void)
{
	if(running)
	{
		exiting = true;
		__eXosip_wakeup_event();
		terminate();
		eXosip_quit();
		BayonneDriver::stopDriver();
	}
}

const char *Driver::registerScript(ScriptImage *img, Line *line)
{
	Name *scr = img->getCurrent();
	Registry *reg = (Registry *)img->getMemory(sizeof(Registry));
        const char *cp = ScriptChecks::findKeyword(line, "timeout");
	char buf[1024];
	unsigned len, pos;
	char *p;
	const char *uri;

	if(!ScriptChecks::useKeywords(line, "=uri=timeout=server=proxy=userid=secret"))
		return "invalid keywords for sip, use uri, timeout, proxy, userid, secret, port";

	reg->active = false;
	reg->protocol = "sip";
	reg->iface = getLast("interface");
	reg->uri = ScriptChecks::findKeyword(line, "uri");
	reg->hostid = ScriptChecks::findKeyword(line, "proxy");
	reg->userid = ScriptChecks::findKeyword(line, "userid");
	reg->secret = ScriptChecks::findKeyword(line, "secret");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = ScriptChecks::findKeyword(line, "server");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = getLast("proxy");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = server->getLast("proxy");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = server->getLast("server");

	if(!reg->hostid || !*reg->hostid)
		reg->hostid = "localhost";

	if(!reg->userid)
		reg->userid = getLast("userid");

	if(!reg->userid)
		reg->userid = server->getLast("userid");

	if(!*reg->userid)
		reg->userid = NULL;

	if(!reg->secret)
		reg->secret = getLast("secret");

	if(!reg->secret)
		reg->secret = server->getLast("secret");

	if(!*reg->secret)
		reg->secret = NULL;	

	if(!cp)
		cp = getLast("duration");

	reg->duration = 1000 * atol(cp);
	reg->line = line;
	reg->scr = scr;

	snprintf(buf, sizeof(buf) - 2, "sip:%s", scr->name);
	p = strchr(buf + 4, ':');
	if(p)
	{
		*(p++) = '.';
		cp = strrchr(scr->name, ':');
		strcpy(p, cp + 1);
	}

	len = strlen(buf);
	buf[len++] = '@';
	buf[len] = 0;
	pos = len;

	setString(buf + pos, sizeof(buf) - pos, reg->iface);
	len = strlen(buf) + 1;
	reg->contact = (const char *)img->getMemory(len);
	strcpy((char *)reg->contact, buf);

        setString(buf + pos, sizeof(buf) - pos, reg->hostid);
	p = strrchr(buf + 5, ':');
	if(p)
		*p = 0;
        len = strlen(buf) + 1;
        uri = (const char *)img->getMemory(len);
        strcpy((char *)uri, buf);

	snprintf(buf, sizeof(buf), "sip:%s", reg->hostid);

	len = strlen(buf) + 1;
	reg->proxy = (const char *)img->getMemory(len);
	strcpy((char *)reg->proxy, buf);

	if(!reg->uri)
		reg->uri = uri;

	slog.debug("register %s as %s to %s", reg->contact, reg->uri, reg->proxy);

	if(!registry)
		registry = true;

	if(reg->userid)
	{
		setString(buf, sizeof(buf), reg->userid);
		p = strchr(buf, '@');
		if(p)
			*(p++) = 0;
		
		if(eXosip_add_authentication_info(buf, buf, reg->secret, NULL, p))
			slog.error("sip cannot authenticate with %s", reg->userid);			
	}

	eXosip_lock();

        reg->regid = eXosip_register_init((char *)reg->uri, (char *)reg->proxy, (char *)reg->contact);

	if(reg->regid < 0)
	{
		eXosip_unlock();
		return "cannot create sip registration";
	}

	if(eXosip_register(reg->regid, reg->duration / 1000) < 0)
	{
		eXosip_unlock();
		return "cannot register with sip proxy";
	}
	eXosip_unlock();

	reg->setTimer(reg->duration / 2 - 1000);
	snprintf(buf, sizeof(buf), "sip.%d", reg->regid);
	img->setPointer(buf, reg);
	snprintf(buf, sizeof(buf), "uri.%s", reg->contact);
	img->setPointer(buf, reg);
	img->addRegistration(line);
	line->scr.registry = reg;

	return "";
}

Session *Driver::getCall(int callid)
{
	timeslot_t ts = 0;
	Session *session;
	BayonneSession *bs;

	while(ts < count)
	{
		bs = getTimeslot(ts++);
		if(!bs)
			continue;
		session = (Session *)bs;
		if(session->cid == callid && session->offhook)
			return session;
	}
	return NULL;
}

void Driver::refresh(void)
{
	ScriptRegistry *sreg;
	ScriptImage *img;
	Registry *reg;
	Line *line;

	img = useImage();

	if(!img)
		return;

	line = img->getRegistration();

	while(line)
	{
		sreg = line->scr.registry;
		line = line->next;

		if(sreg->getTimer() > 0)
			continue;

		if(stricmp(sreg->protocol, "sip"))
			continue;

		reg = static_cast<Registry *>(sreg);

		slog.debug("refreshing %s", reg->contact);

		eXosip_lock();
		eXosip_register(reg->regid, reg->duration / 1000);
		eXosip_unlock();
		reg->setTimer(reg->duration / 2 - 1000);
	}

	endImage(img);
}

void Driver::run(void)
{
        eXosip_event_t *sevent;
        Event event;
	time_t last, now;
	char buf[1024];
	char cbuf[256];
	Registry *reg;
	ScriptImage *img;
	const char *caller = NULL;
	const char *dialed = NULL;
	const char *display = NULL;
	BayonneSession *s;
	Session *session;
	char *k, *v, *l, *p, *line;
	osip_uri_t *uri;
	osip_from_t *from;
	char digit[10];
	char duration[10];
	osip_body_t *mbody;

	Bayonne::waitLoaded();
	time(&last);

	for(;;)
        {
		time(&now);
		sevent = NULL;
		now -= last;
		if(now < 10)
			sevent = eXosip_event_wait(10 - now, 0);

		Thread::yield();

		if(exiting)
			Thread::sync();

		if(!sevent)
		{
			time(&last);
			refresh();
			continue;
		}

		slog.debug("sip: event %04x; cid=%d, did=%d, rid=%d",
			sevent->type, sevent->cid, sevent->did, sevent->rid);

		switch(sevent->type)
		{
		case EXOSIP_REGISTRATION_NEW:
			slog.info("sip: new registration");
			break;
		case EXOSIP_REGISTRATION_SUCCESS:
			snprintf(buf, sizeof(buf), "sip.%d", sevent->rid);
			server->enter();
			img = server->getActive();
			reg = (Registry *)img->getPointer(buf);	
			if(reg)
			{
				reg->active = true;
				slog.debug("registration for %s confirmed", reg->contact);
			}
			else
				slog.warn("unknown sip registration confirmed; rid=%s", sevent->rid);
			server->leave();
			break;
		case EXOSIP_REGISTRATION_FAILURE:
			snprintf(buf, sizeof(buf), "sip.%d", sevent->rid);
			server->enter();
			img = server->getActive();
			reg = (Registry *)img->getPointer(buf);	
			if(reg)
			{
				reg->setTimer(1);
				reg->active = false;
				slog.info("registration for %s failed", reg->contact);
			}
			else
				slog.warn("unknown sip registration failed; rid=%s", sevent->rid);
			server->leave();
			break;
		case EXOSIP_REGISTRATION_REFRESHED:
			slog.info("sip registration refreshed");
			break;
		case EXOSIP_REGISTRATION_TERMINATED:
			slog.info("sip registration terminated");
			break;			
		case EXOSIP_CALL_NEW:
			dialed = sevent->req_uri;

			snprintf(cbuf, sizeof(cbuf), "%s", sevent->remote_uri);
			caller = cbuf;

			if(*caller == '<')
			{
				++caller;
				p = strchr(caller, '>');
				if(p)
					*p = 0;
			}

			slog.debug("sip: invited %s from %s", dialed, caller); 
			snprintf(buf, sizeof(buf), "uri.%s", sevent->req_uri);
			img = useImage();
			reg = (Registry *)img->getPointer(buf);
			if(!reg)
			{
				slog.warn("sip: unknown uri %s", sevent->req_uri);
sip404:
				endImage(img);
		        	eXosip_lock();
                    		eXosip_answer_call(sevent->did, 404, NULL);
                    		eXosip_unlock();
				break;
			}

			s = getIdle();

			if(!s)
			{
				slog.warn("sip: no timeslots available");
//sip480:				              
				endImage(img);                  
				eXosip_lock();
                                eXosip_answer_call(sevent->did, 480, NULL);
                                eXosip_unlock();
				break;
			}
			
			osip_uri_init(&uri);
			osip_from_init(&from);
			session = (Session *)(s);

			session->cid = sevent->cid;
			session->did = sevent->did;

                        if(osip_uri_parse(uri, sevent->req_uri) != 0)
                                dialed = "unknown";
                        else if(!uri->username)
                                dialed = "unknown";
                        else
                                dialed = uri->username;

			if(osip_from_parse(from, sevent->remote_uri) != 0)
				caller = NULL;
			else if(from->url->username == NULL)
				caller = NULL;
			else
				caller = from->url->username;

			if(!caller)

			if(!dialed || !*dialed)
				dialed = "unknown";

			if(!caller || !*caller)
				caller = "unknown";

			display = osip_from_get_displayname(from);
			if(!display || !*display)
				display = caller;

                        session->setConst("session.dialed", dialed);
                        session->setConst("session.caller", caller);
			session->setConst("session.display", display);

			osip_uri_free(uri);
			osip_from_free(from);

			eXosip_get_localip(buf);
			session->local_address = buf;
			session->remote_address = sevent->remote_sdp_audio_ip;
			session->remote_port = sevent->remote_sdp_audio_port;		

			snprintf(buf, sizeof(buf), "%s:%d", 
				inet_ntoa(session->local_address.getAddress()),
				session->getLocalPort());
			session->setConst("session.rtp_local", buf);		

			snprintf(buf, sizeof(buf), "%s:%d",
				inet_ntoa(session->remote_address.getAddress()),
				session->getRemotePort());
			session->setConst("session.rtp_remote", buf);
	
			session->setConst("session.uri_request", sevent->req_uri);
                        setString(buf, sizeof(buf), sevent->local_uri);
                        caller = buf;
                        if(*caller == '<')
                        {
                                p = strchr(++caller, '>');
                                if(p)
                                        *p = 0;
                        }
                        session->setConst("session.uri_local", caller);

			setString(buf, sizeof(buf), sevent->remote_uri);
			caller = strchr(buf, '<');
			if(!caller)
				caller = buf;			
			if(*caller == '<')
			{
				p = strchr(++caller, '>');
				if(p)
					*p = 0;
			}
			session->setConst("session.uri_remote", caller);

                        setString(buf, sizeof(buf), sevent->remote_contact);
                        caller = buf;
                        if(*caller == '<')
                        {
                                p = strchr(++caller, '>');
                                if(p)
                                        *p = 0;
                        }
                        session->setConst("session.uri_contact", caller);
			p = strchr(caller, '@');
			if(p)
				session->setConst("session.server", ++p);

			if(sevent->refer_to[0])
				session->setConst("session.refer", sevent->refer_to);

			if(sevent->subject[0])
				session->setConst("session.subject", sevent->subject);

			slog.debug("sip: starting %s on timeslot %d; connecting %s to %s", 
				reg->scr->name, session->getSlot(),
				session->getSymbol("session.rtp_local"),
				session->getSymbol("session.rtp_remote"));

			memset(&event, 0, sizeof(event));
			event.id = START_INCOMING;
			event.start.img = img;
			event.start.scr = reg->scr;

			if(!s->postEvent(&event))
			{
				// add to restore to idle, 404 to exit
				add(session);
				goto sip404;
			}

			eXosip_lock();
			eXosip_answer_call(sevent->did, 180, NULL);
			eXosip_unlock();
			break;
		case EXOSIP_CALL_ACK:
			session = getCall(sevent->cid);
			if(!session)
			{
				slog.warn("sip: call ack for non-existant call");
killit:
				eXosip_lock();
				eXosip_terminate_call(sevent->cid, sevent->did);
				eXosip_unlock();
				break;
			}
			memset(&event, 0, sizeof(event));
			event.id = CALL_ACCEPTED;
			session->postEvent(&event);
			break;
		case EXOSIP_CALL_RELEASED:
                        session = getCall(sevent->cid);
                        if(session)
				goto exitcall;
			break;
		case EXOSIP_CALL_CLOSED:
			session = getCall(sevent->cid);
			if(!session)
			{
				slog.warn("sip: got bye for non-existant call");
				break;
			}
exitcall:
			memset(&event, 0, sizeof(event));
			event.id = STOP_DISCONNECT;	
			session->did = 0;
			session->postEvent(&event);
			break;
		case EXOSIP_INFO_NEW:
                        session = getCall(sevent->cid);
                        if(!session)
                        {
                                slog.warn("sip: got info for non-existant call");
                                goto killit;
			}
			                
			if(!stricmp(sevent->i_ctt->type, "application") &&
                           !stricmp(sevent->i_ctt->subtype, "dtmf-relay") &&
                           !osip_list_eol(sevent->i_bodies, 0)) 
			{
				memset(&digit, 0, 10);
				memset(&duration, 0, 10);
				mbody = (osip_body_t *)osip_list_get(sevent->i_bodies, 0);
				
				if(!mbody)
					break;

				line = mbody->body;
				while(line && *line)
				{
					k = v = line;
					while(*v && *v != '=' && *v != '\r' && *v != '\n')
						++v;

					if(*v != '=')
						break;
					
					++v;
					while(*v == ' ' || *v == '\b')
						++v;
				
					if(!*v || *v == '\r' || *v == '\n')
						break;

					l = v;
					while(*l && *l != ' ' && *l != '\r' && *l != '\n')
						++l;

					if(!*l)
						break;

					if(!strnicmp(k, "signal", 6))
						strncpy(digit, v, l-v);
					else if(!strnicmp(k, "duration", 8))
						strncpy(duration, v, l-v);
					
					++l;
					while(*l == ' ' || *l == '\n' || *l == '\r')
						++l;
					
					line = l;
					if(!*line)
						break;
				}
				if(!digit[0] || !duration[0])
					break;

				memset(&event, 0, sizeof(event));
				event.id = DTMF_KEYUP;
				switch(*digit)
				{
				case '0':
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					event.dtmf.digit = *digit - '0';
					event.dtmf.duration = atoi(duration);
					session->postEvent(&event);
					break;
				case '*':
                                        event.dtmf.digit = 10;
                                        event.dtmf.duration = atoi(duration);
					session->postEvent(&event);
                                        break;
				case '#':
					event.dtmf.digit = 11;
					event.dtmf.duration = atoi(duration);
					session->postEvent(&event);
					break;
				}												
				
			}
			break;                        
		}			
		eXosip_event_free(sevent);
	}
}

} // namespace
