/* File: gems_mono_kit.c
 * Created: 4/7/92 by John Butare(jb7p@andrew.cmu.edu)
 * Description:
 *	
 * Modifications:
 */

/*
 * GemsMonoKit -- object for creating common UI components
 */

#include "mono_kit.h"
#include "generic_button.h"
#include "toggle_button.h"
#include "radio_button.h"
#include "diamond_beveller.h"
#include "label.h"
#include "text_editor.h"

#include <IV-look/choice.h>
#include <InterViews/bitmap.h>
#include <InterViews/center.h>
#include <InterViews/box.h>
#include <InterViews/font.h>
#include <InterViews/border.h>
#include <InterViews/glue.h>
#include <InterViews/margin.h>
#include <InterViews/style.h>
#include <InterViews/Bitmaps/checkmark.bm>
#include <OS/string.h>

#include <stream.h>

class GemsMonoKitImpl {
private:
    friend class GemsMonoKit;

    GemsMonoKitImpl();
    ~GemsMonoKitImpl();

    Coord frame_thickness(Style*);

    Glyph* GemsMonoKitImpl::body(Glyph* left, Glyph* right, Glyph* g);
    ChoiceItem* GemsMonoKitImpl::choice(Glyph* normal, Glyph* pressed, Glyph* disabled);
    ChoiceItem* GemsMonoKitImpl::choice(Glyph* normal, Glyph* pressed);
  
    Bitmap* checkmark_;
};

GemsMonoKit::GemsMonoKit() {
    _impl = new GemsMonoKitImpl;
}

GemsMonoKit::~GemsMonoKit() {
    delete _impl;
}

GemsMonoKitImpl::GemsMonoKitImpl() {
    checkmark_ = new Bitmap(
	checkmark_bits, checkmark_width, checkmark_height,
	checkmark_x_hot, checkmark_y_hot
    );
    checkmark_->ref();
}

GemsMonoKitImpl::~GemsMonoKitImpl() {
    Resource::unref(checkmark_);
}

Glyph* GemsMonoKit::inset_diamond(Glyph* g, Style* s) const
{
    return new Bevel(g, diamond, s->dark(), s-> dull(), s ->light(),
		     _impl->frame_thickness(s));
}

Glyph* GemsMonoKit::outset_diamond(Glyph* g, Style* s) const
{
    return new Bevel(g, diamond, s->light(), s-> flat(), s ->dark(),
		     _impl->frame_thickness(s));
}

Glyph* GemsMonoKit::flat_diamond(Glyph* g, Style* s) const
{
    return new Bevel(g, diamond, s->flat(), s-> flat(), s ->flat(),
		     _impl->frame_thickness(s));
}

GemsLabel* GemsMonoKit::label(char* label, Style* s) const
{
  return(new GemsLabel(label, s->font(), s->foreground()));
}

GemsTextEditor* GemsMonoKit::text_editor(
       Style* style, char* default_text, Action* ok, Action* cancel, Action* keypress
   ) const
{
  GemsLabel* label = new GemsLabel(default_text, style->font(), style->foreground());

  Glyph* body = 
       new Border(
       	   new LRBox(
       	       new Margin(
       	       	   label,
       	       	   2.0
       	       ),
       	       new HGlue(0.0, fil, 0.0)
       	   ),
       	   style->foreground()
       	 );

  return(new GemsTextEditor(label, body, ok, cancel, keypress));
}

GenericButton* GemsMonoKit::push_button(Glyph* g, Style* s, Action* select,
	Glyph* left, Glyph* right) const
{
  Kit* kit = Kit::instance();
  
  Glyph* normal = _impl->body(left, right,
	kit->flat_frame(new Border(new Margin(g, 4.0), s->foreground(), 1.0), s));

  Glyph* pressed = _impl->body(left, right,
	kit->inset_frame(new Border(new Margin(g, 4.0), s->foreground(), 1.0), s));

  return new GenericButton(_impl->choice(normal, pressed), select);
}

ToggleButton* GemsMonoKit::toggle_button(Style* s, Action* select, Action* deselect,
	Glyph* left, Glyph* right, boolean* value) const
{
  Kit* kit = Kit::instance();
  
  Glyph* on_pic = kit->label("X", s);
  Glyph* off_pic = kit->label(" ", s);
  
  Glyph* normal = _impl->body(left, right, 
       kit->flat_frame(new Border(new Margin(off_pic, 4.0), s->foreground(), 1.0), s));

  Glyph* pressed = _impl->body(left, right,
       kit->flat_frame(new Border(new Margin(on_pic, 4.0), s->foreground(), 1.0), s));

  Glyph* disabled = _impl->body(left, right,
    	kit->flat_frame(new Margin(off_pic, 4.0), s));

  ChoiceItem* choice = _impl->choice(normal, pressed, disabled);

  return(new ToggleButton(choice, select, deselect, value));
}

ToggleButton* GemsMonoKit::toggle_button(Glyph* g, Style* s, Action* select,
	Action* deselect, Glyph* left, Glyph* right, boolean* value) const
{
  Kit* kit = Kit::instance();
  
  Glyph* normal = _impl->body(left, right,
	kit->flat_frame(new Border(new Margin(g, 4.0), s->foreground(), 1.0), s));
	
  Glyph* pressed = _impl->body(left, right,
	kit->inset_frame(new Border(new Margin(g, 4.0), s->foreground(), 1.0), s));

  Glyph* disabled = _impl->body(left, right,
    	kit->flat_frame(new Margin(g, 4.0), s));
    
    ChoiceItem* choice = _impl->choice(normal, pressed, disabled);

    return(new ToggleButton(choice, select, deselect, value));
}

GemsRadioButton* GemsMonoKit::radio_button(Style* s, Action* select, Action* deselect,
	Glyph* left, Glyph* right, boolean* value, Coord size) const
{
    Coord margin = 3.0;
    Coord in_margin = margin + _impl->frame_thickness(s);

//    Coord size = (((s != nil) && (s->font() != nil))? (0.8 * s->font()->ascent()) : 20);

    Glyph* square = new VCenter(new LRBox(new HGlue(size, 0, 0),
					  new VGlue(size, 0, 0)),
				0.0);

    Glyph* normal = _impl->body(left, right,
	new Margin(outset_diamond(square, s), margin));
    
    Glyph* pressed = _impl->body(left, right,
	new Margin(inset_diamond(square, s), margin));

    Glyph* disabled = _impl->body(left, right,
	new Margin(flat_diamond(square, s), margin));

    ChoiceItem* choice = _impl->choice(normal, pressed, disabled);

    return new GemsRadioButton(choice, select, deselect, value);
}

GemsRadioButton* GemsMonoKit::radio_button(Glyph* g, Style* s, Action* select, Action* deselect,
	Glyph* left, Glyph* right, boolean* value) const
{
  Kit* kit = Kit::instance();

  Glyph* normal = _impl->body(left, right,
	kit->flat_frame(new Border(new Margin(g, 4.0), s->foreground(), 1.0), s));
	
  Glyph* pressed = _impl->body(left, right,
	kit->inset_frame(new Border(new Margin(g, 4.0), s->foreground(), 1.0), s));

  Glyph* disabled = _impl->body(left, right,
  	kit->flat_frame(new Margin(g, 4.0),s));
  
  ChoiceItem* choice = _impl->choice(normal, pressed, disabled);
  
  return new GemsRadioButton(choice, select, deselect, value);
}

/*
 * GemsMonoKitImpl
 */

Coord GemsMonoKitImpl::frame_thickness(Style* style)
{
  Coord c;
  if (!style->find_attribute("frameThickness", c)) {
    c = 2.0;
  }
  return c;
}

ChoiceItem* GemsMonoKitImpl::choice(Glyph* normal, Glyph* pressed)
{
  return(new ChoiceItem(normal, pressed));
}

ChoiceItem* GemsMonoKitImpl::choice(Glyph* normal, Glyph* pressed, Glyph* disabled)
{
  ChoiceItem* choice = new ChoiceItem();
  
  for (unsigned int i = 0; i < choice->numlooks(); i++) {
    if (!choice->contains_looks(i, ChoiceItem::is_enabled)) {
      choice->look(i, disabled);
    } else {
      if ((choice->contains_looks(i, ChoiceItem::is_pressed)) ==
	  (choice->contains_looks(i, ChoiceItem::is_chosen))) {
	choice->look(i, normal);
      } else {
	choice->look(i, pressed);
      }
    }
  }
  return(choice);
}

Glyph* GemsMonoKitImpl::body(Glyph* left, Glyph* right, Glyph* g)
{
  Glyph* stretch = new HGlue(5.0, fil, 0.0);
  Glyph* fixed = new HGlue(5.0, 0.0, 0.0);
  Glyph* body;

  if (left  && !right) {
    body = new LRBox(
	left,
	stretch,
	g
   );		     
  } else if (!left && right) {
    body = new LRBox(
	g,
	fixed,
	right		   
   );		     
  } else if (left && right) {
    body = new LRBox(
	left,
	stretch,
	g,
	fixed,
	right		    
   );		     
  } else
    body = g;
  
  return(body);
}
