static char rcsid[] = "consoldb.c,v 1.20 1996/01/04 03:46:42 duane Exp";
/*
 *  consoldb.c - Consolidate GDBM databases.  
 *
 *  Adds 1.gdbm ... N.gdbm to the master.gdbm database.  It only uses
 *  GDBM_INSERT's into the master.gdbm database.  So only the keys that
 *  are written to the master.gdbm database first are used.  This allows
 *  a priority scheme.  The --save-space flag tells consoldb to delete
 *  each 1.gdbm ... N.gdbm file after it's finished with it.
 *
 *  Usage:  consoldb [--save-space] master.gdbm 1.gdbm [... N.gdbm]
 *
 *  Darren Hardy, hardy@cs.colorado.edu, May 1994
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <gdbm.h>
#include "util.h"

/* Local functions */
static void usage();
static void add_file();
static void die();

/* Local variables */
static int do_destruct;
static char *masterfile = NULL;
static GDBM_FILE mdbf = NULL;


static void usage()
{
	fprintf(stderr,
	    "Usage: consoldb [--save-space] master.gdbm 1.gdbm [... N.gdbm]\n");
	exit(1);
}

int main(argc, argv)
     int argc;
     char *argv[];
{
	if (argc < 3)
		usage();

	init_log3("consoldb", NULL, stderr);

	do_destruct = 0;
	argv++;
	argc--;
	if (!strcmp(*argv, "--save-space")) {
		do_destruct = 1;
		argv++;
		argc--;
		if (argc < 2)
			usage();
	}
	masterfile = strdup(*argv);
	mdbf = gdbm_open(masterfile, 0, GDBM_WRCREAT | GDBM_FAST, 0644, NULL);
	if (mdbf == NULL) {
		errorlog("gdbm_open: %s: %s\n", masterfile, gdbm_strerror(gdbm_errno));
		log_errno(masterfile);
		exit(1);
	}
	for (argv++, argc--; argc > 0; argv++, argc--) {
		if (access(*argv, R_OK) == 0)
			add_file(*argv);
	}
	gdbm_close(mdbf);
	exit(0);
}

static void die()
{
	if (mdbf)
		gdbm_close(mdbf);
	exit(1);
}

/*
 *  add_file() - For each element of this GDBM database, add it to the master
 *  database if it doesn't already exist.
 */
static void add_file(filename)
     char *filename;
{
	GDBM_FILE dbf = NULL;
	datum k, nk, d;

	if (do_destruct)
		dbf = gdbm_open(filename, 0, GDBM_WRCREAT | GDBM_FAST, 0644, NULL);
	if (dbf == NULL)	/* try just to read it then */
		dbf = gdbm_open(filename, 0, GDBM_READER, 0644, NULL);
	if (dbf == NULL) {
		errorlog("gdbm_open: %s: %s\n", filename,
		    gdbm_strerror(gdbm_errno));
		return;
	}
	k = gdbm_firstkey(dbf);
	while (k.dptr) {
		nk = gdbm_nextkey(dbf, k);
		if (!gdbm_exists(mdbf, k)) {	/* add to master */
			d = gdbm_fetch(dbf, k);
			if (d.dptr == NULL) {
				errorlog("gdbm_fetch: %s: %s: %s\n", k.dptr,
				    filename, gdbm_strerror(gdbm_errno));
				continue;
			}
			if (gdbm_store(mdbf, k, d, GDBM_INSERT)) {
				errorlog("gdbm_store: %s: %s: %s\n", k.dptr,
				    masterfile, gdbm_strerror(gdbm_errno));
			}
			free(d.dptr);
		}
		if (do_destruct)
			(void) gdbm_delete(dbf, k);
		free(k.dptr);
		k = nk;
	}
	gdbm_close(dbf);

	if (do_destruct) {
		if (unlink(filename) < 0) {
			log_errno(filename);
		}
	}
}
