static char rcsid[] = "main.c,v 1.161 1996/01/17 10:07:40 duane Exp";
/*
 *  main.c -- main module of the Harvest Broker.
 *  William G. Camargo, Mic Bowman, Penn State Univ.
 *  Darren Hardy, Duane Wessels, U. Colorado, Boulder.
 *
 *  DEBUG: section  77, level 1		Broker main
 *
 *  ----------------------------------------------------------------------
 *  Copyright (c) 1994, 1995.  All rights reserved.
 *  
 *    The Harvest software was developed by the Internet Research Task
 *    Force Research Group on Resource Discovery (IRTF-RD):
 *  
 *          Mic Bowman of Transarc Corporation.
 *          Peter Danzig of the University of Southern California.
 *          Darren R. Hardy of the University of Colorado at Boulder.
 *          Udi Manber of the University of Arizona.
 *          Michael F. Schwartz of the University of Colorado at Boulder.
 *          Duane Wessels of the University of Colorado at Boulder.
 *  
 *    This copyright notice applies to software in the Harvest
 *    ``src/'' directory only.  Users should consult the individual
 *    copyright notices in the ``components/'' subdirectories for
 *    copyright information about other software bundled with the
 *    Harvest source code distribution.
 *  
 *  TERMS OF USE
 *    
 *    The Harvest software may be used and re-distributed without
 *    charge, provided that the software origin and research team are
 *    cited in any use of the system.  Most commonly this is
 *    accomplished by including a link to the Harvest Home Page
 *    (http://harvest.cs.colorado.edu/) from the query page of any
 *    Broker you deploy, as well as in the query result pages.  These
 *    links are generated automatically by the standard Broker
 *    software distribution.
 *    
 *    The Harvest software is provided ``as is'', without express or
 *    implied warranty, and with no support nor obligation to assist
 *    in its use, correction, modification or enhancement.  We assume
 *    no liability with respect to the infringement of copyrights,
 *    trade secrets, or any patents, and are not responsible for
 *    consequential damages.  Proper use of the Harvest software is
 *    entirely the responsibility of the user.
 *  
 *  DERIVATIVE WORKS
 *  
 *    Users may make derivative works from the Harvest software, subject 
 *    to the following constraints:
 *  
 *      - You must include the above copyright notice and these 
 *        accompanying paragraphs in all forms of derivative works, 
 *        and any documentation and other materials related to such 
 *        distribution and use acknowledge that the software was 
 *        developed at the above institutions.
 *  
 *      - You must notify IRTF-RD regarding your distribution of 
 *        the derivative work.
 *  
 *      - You must clearly notify users that your are distributing 
 *        a modified version and not the original Harvest software.
 *  
 *      - Any derivative product is also subject to these copyright 
 *        and use restrictions.
 *  
 *    Note that the Harvest software is NOT in the public domain.  We
 *    retain copyright, as specified above.
 *  
 *  HISTORY OF FREE SOFTWARE STATUS
 *  
 *    Originally we required sites to license the software in cases
 *    where they were going to build commercial products/services
 *    around Harvest.  In June 1995 we changed this policy.  We now
 *    allow people to use the core Harvest software (the code found in
 *    the Harvest ``src/'' directory) for free.  We made this change
 *    in the interest of encouraging the widest possible deployment of
 *    the technology.  The Harvest software is really a reference
 *    implementation of a set of protocols and formats, some of which
 *    we intend to standardize.  We encourage commercial
 *    re-implementations of code complying to this set of standards.  
 *  
 */
#include "broker.h"
#include "log.h"
#include <locale.h>

/* Global Variables - administrative */

time_t collect_rate;
time_t clean_rate;
time_t drefresh_rate;
char *HName = NULL;
char *DIRpath = NULL;
char *BrkHomePage = NULL;
char *Gather = NULL;
char *WebServer = NULL;
char *WebPath = NULL;
char *brk_obj_url = NULL;
char *ColConfig = NULL;
char *Tstr = NULL;
char *passwd = NULL;
char *IndexerType = NULL;
char *aproc = NULL;
int IndexType;
int qport;
int reg_limit;
int max_events;
int do_fast_start = 0;
char *obj_desc;
int obj_desc_s;
struct indexing_routines *INDEXER;
extern REGISTRY_HEADER *RegHdr;
extern int broker_offline;

/* Global Variables - system */
int ndenied_connections = 0;
int qsock = -1;
time_t Cur_Time;
extern int Num_Ev;
extern int Log_Terse;
time_t Next_Collection = 0;
time_t Next_Clean = 0;
int collect_flag = 1;		/* default to always do a collection */
int comflag = 0;
int LogFlag = 0;
int IndexServer_pid = 0;
int IndexServer_ForceRestart = 0;	/* used by #restart-index-server */
int ReadQueryTimeout = READ_QUERY_TIMEOUT;

/* Local functions */
static int Indexer_Init();
static int Initialize_Broker();
static void disconnect();

/* 
 *  ------------------------------------------------------------------
 *           Runtime support for switching Indexing subsystems
 *  ------------------------------------------------------------------
 *  Below is the definitions needed for the run-time indexing support.
 *  If you want to integrate a new indexing sub-systems, then you'll
 *  need to add your Indexer functions below.
 */

/*  The Indexer needs a header file containing the function defs */
#ifdef USE_GLIMPSE
#include "Glimpse/index.h"
#endif
#ifdef USE_WAIS
#include "Wais/index.h"
#endif
#ifdef USE_NETFIND
#include "Netfind/index.h"
#endif
#ifdef USE_GRASS
#include "Grass/index.h"
#endif
#ifdef USE_VERITY
#include "index.h"
#endif
#ifdef USE_PLWEB
#include "PLWeb/index.h"
#endif

/*  These are the valid Indexer types from broker.conf */
#define Valid_Indexer_Type(type)        \
        ( \
	!strcmp(type, "Glimpse") || \
	!strcmp(type, "Netfind") || \
	!strcmp(type, "Grass")   || \
	!strcmp(type, "Verity")   || \
	!strcmp(type, "PLWeb")   || \
	!strcmp(type, "WAIS") \
	)

/*  This is the structure that is used to map the Indexing routines */
struct indexing_routines Indexer_Routines[] =
{
#ifdef USE_GLIMPSE
	{"Glimpse",
	 Glimpse_IND_Index_Start,
	 Glimpse_IND_Index_Flush,
	 Glimpse_IND_New_Object,
	 Glimpse_IND_Destroy_Obj,
	 Glimpse_IND_Index_Full,
	 Glimpse_IND_Index_Incremental,
	 Glimpse_IND_initialize,
	 Glimpse_IND_config,
	 Glimpse_IND_do_query,
	 Glimpse_IND_Init_Flags,
	 Glimpse_IND_Set_Flags
	},
#endif
#ifdef USE_WAIS
	{"WAIS",
	 WAIS_IND_Index_Start,
	 WAIS_IND_Index_Flush,
	 WAIS_IND_New_Object,
	 WAIS_IND_Destroy_Obj,
	 WAIS_IND_Index_Full,
	 WAIS_IND_Index_Incremental,
	 WAIS_IND_initialize,
	 WAIS_IND_config,
	 WAIS_IND_do_query,
	 WAIS_IND_Init_Flags,
	 WAIS_IND_Set_Flags
	},
#endif
#ifdef USE_NETFIND
	{"Netfind",
	 NF_IND_Index_Start,
	 NF_IND_Index_Flush,
	 NF_IND_New_Object,
	 NF_IND_Destroy_Obj,
	 NF_IND_Index_Full,
	 NF_IND_Index_Incremental,
	 NF_IND_initialize,
	 NF_IND_config,
	 NF_IND_do_query,
	 NF_IND_Init_Flags,
	 NF_IND_Set_Flags
	},
#endif
#ifdef USE_GRASS
	{"Grass",
	 GRASS_IND_Index_Start,
	 GRASS_IND_Index_Flush,
	 GRASS_IND_New_Object,
	 GRASS_IND_Destroy_Obj,
	 GRASS_IND_Index_Full,
	 GRASS_IND_Index_Incremental,
	 GRASS_IND_initialize,
	 GRASS_IND_config,
	 GRASS_IND_do_query,
	 GRASS_IND_Init_Flags,
	 GRASS_IND_Set_Flags
	},
#endif
#ifdef USE_PLWEB
	{"PLWeb",
	 PLWeb_IND_Index_Start,
	 PLWeb_IND_Index_Flush,
	 PLWeb_IND_New_Object,
	 PLWeb_IND_Destroy_Obj,
	 PLWeb_IND_Index_Full,
	 PLWeb_IND_Index_Incremental,
	 PLWeb_IND_initialize,
	 PLWeb_IND_config,
	 PLWeb_IND_do_query,
	 PLWeb_IND_Init_Flags,
	 PLWeb_IND_Set_Flags
	},
#endif
#ifdef USE_VERITY
	{"Verity",
	 VERITY_IND_Index_Start,
	 VERITY_IND_Index_Flush,
	 VERITY_IND_New_Object,
	 VERITY_IND_Destroy_Obj,
	 VERITY_IND_Index_Full,
	 VERITY_IND_Index_Incremental,
	 VERITY_IND_initialize,
	 VERITY_IND_config,
	 VERITY_IND_do_query,
	 VERITY_IND_Init_Flags,
	 VERITY_IND_Set_Flags
	}
#endif
};
#define MAX_INDEXER_TYPES \
	(sizeof(Indexer_Routines)/sizeof(struct indexing_routines))

char *cffile;			/* global, so can be passed to verityindex */

int main(argc, argv)
int argc;
char *argv[];
{
	int status;

	(void) setlocale (LC_ALL, "");

	disconnect();		/* be a nice, robust daemon */

#ifdef HAVE_SETLINEBUF
	setlinebuf(stdout);
	setlinebuf(stderr);
#else
	setbuf(stdout, NULL);
	setbuf(stderr, NULL);
#endif
	debug_reset();
	debug_init();		/* read $HARVEST_DEBUG */
	init_log3("broker", stdout, stdout);
	Log("Initializing the Broker...\n");
	/* intialize global state */
	if (argc > 1 && argv[1] != NULL) {
		argc--; argv++;
		cffile = xstrdup(*argv);
	} else {
		cffile = xstrdup("admin/broker.conf");
	}
	/* set up signal handlers */
	(void) signal(SIGINT,  sigdie);
	(void) signal(SIGTERM, sigdie);
	(void) signal(SIGPIPE, SIG_IGN);
	(void) signal(SIGCHLD, sigreap);

        for (argc--, argv++; argc > 0 && **argv == '-'; argc--, argv++) {
		if (!strcmp(*argv, "-nocol")) {
			collect_flag = 0;
		} else if (!strcmp(*argv, "-fast")) {
			do_fast_start = 1;
		} else if (!strcmp(*argv, "-new")) {
			/* start fresh by deleting LASTUPDATE */
			(void) unlink("./admin/LASTUPDATE");
		} else if (!strncmp(*argv, "-D", 2)) {
#ifdef USE_FAST_DEBUG_LOGGING
			extern int do_log_sync;
			do_log_sync = 0; 	/* very fast logging */
			freopen("broker.out", "a", stdout);
			freopen("broker.out", "a", stderr);
#endif
			debug_flag(*argv);
		}
	}
	if (Initialize_Broker(cffile) == ERROR)
		fatal("Could not initialize Broker.\n");

	if (collect_flag && (Next_Collection > 0)) {
		if (EV_add_aevent(COLLECTION) != ERROR)
			collect_flag = -1;
	}
	/* main loop */
	if (listen(qsock, 5) < 0) {
		log_errno("listen");
		sigdie(98);
	}
	Log("Broker is now on-line (pid %d, port %d)...\n", getpid(), qport);
	while (1) {
		(void)UTIL_Get_Time();	/* sets Cur_Time */
		/* try to add maintence events */
		if ((collect_flag == 0) && (Cur_Time >= Next_Collection)) {
			if (EV_add_aevent(COLLECTION) != ERROR)
				collect_flag = -1;
		}
		if ((Next_Clean >= 0) && (Cur_Time >= Next_Clean)) {
			if (EV_add_aevent(CLEANING) != ERROR)
				Next_Clean = -1;
		}
		if ((RegHdr->nrecords_deleted > reg_limit) && (comflag == 0)) {
			if (EV_add_aevent(RCOMP) != ERROR)
				comflag = -1;
		}
		(void)EV_Do_Event();
		/* 
		 *  Only block for 15 seconds, since SunOS 4.1.x may
		 *  swap out any process that waits longer than 20 secs.
		 */
		while (select_loop(Num_Ev ? 0 : 15, 0, 1) == 1)
			; /* add as many events as possible */
		if (Num_Ev == 0) {	/* Timeout, no events */
			/* catch anything that sigreap may have missed */
			while (waitpid(-1, &status, WNOHANG) > 0);
		}
	}
	exit(0);	/* END OF PROGRAM */
	/*NOTREACHED*/
}

/*===============================================================*/
/* signal handlers */

int Broker_Shutdown()
{
	Log("Shuting down the broker...\n");

	/* Stop the important stuff */
	RG_Registry_Shutdown();
	(void)LOG_close_log();
	(void)close(qsock);

	/* Close all file descriptors */
	close_all_fds(3);

	/* kill the index server */
	if (IndexServer_pid > 0) {
		Log("Killing IndexServer (pid %d)...\n", IndexServer_pid);
		(void)kill(IndexServer_pid, SIGTERM);
		sleep(5);
		(void)kill(IndexServer_pid, SIGKILL);
	}

	AD_run_admin_process();

	Log("Denied %d connections during this session.\n",ndenied_connections);
	Log("************\n");
	Log("*** DONE ***\n");
	Log("************\n");
	fflush(stdout);
	fflush(stderr);

	exit(0);
	return SUCCESS;
}

void sigdie(sig)
int sig;
{
	Log("Received signal %d...\n", sig);
	(void)Broker_Shutdown();
	_exit(sig);
}

void sigreap(sig)
int sig;
{
	int status, cpid;
	if ((cpid = waitpid(-1, &status, WNOHANG)) > 0) {
		Debug(77,1,("Child process exited: pid %d\n", cpid));
	}
#ifdef _HARVEST_SYSV_
	(void) signal(sig, sigreap);	/* reset signal handler */
#endif
}


/*===============================================================*/

/* 
 * verify_path() - Verify that the absolute path is readable; if not, exit
 */
int verify_path(file)
char *file;
{
	if ((file[0] == '/') && (access(file, R_OK) < 0)) {
		Log("ERROR: %s is not readable.  Change your broker.conf file!\n", file);
		return ERROR;
	}
	return SUCCESS;
}

/* 
 * verify_exe() - Verify that the absolute path is executable; if not, exit
 */
int verify_exe(file)
char *file;
{
	if ((file[0] == '/') && (access(file, X_OK) < 0)) {
		Log("ERROR: %s is not executable.  Change your broker.conf file!\n", file);
		return ERROR;
	}
	return SUCCESS;
}

/*===============================================================*/

/* Get Host Name */
int Initialize_Name()
{
	char *p = getfullhostname();

	if (p == NULL) {
		Log("WARNING: Internal Error: getfullhostname() failed...\n");
		return ERROR;
	}
	HName = xstrdup(p);
	Debug(77,1,("Broker Host is %s.\n", HName));
	return SUCCESS;
}

/* Initialize communication */
int Initialize_Com()
{
	static struct sockaddr_in brkr;
	int one = 1;

	if ((qsock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		log_errno("socket");
		fatal("socket failed");
	}
    	if ((setsockopt(qsock, SOL_SOCKET, SO_REUSEADDR, (char*) &one,
	     sizeof(one))) < 0) {
		log_errno("setsockopt (SO_REUSEADDR = on)");
		fatal("setsockopt failed");
	}

	/* name socket using wildcards */
	brkr.sin_family = AF_INET;
	brkr.sin_addr.s_addr = htonl(INADDR_ANY);
	brkr.sin_port = htons(qport);
	if (bind(qsock, (struct sockaddr *) &brkr, sizeof brkr) < 0) {
		log_errno("bind");
		(void)close(qsock);
		fatal("Cannot not bind to port %d.  Exiting...\n", qport);
	}

	return SUCCESS;
}

/* 
 *  Change global variables 
 */
int set_str(tag, value)
char *tag;
char *value;
{
	unsigned int logf;
	int Set_Next_Collection();

	/* These are all paths that need to be verified */
	if (strcasecmp(tag, S_BRKDIR) == 0) {
		/* Assign Value */
		DIRpath = xstrdup(value);
		if (verify_path(DIRpath) == ERROR)
			return ERROR;
	} else if (strcasecmp(tag, S_BRKOFFLINE) == 0) {
		if (value != NULL && !strcasecmp(value, "yes"))
			broker_offline = 1;
	} else if (strcasecmp(tag, S_BRKHP) == 0) {
		/* Assign Value */
		BrkHomePage = xstrdup(value);
	} else if (strcasecmp(tag, S_COLCONF) == 0) {
		ColConfig = xstrdup(value);
		if (verify_path(ColConfig) == ERROR)
			return ERROR;
	} else if (strcasecmp(tag, S_GATHER) == 0) {
		Gather = xstrdup(value);
		if (verify_exe(Gather) == ERROR)
			return ERROR;
		/* The rest is not paths that need to be verified */
	} else if (strcasecmp(tag, S_WEBS) == 0) {
		WebServer = xstrdup(value);
	} else if (strcasecmp(tag, S_WEBPATH) == 0) {
		WebPath = xstrdup(value);
	} else if (strcasecmp(tag, S_PASSWD) == 0) {
		passwd = xstrdup(value);
	} else if (strcasecmp(tag, S_APROC) == 0) {
		aproc = xstrdup(value);
	} else if (strcasecmp(tag, S_DESC) == 0) {
		if (obj_desc != NULL)
			xfree(obj_desc);
		obj_desc = xstrdup(value);
		(void)COL_Normalize_Name(obj_desc);
		obj_desc_s = strlen(obj_desc);
	} else if (strcasecmp(tag, S_CLEANR) == 0) {
		if (sscanf(value, "%d", &clean_rate) < 1)
			errorlog("Could not convert clean rate in config. \n");
		if (clean_rate < 0) {
			errorlog("Invalid clean rate %d. Set to %d.\n", clean_rate, CLEAN_RATE);
			clean_rate = CLEAN_RATE;
		}
	} else if (strcasecmp(tag, S_COLLR) == 0) {
		if (sscanf(value, "%d", &collect_rate) < 1)
			errorlog("Could not convert collection rate in config. \n");
		if (collect_rate < 0) {
			errorlog("Invalid collection rate %d. Set to %d.\n", collect_rate, COLLECT_RATE);
			collect_rate = COLLECT_RATE;
		}
	} else if (strcasecmp(tag, S_RFR) == 0) {
		if (sscanf(value, "%d", &drefresh_rate) < 1)
			errorlog("Could not convert refresh rate in config. \n");
		if (drefresh_rate <= 0) {
			errorlog("Invalid refresh rate %d. Setting to %d.\n", drefresh_rate, REFRESH_RATE);
			drefresh_rate = REFRESH_RATE;
		}
	} else if (strcasecmp(tag, S_PORT) == 0) {
		if (sscanf(value, "%d", &qport) < 1)
			errorlog("Could not convert port in config. \n");
		if (qport < 1) {
			errorlog("Invalid port number %d. Set to %d.\n", qport, Q_PORT);
			qport = Q_PORT;
		}
	} else if (strcasecmp(tag, S_DEDLIM) == 0) {
		if (sscanf(value, "%d", &reg_limit) < 1)
			errorlog("Could not convert dead entry limit in config. \n");
		if (reg_limit < 0) {
			errorlog("Invalid dead entry limit %d. Set to %d.\n", reg_limit, MAX_DEAD);
			reg_limit = MAX_DEAD;
		}
	} else if (strcasecmp(tag, S_MAXEV) == 0) {
		if (sscanf(value, "%d", &max_events) < 1)
			errorlog("Could not convert event queue limit in config. \n");
		if (max_events < 1) {
			errorlog("Invalid event queue limit %d. Set to %d.\n", max_events, MAX_EVENTS);
			max_events = MAX_EVENTS;
		}
	} else if (strcasecmp(tag, S_LOGK) == 0) {
		if (sscanf(value, "%i", &logf) < 1)
			errorlog("Could not convert log key in config. \n");
		LOG_turn_on(logf);
		LogFlag = 1;
	} else if (strcasecmp(tag, S_FCOLL) == 0) {
		Tstr = xstrdup(value);
		Set_Next_Collection(Tstr);
	} else if (strcasecmp(tag, S_INDTP) == 0) {
		if (strcasecmp(value, FULLI) == 0)
			IndexType = I_FULL;
		else if (strcasecmp(value, INCRI) == 0)
			IndexType = I_INCR;
		else if (strcasecmp(value, PEROBJI) == 0)
			IndexType = I_PER_OBJ;
	} else if (strcasecmp(tag, S_INDEXER) == 0) {
		IndexerType = xstrdup(value);
		return (Indexer_Init());
	} else if (strcasecmp(tag, S_TERSELOG) == 0) {
		if (value != NULL && strcasecmp(value, "on") == 0)
			Log_Terse = 1;
		else
			Log_Terse = 0;
	} else if (strcasecmp(tag, S_FASTSTART) == 0) {
		if (value != NULL && strcasecmp(value, "on") == 0)
			do_fast_start = 1;
		else
			do_fast_start = 0;
	} else if (strcasecmp(tag, S_RDQTO) == 0) {
		if (value != NULL)
			ReadQueryTimeout = atoi(value);
		if (ReadQueryTimeout < 0)
			ReadQueryTimeout = READ_QUERY_TIMEOUT;
	} else {
		if (do_IND_config(value, tag) == ERROR)
			return ERROR;
	}

	return SUCCESS;
}

int Do_Collection()
{
	if (collect_flag == -1)
		Next_Collection = Next_Collection + collect_rate;
	if (collect_rate > 0)
		collect_flag = 0;
	return (COL_Do_Collection());
}

int Do_Compression()
{
	comflag = 0;
	return (RG_Compress());
}

int Do_Uniqify()
{
	comflag = 0;
	return (RG_Uniqify_by_URL());
}

int Do_Restart_Index_Server()
{
	IndexServer_ForceRestart = 1;
	return SUCCESS;
	/* The index server will be restarted after the next query */
}

int Do_Cleaning()
{
	RG_Cleaner();
	if (clean_rate > 0)
		Next_Clean = Cur_Time + clean_rate;
	return SUCCESS;
}

/* Configure the Broker according to a given config file */
int Config_Broker(cfile)
char *cfile;
{
	FILE *cfp;
	char dspace[BUFSIZ], tag[BUFSIZ], value[BUFSIZ];

	Debug(77,1,("Configuring Broker from the file: %s\n", cfile));

	/* assign defaults to non-critical globals */
	Gather = xstrdup("gather");
	obj_desc = xstrdup(OBJ_DESC);
	obj_desc_s = strlen(obj_desc);
	collect_rate = COLLECT_RATE;
	clean_rate = CLEAN_RATE;
	qport = Q_PORT;
	drefresh_rate = REFRESH_RATE;
	reg_limit = MAX_DEAD;
	max_events = MAX_EVENTS;
	WebPath = NULL;
	ColConfig = NULL;

	/* open config file */
	if ((cfp = fopen(cfile, "r")) == NULL) {
		Log("WARNING: Cannot read the Broker configuration file: %s\n", cfile);
		log_errno(cfile);
		return ERROR;
	}
	while (fgets(dspace, BUFSIZ, cfp) != NULL) {
		/* Parse config line */
		if (dspace[0] == '#') {
			continue;
		} else if (sscanf(dspace, "%s %s", tag, value) < 2) {
			continue;
		} else {
			if (set_str(tag, value) == ERROR) {
				fclose(cfp);
				return ERROR;
			}
		}
		memset(tag, 0, BUFSIZ);
		memset(value, 0, BUFSIZ);
		memset(dspace, 0, BUFSIZ);
	}

	fclose(cfp);

	/* if logging has not been set, turn on all logging. */
	if (LogFlag == 0)
		LOG_turn_on(~00);

	if (DIRpath == NULL) {
		Log("WARNING: Configuration file %s does not set the critical value: DIRpath.\n", cfile);
		return ERROR;
	} else if (WebServer == NULL) {
		Log("WARNING: Configuration file %s does not set the critical value: WebServer.\n", cfile);
		return ERROR;
	}
	return SUCCESS;
}


int Set_Next_Collection(mytime)
char *mytime;
{
	char *tt;
	int hr, min;
	struct tm *tmptr;
	time_t tmp, addt = 0;

	if ((tt = strchr(mytime, ':')) == NULL) {
		Log("WARNING: Bad Time Format for Next Collection.\n");
		Next_Collection = 0;
		return ERROR;
	}
	hr = atoi(mytime);
	min = atoi(tt + 1);

	tzset();
	tmp = UTIL_Get_Time();
	tmptr = localtime(&tmp);
	if (tmptr->tm_hour > hr)
		addt = 86400;	/* 24 hrs in sec. */
	else
		addt = 0;

	tmptr->tm_hour = hr;
	tmptr->tm_min = min;

#ifdef HAVE_MKTIME
	Next_Collection = mktime(tmptr) + addt;
#else
	Next_Collection = timelocal(tmptr) + addt;
#endif

	return SUCCESS;
}

static int Initialize_Broker(deffile)
char *deffile;
{

	Log("Using the configuration file: %s\n", deffile);
	harvest_add_broker_path();

	/* Randomize */
#if defined(HAVE_SRAND48)
	srand48(time(NULL));
#elif defined(HAVE_SRANDOM)
	int srandom();
	(void) srandom(time(NULL));
#else
	srand(time(NULL));
#endif

	LOG_reset_master();
	(void)UTIL_Get_Time();	/* sets Cur_Time */

	if ((Initialize_Name() == ERROR) ||	/* sets current hostname */
	    (Indexer_Init() == ERROR) ||	/* indexer: before config */
	    (Config_Broker(deffile) == ERROR) ||/* must be first */
	    (LOG_Init() == ERROR) ||		/* log file */
	    (SM_Init() == ERROR) ||		/* storage manager */
	    (RG_Init() == ERROR) ||		/* registry depends on SM */
	    (EV_Init() == ERROR) ||		/* event list */
	    (Initialize_Com() == ERROR))	/* do port init last */
		return ERROR;

	/* do some misc initialization below */

	/* WebServer is non-NULL at this point */
	brk_obj_url = (char *) xmalloc(strlen(WebServer) + 100);
	if (WebPath) {
		sprintf(brk_obj_url, "http://%s%s/objects", WebServer, WebPath);
	} else {
		sprintf(brk_obj_url, "http://%s/BrkDir/objects", WebServer);
	}

	if (ColConfig == NULL)
		ColConfig = UTIL_make_admin_filename("Collection.conf");

	if (collect_rate == 0)
		collect_flag = 2;

	if (clean_rate == 0)
		Next_Clean = -1;

	AD_run_admin_process();
	return SUCCESS;
}

/*
 *  Initialize_Indexer_Routines - inits the run-time selectable indexing
 *  routines so that a user may switch indexers.
 */
static int Indexer_Init()
{
	int i;

	if (MAX_INDEXER_TYPES <= 0) {
		Log("ERROR: No Indexers defined for this broker!\n");
		return ERROR;
	}

	if (IndexerType != NULL) {
		if (!Valid_Indexer_Type(IndexerType)) {
			Log("ERROR: Invalid IndexerType: %s\n", IndexerType);
			return ERROR;
		}
	} else {
		/* default to first indexer in list, not necessarily Glimpse */
		IndexerType = xstrdup(Indexer_Routines[0].indexer_type);
	}
	for (i = 0; i < MAX_INDEXER_TYPES; i++) {
		if (!strcmp(Indexer_Routines[i].indexer_type, IndexerType))
			break;
	}
	if (i >= MAX_INDEXER_TYPES) {
		Log("ERROR: Invalid IndexerType: %s\n", IndexerType);
		return ERROR;
	}
	INDEXER = &Indexer_Routines[i];

	do_IND_initialize();
	return SUCCESS;
}

/* 
 *  disconnect() - Disconnect the process from the controlling terminal.
 *  Adapted from Harvest's floodd daemon in the replicator.
 */
static void disconnect()
{
  	int pid, fd;

	/* ignore all job control signals */
#ifdef SIGTTOU 
	signal (SIGTTOU, SIG_IGN);
#endif
#ifdef SIGTTIN
	signal (SIGTTIN, SIG_IGN);
#endif
#ifdef SIGTSTP
	signal (SIGTSTP, SIG_IGN);
#endif

#ifndef NO_FORK_FOR_DAEMON
	if ((pid = fork()) < 0) {
		log_errno("fork");
		return;
	}
	if (pid) {		/* parent */
		exit(0);	/* quietly exit to let child do the work */
	}
				/* child continues */
	(void) setsid();	/* Reset the session ID */
#endif

	/* Logging beneath here will not work */
	/* Close all file descriptors */
	close_all_fds(0);

  	/* Redirect the stdin to /dev/null */
  	if ((fd = open("/dev/null", O_RDONLY, 0)) < 0) 
		exit(99);
  	(void) dup2(fd, 0);

  	/* Redirect the stdout and stderr to broker.out */
  	if ((fd = open("broker.out", O_WRONLY|O_CREAT|O_APPEND, 0640)) < 0)
		exit(99);

  	(void) dup2(fd, 1);
  	(void) dup2(fd, 2);
	/* stdout and stderr is now to broker.out for logging */
}
