#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <utility.h>
#include <dynamic_string.h>
#ifdef READLINE
#include <readline/readline.h>
#include <readline/history.h>
#else
char *readline();
#endif

#include <command.h>

#define whitespace(c) ((c) == ' ' || (c) == '\t' || (c) == '\r' || (c) == '\n')

/* The names of functions that actually do the manipulation. */
void com_help ();

extern void *xmalloc ();
extern void *xrealloc ();
extern void  xfree ();
typedef void *Function ();

#ifdef READLINE 
extern int completion_case_fold; /* defined in readline */

char *cmd_default_line = NULL;	/* A default line for readline */
#else
int completion_case_fold = 0;
#endif

void stripwhite ();

/* A structure which contains information on the commands this program
   can understand. */

typedef struct {
  char *name;			/* User printable name of the function. */
  void (*func)();		/* Function to call to do the job. */
  char *doc;			/* Documentation for this function.  */
  void (*generator)();		/* Completion for command arguments */
} COMMAND;

typedef struct {
  char *name;
  char *summary;
  char *doc;
} TOPIC;

static COMMAND *commands = NULL ;
static commands_count = 0;
static commands_max = 5;

static TOPIC *topics = NULL;
static topics_count = 0;
static topics_max = 5;

void
cmd_initialize ()
{
  char *cmd_help_generator ();

  commands = xmalloc ((commands_max + 1) * sizeof (COMMAND));
  commands[0].name = NULL;
  commands[0].func = NULL;
  commands[0].doc= NULL;

  topics = xmalloc ((topics_max + 1) * sizeof (TOPIC));
  topics[0].name = NULL;
  topics[0].summary = NULL;
  topics[0].doc= NULL;

  /* Add in a default command so we can always know what commands
   * are available
   */
  cmd_add_command ("help", com_help, "Display this text.", NULL);
  cmd_add_command ("?", com_help, "Alias for help", NULL);
}

/* When non-zero, this global means the user is done using this program. */
int cmd_done = 0;

void
cmd_add_command (name, func, doc, generator)
     char *name;
     void (*func)();
     char *doc;
     void (*generator)();
{
  if (commands == NULL)
    cmd_initialize ();

  if (commands_count >= commands_max)
    {
      commands_max *= 2;
      commands = xrealloc (commands, (commands_max + 1) * sizeof (COMMAND));
    }
  
  /* Add the new command */
  commands[commands_count].name = (char *)strdup(name);
  commands[commands_count].func = func;
  commands[commands_count].doc= (char *)strdup(doc);
  commands[commands_count].generator = generator;
  commands_count++;
  
  /* Make sure command list ends with NULL */
  commands[commands_count].name = NULL;
  commands[commands_count].func = NULL;
  commands[commands_count].doc= NULL;
  commands[commands_count].generator = NULL;
}

void
cmd_add_topic (name, summary, doc)
     char *name;
     char *summary;
     char *doc;
{

  if (topics == NULL)
    cmd_initialize ();

  if (topics_count >= topics_max)
    {
      topics_max *= 2;
      topics = xrealloc (topics, (topics_max + 1) * sizeof (TOPIC));
    }
  
  /* Add the new topic */
  topics[topics_count].name = (char *)strdup(name);
  topics[topics_count].summary = (char *)strdup(summary);
  topics[topics_count].doc= (char *)strdup(doc);
  
  topics_count++;
  
  /* Make sure topic list ends with NULL */
  topics[topics_count].name = NULL;
  topics[topics_count].summary = NULL;
  topics[topics_count].doc= NULL;

}

/* Execute a command line. */
void
cmd_execute_line (line, fd, data)
     char *line;
     int fd;
     void *data;
{
  COMMAND *cmd_find (), *command;
  char **words;
  
  words = cmd_args_from (line);
  if (words == NULL || words[0] == NULL)
    return;

  command = cmd_find(words[0]);

  if (!command)
    {
      fd_error ("Unknown command `%s'\n", words[0]);
      cmd_args_free (words);
      return;
    }

  /* Call the function. */
  (*(command->func)) (words, fd, data);
  cmd_args_free (words);
}

#ifdef READLINE
/* Attempt to complete on the contents of TEXT.  START and END show the
   region of TEXT that contains the word to complete.  We can use the
   entire line in case we want to do some simple parsing.  Return the
   array of matches, or NULL if there aren't any. */
char **
cmd_completion (text, start, end)
     char *text;
     int start, end;
{
  char **matches;
  char *cmd_generator ();

  matches = (char **)NULL;

  /* If this word is at the start of the line, then it is a command
     to complete.  Otherwise it is an argument to a command. */

  if (start == 0)
    matches = completion_matches (text, cmd_generator);
  else
    {
      COMMAND  *command;
      char **args;

      args = cmd_args_from (rl_line_buffer);
      command = cmd_find (args[0]);
      cmd_args_free (args);
      if (command && command->generator)
	matches = completion_matches (text, command->generator);
      else
	matches = NULL;
    }
  return (matches);
}
#endif

void
cmd_loop (prompt, completer, file)
     char *prompt;
     Function *completer;
     FILE *file;
{
#ifdef READLINE
  void cmd_initialize_readline ();
#endif
  if (commands == NULL)		/* we have to initialize things */
    cmd_initialize ();

#ifdef READLINE
  cmd_initialize_readline (prompt, cmd_completion); /* Bind our completer. */
/*  cmd_initialize_readline (prompt, completer); /* Bind our completer. */
#endif
  /* Loop reading and executing lines until the user quits. */ 
 while (!cmd_done)
    {
      char *line;

      line = readline (prompt, file);

      if (!line)
	{
	  cmd_done = 1;		/* Encountered EOF at top level. */
	}
      else
	{
	  /* Remove leading and trailing whitespace from the line.
	     Then, if there is anything left, add it to the history list
	     and execute it. */
	  stripwhite (line);
#ifdef READLINE
	  if (*line)
	    {
	      add_history (line);
	      cmd_execute_line (line);
	    }
#endif
	}

      if (line)
	xfree (line);
    }
}


/* Look up NAME as the name of a command, and return a pointer to that
   command.  Return a NULL pointer if NAME isn't a command name. */
COMMAND *
cmd_find (name)
     char *name;
{
  register int i;

  if (commands == NULL)
    cmd_initialize ();
  for (i = 0; commands[i].name; i++)
    if ((strcmp (name, commands[i].name) == 0) ||
	(completion_case_fold && (strcasecmp (name, commands[i].name) == 0)))
      return (&commands[i]);

  return ((COMMAND *)NULL);
}

/* Break a line into a list of strings */
char **
cmd_args_from (string)
     char *string;
{
  char **result = (char **)NULL;
  int result_len = 0;
  int result_maxlen = 6;

  if (!string)
    return (result);

#ifdef OLD
  while (string[i])
    {
      for (; c = string[i] && isspace (string[i]); i++);
      for (start = i; c = string[i] && !isspace (string[i]); i++);

      if (start == i)
	return (result);

    }
#endif /* OLD */
  string = skip_space (string);
  token_set_delimiters ("<>");
  while (string && *string)
    {
      if (!result)
	result = (char **)xmalloc ((2 + result_maxlen) * sizeof (char *));
      else if (result_len >= result_maxlen - 1)
	{
	  result_maxlen *= 2;
	  result =
	    (char **)xrealloc (result, (2 + result_maxlen) * sizeof (char *));
	}
      result[result_len + 1] = (char *)NULL;
      result[result_len] = return_token (string);
      result_len++;
      string = next_token (string);
    }

  if (result)
    result[result_len] = NULL;
  return (result);
}

/* Free the memory allocated to this array of char *. */
void
cmd_args_free (args)
     char **args;
{
  register int i;

  if (args)
    {
      for (i = 0; args[i]; i++)
	xfree (args[i]);
      xfree (args);
    }
}

/* Strip whitespace from the start and end of STRING. */
void
stripwhite (string)
     char *string;
{
  register int i = 0;

  while (whitespace (string[i]))
    i++;

  if (i)
    strcpy (string, string + i);

  i = strlen (string) - 1;

  while (i > 0 && whitespace (string[i]))
    i--;

  string[++i] = '\0';
}

#ifdef READLINE
/* **************************************************************** */
/*                                                                  */
/*                  Interface to Readline Completion                */
/*                                                                  */
/* **************************************************************** */

cmd_null()
{
  return (NULL);
}

cmd_stuff_default_line ()
{
  if (cmd_default_line)
    rl_insert_text (cmd_default_line);
}
     
/* Read a line of input.  Prompt with PROMPT.  A NULL PROMPT means
   none.  A return value of NULL means that EOF was encountered.
   Supply a default_line for initial editing.
 */

char *
cmd_readline_default (prompt, default_line, generator)
     char *prompt;
     char *default_line;
     Function *generator;
{
  char *readline_internal ();
  char *value;
  Function *old_generator;
  Function *old_completion;
  Function *old_startup_hook;

  /* Install our hook for a default line */
  old_startup_hook = rl_startup_hook;
  rl_startup_hook = cmd_stuff_default_line;
  cmd_default_line = default_line;
  
  old_completion =  rl_attempted_completion_function;
  old_generator = rl_completion_entry_function;

  rl_attempted_completion_function = NULL; 
  rl_completion_entry_function = generator;

  value = readline (prompt);

  rl_attempted_completion_function = old_completion;
  rl_completion_entry_function = old_generator;
  rl_startup_hook = old_startup_hook;

  return (value);
}

/* Tell the GNU Readline library how to complete.  We want to try to complete
   on command names if this is the first word in the line, or on filenames
   if not. */
void
cmd_initialize_readline (prompt, completer)
     char *prompt;
     Function *completer;
{
  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = "Lexedit";

  /* Tell the completer that we want a crack first. */
  if (completer)
    rl_attempted_completion_function = completer;

  /* Set the rl_completion_entry_function to a null function so
   * we don't get filenames back.
   */
  rl_completion_entry_function = cmd_null;
}

/* Generator function for command completion.  STATE lets us know whether
   to start from scratch; without any state (i.e. STATE == 0), then we
   start at the top of the list. */
char *
cmd_generator (text, state)
     char *text;
     int state;
{
  static int list_index, len;
  char *name;

  /* If this is a new word to complete, initialize now.  This includes
     saving the length of TEXT for efficiency, and initializing the index
     variable to 0. */
  if (!state)
    {
      list_index = 0;
      len = strlen (text);
    }

  /* Return the next name which partially matches from the command list. */
  while (name = commands[list_index].name)
    {
      list_index++;

      if (completion_case_fold)
	{
	  if (strncasecmp (name, text, len) == 0)
	    return ((char *)strdup(name));	  
	}
      else
	{
	if (strncmp (name, text, len) == 0)
	  return ((char *)strdup(name));
      }
    }

  /* If no names matched, then return NULL. */
  return ((char *)NULL);
}

/* Generator for help completion */
char *
cmd_help_generator (text, state)
     char *text;
     int state;
{
  static int list_index, len;
  char *name;
  static int commands_done;
  static int cmd_state;

  /* If this is a new word to complete, initialize now.  This includes
     saving the length of TEXT for efficiency, and initializing the index
     variable to 0. */
  if (!state)
    {
      list_index = 0;
      len = strlen (text);
      commands_done = 0;
      cmd_state = 0;
    }

  if (commands_done == 0)
    {
      name = cmd_generator (text, cmd_state);
      cmd_state = 1;
      if (name != NULL)
	return (strdup (name));
    }

  /* Return the next name which partially matches from the command list. */
  while (name = topics[list_index].name)
    {
      list_index++;

      if (completion_case_fold)
	{
	  if (strncasecmp (name, text, len) == 0)
	    return ((char *)strdup(name));	  
	}
      else
	{
	if (strncmp (name, text, len) == 0)
	  return ((char *)strdup(name));
      }
    }

  /* If no names matched, then return NULL. */
  return ((char *)NULL);
}
#endif

void
cmd_show_topic (topic, fd, data)
     TOPIC *topic;
     int fd;
     void *data;
{
#ifdef NEVER
  FILE *input;
  char *filename;
  char c;

  if (strncmp ("File:", topic->doc, 5) == 0)
    {
      filename = &topic->doc[5];
      while (*filename && isspace(*filename))
	filename++;
      input = fopen (filename, "r");
      if (input == NULL)
	{
	  fprintf (stderr, "Cannot open file `%s'\n", filename);
	  return;
	}

      while ((c = fgetc(input)) != EOF)
	putchar (c);
      fclose (input);
    }
  else
    fprintf (file, "%s\n%s\n", topic->name, topic->doc);
#endif
} 

/*
 * Some default commands 
 */

/* Print out help for ARG, or for all of the commands if ARG is
   not present. */
void
com_help (args, fd, data)
     char **args;
     int fd;
     void *data;
{
  register int i;
  int printed = 0;
  string_declare (output);

  string_init (output, 128);

  if (args[1] == NULL)      /* No args to give a summary */
    {
      for (i = 0; commands[i].name; i++)
	{
	  string_append_string (output, commands[i].name);
	  string_append_string (output, "\t\t");
	  string_append_string (output, commands[i].doc);
	  string_append_string (output, ".\n");
	}
      for (i = 0; topics[i].name; i++)
	{
	  string_append_string (output, topics[i].name);
	  string_append_string (output, "\t\t");
	  string_append_string (output, topics[i].summary);
	  string_append_string (output, ".\n");
	}
    }
  else
    {
      for (i = 0; commands[i].name; i++)
	{
	  if ((strcmp (args[1], commands[i].name) == 0) ||
	      (completion_case_fold && 
	       (strcasecmp (args[1], commands[i].name) == 0)))
	    {
	      string_append_string (output, commands[i].name);
	      string_append_string (output, "\t\t");
	      string_append_string (output, commands[i].doc);
	      string_append_string (output, ".\n");
	      printed++;
	    }
	}
      for (i = 0; topics[i].name; i++)
	{
	  if ((strcmp (args[1], topics[i].name) == 0) ||
	      (completion_case_fold && 
	       (strcasecmp (args[1], topics[i].name) == 0)))
	    {
	      cmd_show_topic (topics[i], fd, data);
	      printed++;
	    }
	}
      if (!printed)
	{
	  string_append_string (output, "No commands match `");
	  string_append_string (output, args[1]);
	  string_append_string (output, "'.  Possibilities are:\n");
	  for (i = 0; commands[i].name; i++)
	    {
	      /* Print in six columns. */
	      if (printed == 6)
		{
		  printed = 0;
		  string_append_string (output, "\n");
		}
	      string_append_string (output, commands[i].name);
	      string_append_string (output, "\t");
	      printed++;
	    }
	  
	  for (i = 0; topics[i].name; i++)
	    {
	      /* Print in six columns. */
	      if (printed == 6)
		{
		  printed = 0;
		  string_append_string (output, "\n");
		}
	      string_append_string (output, topics[i].name);
	      string_append_string (output, "\t");
	      printed++;
	    }
	  
	  if (printed)
	    string_append_string (output, "\n");
	}
    }
  write (fd, output, output_length);
}


output_string (fd, string)
     int fd;
     char *string;
{
  if (string == NULL)
    write (fd, "NULL", 4);
  else
    write (fd, string, strlen (string));
}

output_int (fd, n)
     int fd;
     int n;
{
  char buffer[128];
  sprintf (buffer, "%d", n);
  output_string (fd, buffer);
}

#if defined (STATIC_MALLOC)

/* **************************************************************** */
/*								    */
/*			xmalloc and xrealloc ()		     	    */
/*								    */
/* **************************************************************** */

static void memory_error_and_abort ();

static void *
xmalloc (bytes)
     int bytes;
{
  void *temp = (void *)malloc (bytes);

  if (!temp)
    memory_error_and_abort ();
  return (temp);
}

static void *
xrealloc (pointer, bytes)
     void *pointer;
     int bytes;
{
  void *temp;

  if (!pointer)
    temp = (void *)malloc (bytes);
  else
    temp = (void *)realloc (pointer, bytes);

  if (!temp)
    memory_error_and_abort ();

  return (temp);
}

static void
memory_error_and_abort ()
{
  fprintf (stderr, "readline: Out of virtual memory!\n");
  abort ();
}
#endif /* STATIC_MALLOC */


#ifndef READLINE
char *
readline (prompt, file)
     char *prompt;
     FILE *file;
{
  char c;
  char *line;
  int line_max = 8;
  int line_count = 0;

  line = xmalloc (line_max);
  *line = '\0';

  while ((c = fgetc (file)) != EOF)
    {
      if (c == '\n')
	break;
      if (c == '\\')
	{
	  c = fgetc (file);
	  if (c == '\n')
	    continue;
	}

      if ((line_count + 1) >= line_max)
	{
	  line_max *= 2;
	  line = xrealloc (line, line_max);
	}
      line[line_count] = c;
      line_count++;
    }
  if (line_count == 0 &&  c == EOF)
    return (NULL);
    
  line[line_count] = '\0';
  return (line);
}
#endif

#ifdef TEST

com_quit ()
{
  cmd_done = 1;
}

/* Test driver */
main (argc, argv)
     int argc;
     char **argv;
{
  cmd_add_command ("quit", com_quit, "Exit program.", NULL);
  cmd_loop ("Testing> ", cmd_completion);
}
#endif
