/*  bufferize.c: Functions to make bprintf work like fprintf, but to a buffer.
    Author: Brian J. Fox (bfox@ai.mit.edu) Thu Apr 20 12:44:17 1995. */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <ctype.h>
#include "xmalloc.h"
#include "bprintf.h"

/* Create a new empty output buffer. */
BprintfBuffer *
create_output_buffer (void)
{
  BprintfBuffer *buffer;

  buffer = (BprintfBuffer *)xmalloc (sizeof (BprintfBuffer));
  memset (buffer, 0, sizeof (BprintfBuffer));

  return (buffer);
}

/* Free the contents of an output buffer. */
void
free_output_buffer (BprintfBuffer *buffer)
{
  if (buffer->buffer)
    free (buffer->buffer);

  free (buffer);
}

/* Resize BUFFER so that it has enough space to store LEN bytes. */
static void
resize_BprintfBuffer (BprintfBuffer *buffer, int len)
{
  if ((buffer->bindex + len + 2) > buffer->bsize)
    buffer->buffer = (char *)xrealloc
      (buffer->buffer, (buffer->bsize += (len + 100)));
}

/* Add STRING to BUFFER. */
static void
add_string_to_buffer (char *string, BprintfBuffer *buffer)
{
  if (string)
    {
      int len = strlen (string);

      resize_BprintfBuffer (buffer, len);
      strcpy (buffer->buffer + buffer->bindex, string);
      buffer->bindex += len;
    }
}

/* The list of characters which terminate a printf specification. */
static char *printf_arg_specifiers = "diouxXDOUeEfgcspn%";

/* The functions which actually format a small amount of information. */

/* Only used to format numbers, or strings which are guaranteed to fit. */
#define FORMAT_BUFF_SIZE 512
static char format_buff[FORMAT_BUFF_SIZE];
static int min_field_width = 0;
static int precision = 0;

/* Parse the MIN_FIELD_WIDTH and PRECISION specifiers in SPEC.  SPEC is
   the complete printf specification of what to display, such as "%6.2f". */
static void
parse_field_specifiers (char *spec)
{
  register int start, end;
  int parsing_width = 1;
  char numbuff[20];

  min_field_width = 0;
  precision = 0;

  /* If no field width or precision specifiers, then we are all done. */
  if (strlen (spec) == 2)
    return;

  /* Skip non-digit characters. */
  for (start = 0; spec[start] != '\0' && !isdigit (spec[start]); start++)
    if (spec[start] == '.')
      parsing_width = 0;

  /* If no more characters left, no field width or precision specifiers. */
  if (spec[start] == '\0')
    return;

  /* Gather all of the digits up to a decimal point. */
  for (end = start; spec[end] != '\0' && isdigit (spec[end]); end++);

  strncpy (numbuff, spec + start, end - start);
  numbuff[end - start] = '\0';

  sscanf (numbuff, "%d", parsing_width ? &min_field_width : &precision);

  if (!parsing_width)
    return;

  /* Time to parse the precision. */
  for (start = end; spec[start] != '\0' && !isdigit (spec[start]); start++);

  if (!isdigit (spec[start]))
    return;

  for (end = start; spec[end] != '\0' && isdigit (spec[end]); end++);

  strncpy (numbuff, spec + start, end - start);
  numbuff[end - start] = '\0';

  sscanf (numbuff, "%d", &precision);
}

static char *
format_long (char *spec, long value)
{
  sprintf (format_buff, spec, value);
  return (format_buff);
}

static char *
format_short (char *spec, short value)
{
  sprintf (format_buff, spec, value);
  return (format_buff);
}

static char *
format_int (char *spec, int value)
{
  sprintf (format_buff, spec, value);
  return (format_buff);
}

static char *
format_double (char *spec, double value)
{
  sprintf (format_buff, spec, value);
  return (format_buff);
}

static char *
format_character (char *spec, int character)
{
  format_buff[0] = character;
  format_buff[1] = '\0';
  return (format_buff);
}

static char *
format_string (char *spec, char *string)
{
  parse_field_specifiers (spec);

  if (min_field_width)
    {
      if ((min_field_width + strlen (string)) < FORMAT_BUFF_SIZE)
	{
	  sprintf (format_buff, spec, string);
	  return (format_buff);
	}
      else
	{
	  static char *fb = (char *)NULL;

	  if (fb) free (fb);
	  fb = (char *)xmalloc (20 + min_field_width + strlen (string));
	  sprintf (fb, spec, string);
	  return (fb);
	}
    }
  return (string);
}

/* The main function in this library.  Print to BUFFER with FORMAT and
   any additional args. */
void
bprintf (BprintfBuffer *buffer, char *format, ...)
{
  va_list args;

  va_start (args, format);

  vbprintf (buffer, format, args);
}

void
vbprintf (BprintfBuffer *buffer, char *format, va_list args)
{
  register int i, c;
  char *accum = (char *)NULL;
  int accum_index = 0;
  int accum_size = 0;

  for (i = 0; (c = format[i]) != '\0'; i++)
    {
      if ((c != '%') || (format[i + 1] == '\0'))
	{
	  if ((accum_index + 3) > accum_size)
	    accum = (char *)xrealloc (accum, (accum_size += 100));

	  accum[accum_index++] = c;
	  accum[accum_index] = '\0';
	}
      else
	{
	  int start = i;
	  int long_format = 0;
	  int short_format = 0;
	  char *result = (char *)NULL;
	  char *spec;

	  if (accum_index)
	    {
	      add_string_to_buffer (accum, buffer);
	      accum_index = 0;
	      accum[0] = '\0';
	    }

	  while ((c = format[++i]) != '\0')
	    {
	      if (c == 'l')
		long_format++;
	      else if (c == 'h')
		short_format++;
	      else if (strchr (printf_arg_specifiers, c) != (char *)NULL)
		break;
	    }

	  spec = (char *)xmalloc (2 + (i - start));
	  memcpy (spec, format + start, ((i - start) + 1));
	  spec[(i - start) + 1] = '\0';

	  /* Handle the specific formatter. */
	  switch (c)
	    {
	    case 'd':
	    case 'i':
	    case 'o':
	    case 'u':
	    case 'x':
	    case 'X':
	      {
		if (long_format)
		  {
		    long value;

		    value = va_arg (args, long);
		    result = format_long (spec, value);
		  }
		else if (short_format)
		  {
		    short value;

		    value = va_arg (args, short);
		    result = format_short (spec, value);
		  }
		else
		  {
		    int value;

		    value = va_arg (args, int);
		    result = format_int (spec, value);
		  }
	      }
	      break;

	    case 'O':
	    case 'U':
	    case 'p':
	      {
		unsigned long value;

		value = va_arg (args, long);
		result = format_long (spec, value);
	      }
	      break;

	    case 'e':
	    case 'E':
	    case 'f':
	    case 'g':
	    case 'G':
	      {
		double value;

		value = va_arg (args, double);

		result = format_double (spec, value);
	      }
	      break;

	    case 'c':
	      {
		int value;

		value = va_arg (args, int);
		result = format_character (spec, value);
	      }
	      break;

	    case 's':
	      {
		char *value;

		value = va_arg (args, char *);
		result = format_string (spec, value);
	      }
	      break;

	    case '%':
	      result = "%";
	      break;

	      /* The number of characters written so far is stored into the
		 integer indicated by the ``int *'' (or variant) pointer
		 argument.  No argument is converted. */
	    case 'n':
	      {
		int * value;

		value = va_arg (args, int *);
		*value = buffer->bindex;
	      }
	      break;

	    default:
	      {
		static char bad_idea[3];

		bad_idea[0] = '%';
		bad_idea[1] = c;
		bad_idea[2] = '\0';
	      
		result = bad_idea;
	      }
	    }

	  if (result)
	    add_string_to_buffer (result, buffer);

	  free (spec);
	}
    }
  /* If there are any accumulated characters left over, add them to the
     buffer as well. */
  if (accum_index)
    add_string_to_buffer (accum, buffer);
  if (accum)
    xfree (accum);
}
