/*
 * @(#)BranchCountMeasure.java
 *
 * Copyright (C) 2003-2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.module;


import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCode;

import org.apache.bcel.classfile.CodeException;
import org.apache.bcel.generic.IfInstruction;
import org.apache.bcel.generic.Instruction;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.JsrInstruction;
import org.apache.bcel.generic.Select;


/**
 * Processes methods for branch coverage analysis, where each branch
 * instruction and its corresponding not-taken instruction are marked.
 * Currently, this does not support localization.
 * <P>
 * This is more accurately called "Object Code Branch Coverage", since
 * true branch coverage requires the originating source code to correctly
 * discover the branches.
 * <P>
 * This measure can be superior to line coverage due to the Java construct
 * of the <tt>?:</tt> operation.  This hides a branch inside a single
 * statement.  Also, some developers may put an <tt>if</tt> statement and
 * its one-line branch all on the same line, which will hide the branch
 * that was taken.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @author    Juergen Kindler <a href="mailto:jkindler@freenet.de">jkindler@freenet.de</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     January 26, 2003
 * @see       IAnalysisMetaData
 */
public class BranchCountMeasure extends AbstractMeasure
{
    //private static final org.apache.log4j.Logger LOG =
    //    org.apache.log4j.Logger.getLogger( BranchCountMeasure.class );
    
    /**
     * Returns the human-readable name of the measure.
     */
    public String getMeasureName()
    {
        return "Branch";
    }
    
    /**
     * Returns the unit name for this particular coverage measure.
     */
    public String getMeasureUnit()
    {
        return "branches";
    }
    
    
    /**
     * Returns the text format used in meta-data formatted text.  This should
     * be the mime encoding type, such as "text/plain" or "text/html".
     */
    public String getMimeEncoding()
    {
        return "text/plain";
    }
    
    
    /**
     * Perform the analysis on the method.
     */
    public void analyze( IMethodCode method )
    {
        BytecodeLineUtil blu = new BytecodeLineUtil( method );
        InstructionHandle handles[] = blu.getHandles();
        
        // always mark the first instruction, even if there are no
        // instructions.
        markInstruction( method, 0, createAMD( "Start of Method",
            blu.getLineNumberForInstructionPos( 0 ) ), false );

        
        // find the positions of the instructions in the bytecode
        for (int i = 0; i < handles.length; ++i)
        {
            InstructionHandle h = handles[i];
            Instruction instr = h.getInstruction();
            
            // Mark the if, select, and jsr instructions
            if (instr instanceof IfInstruction)
            {
                markIf( method, (IfInstruction)instr, i, blu );
            }
            else
            if (instr instanceof Select)
            {
                markSelect( method, (Select)instr, i, blu );
            }
            else
            if (instr instanceof JsrInstruction)
            {
                markJsr( method, (JsrInstruction)instr, i, blu );
            }
        }
        
        // Also need to mark all exception handlers.
        CodeException exceptions[] = method.getOriginalMethod().getCode().
            getExceptionTable();
        for (int i = 0; i < exceptions.length; ++i)
        {
            markExceptionHandler( method, exceptions[i], blu );
        }
    }
    
    
    /**
     * The target will be the "else" or "default" instruction.
     */
    private void markIf( IMethodCode method, IfInstruction instr,
            int i, BytecodeLineUtil blu )
    {
        int lineNo = blu.getLineNumberForInstructionPos( i+1 );
        
        // the mark on this part goes AFTER the branch instruction,
        // not before it.
        // Bug 906198
        // Bug 906207
        markInstruction( method, i+1,
            createAMD( "'True' branch", lineNo ), false );
        
        InstructionHandle target = instr.getTarget();
        markTarget( method, target, blu, "'False' branch" );
    }
    
    
    /**
     * 
     */
    private void markSelect( IMethodCode method, Select instr,
            int i, BytecodeLineUtil blu )
    {
        int lineNo = blu.getLineNumberForInstructionPos( i+1 );
        markInstruction( method, i+1,
            createAMD( "After switch", lineNo ), false );
        
        InstructionHandle targets[] = instr.getTargets();
        for (int tIndex = 0; tIndex < targets.length; ++tIndex)
        {
            markTarget( method,
                targets[ tIndex ],
                blu,
                "Case index "+Integer.toString( tIndex + 1 ) );
        }
    }
    
    
    /**
     * 
     */
    private void markJsr( IMethodCode method, JsrInstruction instr,
            int i, BytecodeLineUtil blu )
    {
        // For a JSR, the target indicates the location of a finally block.
        InstructionHandle target = instr.getTarget();
        markTarget( method, target, blu, "Finally block" );
    }
    
    
    /**
     * 
     */
    private void markExceptionHandler( IMethodCode method, CodeException ce,
            BytecodeLineUtil blu )
    {
        int pc = ce.getHandlerPC();
            
        // Bug 906207
        markInstruction( method,
            blu.getInstructionPosForBytecodePos( pc ),
            createAMD( "Exception handler",
                blu.getLineNumberForBytecodePos( pc ) ),
            false );
    }
    
    
    private void markTarget( IMethodCode method, InstructionHandle target,
            BytecodeLineUtil blu, String branchDesc )
    {
        if (target != null)
        {
            // This does not relate to bug 906207 due to bug 906211
            markInstruction( method,
                blu.getInstructionPosForBytecodePos( target.getPosition() ),
                createAMD(
                    branchDesc,
                    blu.getLineNumber( target ) ),
                false );
        }
    }
    
    
    /**
     * 
     */
    private IAnalysisMetaData createAMD( String type, int lineNo )
    {
        return new DefaultAnalysisMetaData(
            type+" at line "+lineNo,
            "Didn't cover "+type+" at line "+lineNo,
            (byte)0 );
    }
}
