/**
 * @file inipool.cpp
 * @author Christian Schenk
 * @brief WEB pool to C (based on makecpool)
 *
 * @copyright Copyright © 2008-2022 Christian Schenk, © 2007-2008 Taco Hoekwater
 *
 * This file is free software; the copyright holder gives unlimited permission
 * to copy and/or distribute it, with or without modifications, as long as this
 * notice is preserved.
 */

#include <cstdio>
#include <cstring>
#include <cstdlib>

int main(int argc, char* argv[])
{
    if (argc != 5)
    {
        fprintf(stderr, "%s: need exactly four arguments: pool file name, header file name, program class name and program instance name.\n", argv[0]);
        exit(EXIT_FAILURE);
    }
    char* filename = argv[1];
    char* headername = argv[2];
    char* progclass = argv[3];
    char* progname = argv[4];
    FILE* fh = fopen(filename, "r");
    if (fh == nullptr)
    {
        fprintf(stderr, "%s: can't open %s for reading.\n", argv[0], filename);
        exit(EXIT_FAILURE);
    }
    bool is_metafont = false;
    bool is_metapost = false;
    bool is_luatex = false;
    if (strstr(filename, "luatex.pool") != nullptr)
    {
        is_luatex = true;
    }
    else if (strstr(filename, "mp.pool") != nullptr)
    {
        is_metapost = true;
    }
    else if (strstr(filename, "mf.pool") != nullptr)
    {
        is_metafont = true;
    }
    printf("\
/*\n\
 * This file is auto-generated by inipool.\n\
 *   %s %s %s\n\
 */\n\
\n\
#include <string.h>\n\
#include \"%s\"\n\
\n\
\n\
extern %s %s;\n\
#define PROG %s\n\
\n\
static const char* poolfilearr[] = {\n",
        argv[0],
        filename,
        headername,
        headername,
        progclass,
        progname,
        progname);
    char data[1024];
    while (fgets(data, 1024, fh))
    {
        int len = strlen(data);
        if (data[len - 1] == '\n')
        {
            data[len - 1] = 0;
            len--;
        }
        if (data[0] == '*')
        {
            break;
        }
        // skip first bytes
        int o = 0;
        if (data[0] >= '0'
            && data[0] <= '9'
            && data[1] >= '0'
            && data[1] <= '9')
        {
            o = 2;
        }
        printf("  \"");
        for (int i = o; i < len; i++)
        {
            if (data[i] == '"' || data[i] == '\\')
            {
                putchar('\\');
            }
            if (data[i] == '?')
            {
                printf("\" \"");        // suppress trigraphs
            }
            putchar(data[i]);
        }
        printf("\",\n");
    }
    fclose(fh);
    printf("\
  nullptr\n\
};\n\
\n\
int miktexloadpoolstrings(int poolSize)\n\
{\n\
  const char* lpsz;\n\
  int g = 0;\n\
  int size = 0;\n\
  for (int idx = 0; (lpsz = poolfilearr[idx]) != nullptr; ++idx)\n\
  {\n\
    int len = strlen(lpsz);\n\
    size += len;\n\
    if (size >= poolSize)\n\
    {\n\
      return 0;\n\
    }\n");
    if (is_luatex)
    {
        printf("\
    while (len-- > 0)\n\
    {\n\
      PROG.str_pool[PROG.pool_ptr++] = *lpsz++;\n\
    }\n\
    g = PROG.make_string();\n");
    }
    else
    {
        printf("\
    while (len-- > 0)\n\
    {\n\
      PROG.strpool[PROG.poolptr++] = *lpsz++;\n\
    }\n\
    g = PROG.makestring();\n");
    }
    if (is_metapost || is_metafont)
    {
        printf("\
    PROG.strref[g] = 127;\n");
    }
    printf("\
  }\n\
  return g;\n\
}\n");
    return EXIT_SUCCESS;
}
