#include "system.h"
#include "config.h"
#include "mem.h"
#include "mfileio.h"
#include "pdfobj.h"
#include "error.h"
#include "filter.h"
#include "concat.h"

#define MAX_BUFFER_SIZE 409600
static unsigned long get_content_stream(char *dest, unsigned long destlen,
					pdf_stream *stream);

pdf_obj *cat_contents (struct pdf_array *contents)
{
  pdf_obj *result = NULL;
  pdf_obj *content_ref = NULL;
  pdf_obj *component = NULL;
  pdf_stream *stream;
  unsigned long length, filtered_length;
  int i, fail;
  char *buffer;
  unsigned long buffer_length;

#ifdef MEM_DEBUG
MEM_START
#endif
#ifdef DEBUG
  fprintf(stderr, "\n%ld content streams found\n", contents->size);
#endif /* DEBUG */

  result = pdf_new_stream(STREAM_COMPRESS);

  fail = 0;
  for (i=0;i<(contents->size);i++) {
    content_ref = (contents->values)[i];
    if ((component = pdf_deref_obj(content_ref)) != NULL) {
      if (component->type == PDF_STREAM) {
	stream = component->data;
	length = (unsigned long) stream->stream_length;
	if( length < MAX_BUFFER_SIZE ) {
	  /* Maybe sufficient for most PDF content streams.
	     (If in-line images are not used.) */
	  if( 100*length < MAX_BUFFER_SIZE ) {
	    buffer_length = 100*length;
	  } else {
	    buffer_length = MAX_BUFFER_SIZE;
	  }
	} else {
	  fprintf(stderr, "Stream size too large\n");
	  fail = 1;
	  break;
	}
	buffer = NEW(buffer_length, char);
	filtered_length = get_content_stream(buffer, buffer_length, stream);
	if (filtered_length == 0 && length != 0) {
	  RELEASE(buffer);
	  fail = 1;
	  break;
	} else {
	  pdf_add_stream(result, buffer, filtered_length);
#ifdef DEBUG
	  {
	    unsigned long debug_k;

	    fprintf(stderr, "appended %lu bytes stream\n", length);
	    fprintf(stderr, "--- stream ---\n");
	    for(debug_k=0;debug_k<length;debug_k++)
	      fputc(buffer[debug_k], stderr);
	    fprintf(stderr, "\n--- endstream ---\n");
	  }
#endif /* DEBUG */
	  RELEASE(buffer);
	  /* continue */
	}
      } else {
	fprintf(stderr, "Not a stream object !\n");
	fail = 1;
	break;
      }
    } else {
      fprintf(stderr, "Could not find contets stream !\n");
      fail = 1;
      break;
    }
  }

  if (fail) {
    pdf_release_obj(result);
    result = NULL;
  }

#ifdef MEM_DEBUG
MEM_END
#endif

  return result;
}

unsigned long
get_content_stream(char *dest, unsigned long destlen, pdf_stream *stream)
{
  unsigned long stream_length, filtered_length = 0;
  char *stream_data;
  pdf_obj *tmp;
  pdf_obj *stream_dict;
  pdf_obj *filter;
  char *filter_name;
  pdf_array *filter_array;
  pdf_obj *a_filter;
  char *buffer;
  unsigned long buffer_length;
  int i;

#ifdef MEM_DEBUG
MEM_START
#endif

  stream_dict = stream->dict;
  /* Can contain Kids ? */
  if ( (tmp = pdf_lookup_dict(stream_dict, "Kids")) != NULL ) {
    fprintf(stderr, "Kids found !!!\n");
    return 0;
  }

  if ( (tmp = pdf_lookup_dict(stream_dict, "DecodeParms")) != NULL ) {
    fprintf(stderr, "DecodeParms not supported !!!\n");
    return 0;
  }

  stream_length = (unsigned long) stream->stream_length;
  if ((stream_data = stream->stream) != NULL) {
    if ((filter = pdf_lookup_dict(stream_dict, "Filter")) == NULL) {
#ifdef DEBUG
      fprintf(stderr, "No filter required\n");
#endif /* DEBUG */
      memcpy(dest, stream_data, stream_length);
      filtered_length = stream_length;
    } else {
#ifdef DEBUG
      fprintf(stderr, "Filter required\n");
#endif /* DEBUG */
      if (filter->type == PDF_NAME) {
	filter_name = pdf_name_value(filter);
	filtered_length = decode_stream(dest, destlen, stream_data,
					stream_length, filter_name);
      } else if (filter->type == PDF_ARRAY) {
	filter_array = filter->data;
	buffer_length = destlen;
	buffer = NEW(buffer_length, char);
	memcpy(buffer, stream_data, stream_length);
	for(i=0;i<filter_array->size;i++) {
	  a_filter = pdf_get_array(filter, i);
	  filter_name = pdf_name_value(a_filter);
	  filtered_length = decode_stream(dest, destlen, buffer,
					  stream_length, filter_name);
	  if(filtered_length > 0 && filtered_length <= buffer_length) {
	    if(i != filter_array->size -1) {
	      memcpy(buffer, dest, filtered_length); /* apply next filter */
	      stream_length = filtered_length;
	    }
	  } else {
	    filtered_length = 0;
	    break;
	  }
	}
	RELEASE(buffer);
      } else {
	fprintf(stderr, "Filter value not a name/array !\n");
	filtered_length = 0;
      }
    }
  }

#ifdef MEM_DEBUG
MEM_END
#endif

return filtered_length;
}
