/*  Universal character set conversion utility

  ucscu.c

  To compile on most systems:

     #define UCSCU_STANDALONE

     cc -o ucscu ucscu.c

   When compiled as a standalone program, it reads a text file and converts
   characters from one character set to another.

   When compiled as a library (STANDALONE not defined)
   the following functions are available:

   UCSCU_file()   Converts file just like standalone.
   UCSCU_string() Converts the character string.
   UCSCU_char()   Given a UCSCU symbol code, returns the character
                  string to draw the symbol.

   UCSCU_prompt_input_options()  Prompts the user for input options.
   UCSCU_prompt_output_options() Prompts the user for output options.
   UCSCU_prompt_options()        Prompts user for both I/O options.
   UCSCU_init()   Must be called once before using UCSCU_string, UCSCU_char,
                  or UCSCU_file if UCSCU_prompt_options.
                  If prompting user for just input or just output options,
                  then UCSCU_init must be called before calling the
                  UCSCU_prompt_xxx_options().

   This program is freely distributable, but not in the Public Domain.
   Do not distribute MODIFIED versions of this program; however,
   feel free to incorporate portions of this program in your own software.
   I reserve all rights in the distribution of the program.
   Please send me your modifications for incorporation in future versions.

   The file should be UUENCODE or transfered in BINARY mode, because
   it contains 8-bit characters (only in comments)!

   Written by Roger Nelson

       Biological Systems Engr. Dept.
       Washington State University
       Pullman, WA 99164-6120

       rnelson@wsuaix.csc.wsu.edu

       Phone: (509)335-1100  or 335-1578 (main office)
       Home:  (509)332-8387
       FAX  : (509)335-2722

   Version 0.0  Jul 25 '93
           1.0  Aug 30 '93 -INPUT/-OUTPUT in docs should read INPUT/OUTPUT.
                           End of file character was inserted into text.
                           Significantly faster operation.
           1.01 Oct  3 '94 Some corrections for strict C/C++ compilers

   Thanks to:

     Hermann Hueni Seidenweg 40, CH-3012 Bern,		    +41 31 234 607
     GLUE Software Engineering, Seidenweg 5, CH-3012 Bern,  +41 33 457 888
     e-mail: hueni@iam.unibe.ch			            +41 31 248 362
     For suggestions and bug fixes

     Andreas Scherer
     Dipl.-Math., Stud. O.R. an der RWTH Aachen
     Roland-Strae 16
     52070 Aachen, Deutschland (Germany)
     <scherer@genesis.informatik.rwth-aachen.de>
     For German translation and several corrections for ANSI C++ and Amiga SAS/C++

*/

#include <stdio.h>
#include <string.h> /* may be <strings.h> on some systems */
#include <stdlib.h>
#include <assert.h>

#ifdef UCSCU_STANDALONE
#include "ucscu.h"
#else
#include <rlib/ucscu.h>
#endif

#define UCSCU_FAILED    1
#define UCSCU_SUCCEEDED 0

/* VMS users should #define VMS */
/* #define VMS */

/* MSDOS (and probably OS2 users should already have __MSDOS__ defined by
   the compiler otherwise it needs to be defined*/
/* #define __MSDOS__   */

#ifdef __cplusplus
#ifndef __STDC__
#define __STDC__
#endif
#endif

#ifdef __MSDOS__
#define WRITEMODE "wt"
#define READMODE  "rt"
#else
#define WRITEMODE "w"
#define READMODE "r"
#endif

/* 7Bit ASCII/ISO character sets:     */

#define CS_NONE         -1

#define CS_ENGLISH7      0
#define CS_GERMAN7       1  /* Also Austrian */
#define CS_DUTCH7        2
#define CS_FRENCH7       3
#define CS_SWEDISH7      4
#define CS_NORDIC7       5  /* Norwegian and Danish */
#define CS_ITALIAN7      6
#define CS_SPANISH7      7
#define CS_PORTUGUESE7   8
#define CS_FINNISH7      9
#define CS_BELGIAN7     10  /* Belgian french and flemmish */
#define CS_SWISS7       11

#define CS_TeX         100  /* To output TeX symbols */

/* Do the following character sets exists? others?
#define CS_GREEK7
#define CS_CHECH7
#define CS_SLOVAK7
#define CS_HUNGARIAN7
#define CS_POLISH7
#define CS_YUGOSLAVIAN7
*/

/* 8bit character sets */

#define CS_PC8     0  /* English IBM-PC also known as  */
#define CS_MNCS    1  /* Multinational caracter set */
#define CS_DEC8    1  /* Identical to MNCS except 215 is OE and 247 is oe and DEC doesn't define several*/
#define CS_ECMA94  1  /* Identical to MNCS */
#define CS_AMIGA   1  /* Identical to MNCS */
#define CS_ROMAN8  2  /* Most English HP */
#define CS_EPSON8  3  /* Most Epson compatible printers */
#define CS_PC850   4  /* MS-DOS PC Multilingual (Latin 1) */
#define CS_PC852   5  /* IBM-PC-852 Slavic (Latin I) */
#define CS_PC860   6  /* IBM-PC-860 Portuguese */
#define CS_PC863   7  /* IBM-PC-863 Canadian-French */
#define CS_PC865   8  /* IBM-PC-865 Nordic */
#define CS_PC8DN   9  /* Denmark/Norway IBM-PC8*/
#define CS_QUME             10
#define CS_TOSHIBA_1340     10
#define CS_QUME_SPRINT_11   10
#define CS_DIABLO_630       10

/* Backspacing options in generating ASCII representation:*/

#define BS_NORMAL  0 /* output entire string */
#define BS_NEVER   1 /* Only print first character */
#define BS_NOT_CAP 2 /* Backspace only on lower case characters */
#define BS_OMIT    3 /* Add diacritical marks, but leave out BS */

#define GR_NORMAL  0 /* output entire string */
#define GR_MARK    1 /* use maker character underscore */
#define GR_NONE    2 /* use space */


#define UNKNOWN_MARKER '?'

struct UCSCU_options
{
  char *filename;
  FILE *file;
  int  bit7_CS;
  int  bit8_CS;
};

struct UCSCU_modes
{
  int quiet;       /* 0 */
  int prefer_7bit; /* 0 */
  int backspacing; /*  BS_NORMAL*/
  int graphics;    /*  GR_NORMAL*/
};

struct UCSCU_options UCSCU_input_option;
struct UCSCU_options UCSCU_output_option;
struct UCSCU_modes   UCSCU_output_mode;

int UCSCU_line_count = 0;
int UCSCU_do_direct_conversion = 1;


/* If you wish to provide me with a character set to add to this utility,
   Please provide me with the following information:

   The name of the character set.
   The countries / situation where the character set is used.
   If it is specified to a certian model of computer/terminal/printer etc..

   Send me a copy of the following enumeration
   where the character code is append to the respective line.
   Delete lines where there is no code for the named character in the
   character set.

   If there are new characters and/or symbols or diacritical marks that
   need to be added.  Highlight any lines you add with an * and give a
   brief description of the character/symbol.
*/


/* The leading \ is not included in the TeX string */

struct UCSCU_TeX_layout UCSCU_TeX_table[] =
{
  {"b{a}",  a_underlined},
  {"b{o}",  o_underlined},
  {"`{A}",  A_grave},
  {"'{A}",  A_acute},
  {"^{A}",  A_circumflex},
  {"~{A}",  A_tilde},
  {"\"{A}", A_diaeresis},
  {"o{A}",  A_circle},
  {"t{AE}",  A_E},
  {"c{C}",  C_cedilla},
  {"`{E}",  E_grave},
  {"'{E}",  E_acute},
  {"^{E}",  E_circumflex},
  {"\"{E}", E_diaeresis},
  {"`{I}",  I_grave},
  {"'{I}",  I_acute},
  {"^{I}",  I_circumflex},
  {"\"{I}", I_diaeresis},
  {"~{N}",  N_tilde},
  {"`{O}",  O_grave},
  {"'{O}",  O_acute},
  {"^{O}",  O_circumflex},
  {"~{O}",  O_tilde},
  {"\"{O}", O_diaeresis},
  {"t{OE}",  O_E},
  {"`{U}",  U_grave},
  {"'{U}",  U_acute},
  {"^{U}",  U_circumflex},
  {"\"{U}", U_diaeresis},
  {"\"{Y}", Y_diaeresis},
  {"`{a}",  a_grave},
  {"'{a}",  a_acute},
  {"^{a}",  a_circumflex},
  {"~{a}",  a_tilde},
  {"\"{a}", a_diaeresis},
  {"o{a}",  a_circle},
  {"t{a}",  a_e},
  {"t{ae}",  a_e},
  {"c{c}",  c_cedilla},
  {"`{e}",  e_grave},
  {"'{e}",  e_acute},
  {"^{e}",  e_circumflex},
  {"\"{e}", e_diaeresis},
  {"`{i}",  i_grave},
  {"'{i}",  i_acute},
  {"^{i}",  i_circumflex},
  {"\"{i}", i_diaeresis},
  {"~{n}",  n_tilde},
  {"`{o}",  o_grave},
  {"'{o}",  o_acute},
  {"^{o}",  o_circumflex},
  {"~{o}",  o_tilde},
  {"\"{o}", o_diaeresis},
  {"t{oe}",  o_e},
  {"`{u}",  u_grave},
  {"'{u}",  u_acute},
  {"^{u}",  u_circumflex},
  {"\"{u}", u_diaeresis},
  {"\"{y}", y_diaeresis},

/* Abbreviated tex forms.*/
  {"ba",  a_underlined},
  {"bo",  o_underlined},
  {"`A",  A_grave},
  {"'A",  A_acute},
  {"^A",  A_circumflex},
  {"~A",  A_tilde},
  {"\"A", A_diaeresis},
  {"oA",  A_circle},
  {"tA",  A_E},
  {"cC",  C_cedilla},
  {"`E",  E_grave},
  {"'E",  E_acute},
  {"^E",  E_circumflex},
  {"\"E", E_diaeresis},
  {"`I",  I_grave},
  {"'I",  I_acute},
  {"^I",  I_circumflex},
  {"\"I", I_diaeresis},
  {"~N",  N_tilde},
  {"`O",  O_grave},
  {"'O",  O_acute},
  {"^O",  O_circumflex},
  {"~O",  O_tilde},
  {"\"O", O_diaeresis},
  {"tO",  O_E},
  {"`U",  U_grave},
  {"'U",  U_acute},
  {"^U",  U_circumflex},
  {"\"U", U_diaeresis},
  {"\"Y", Y_diaeresis},
  {"`a",  a_grave},
  {"'a",  a_acute},
  {"^a",  a_circumflex},
  {"~a",  a_tilde},
  {"\"a", a_diaeresis},
  {"oa",  a_circle},
  {"ta",  a_e},
  {"cc",  c_cedilla},
  {"`e",  e_grave},
  {"'e",  e_acute},
  {"^e",  e_circumflex},
  {"\"e", e_diaeresis},
  {"`i",  i_grave},
  {"'i",  i_acute},
  {"^i",  i_circumflex},
  {"\"i", i_diaeresis},
  {"~n",  n_tilde},
  {"`o",  o_grave},
  {"'o",  o_acute},
  {"^o",  o_circumflex},
  {"~o",  o_tilde},
  {"\"o", o_diaeresis},
  {"to",  o_e},
  {"`u",  u_grave},
  {"'u",  u_acute},
  {"^u",  u_circumflex},
  {"\"u", u_diaeresis},
  {"\"y", y_diaeresis},

  {"dag",       SYM_daggar},
  {"ddag",      SYM_double_daggar},
  {"paragraph", SYM_paragraph},
  {"copyright", SYM_copyright},
  {"pounds",    SYM_pounds},
  {"section",   SYM_section},
  {"",          UCSCU_END_TABLE},
};

/* The leading \ is not included in the TeX string */

struct UCSCU_HTML_layout UCSCU_HTML_table[] =
{
/*  {"b{a}",  a_underlined}, */
/*  {"b{o}",  o_underlined}, */
  {"&Agrave;",  A_grave},
  {"&Aacute;",  A_acute},
  {"&Acirc;",  A_circumflex},
  {"&Atilde;",  A_tilde},
  {"&Auml;", A_diaeresis},
  {"&Aring;",  A_circle},
  {"&AElig;",  A_E},
  {"&Ccedil;",  C_cedilla},
  {"&Egrave;",  E_grave},
  {"&Eecute;",  E_acute},
  {"&Ecirc;",  E_circumflex},
  {"&Euml;", E_diaeresis},
  {"&Igrave;",  I_grave},
  {"&Iacute;",  I_acute},
  {"&Icirc;",  I_circumflex},
  {"&Iuml;", I_diaeresis},
  {"&Ntilde;",  N_tilde},
  {"&Ograve;",  O_grave},
  {"&Oacute;",  O_acute},
  {"&Ocirc;",  O_circumflex},
  {"&Otilde;",  O_tilde},
  {"&Ouml;", O_diaeresis},
  {"&OElig;",  O_E},
  {"&Ugrave;",  U_grave},
  {"&Uacute;",  U_acute},
  {"&Ucirc;",  U_circumflex},
  {"&Uuml;", U_diaeresis},
  {"&Yuml;", Y_diaeresis},
  {"&agrave;",  a_grave},
  {"&aacute;",  a_acute},
  {"&acirc;",  a_circumflex},
  {"&atilde;",  a_tilde},
  {"&auml;", a_diaeresis},
  {"&oring;",  a_circle},
  {"&aelig;",  a_e},
  {"&ccedil;",  c_cedilla},
  {"&egrave;",  e_grave},
  {"&eacute;",  e_acute},
  {"&ecirc;",  e_circumflex},
  {"&euml;", e_diaeresis},
  {"&igrave;",  i_grave},
  {"&iacute;",  i_acute},
  {"&icirc;",  i_circumflex},
  {"&iuml;", i_diaeresis},
  {"&ntilde;",  n_tilde},
  {"&ograve;",  o_grave},
  {"&oacute;",  o_acute},
  {"&ocirc;",  o_circumflex},
  {"&otilde",  o_tilde},
  {"&ouml;", o_diaeresis},
  {"&oelig;",  o_e},
  {"&Oslash;", O_slash},
  {"&oslash;", o_slash},

  {"&ugrave;",  u_grave},
  {"&uacute;",  u_acute},
  {"&ucirc;",  u_circumflex},
  {"&uuml;", u_diaeresis},
  {"&yuml;", y_diaeresis},

  {"&szlig;", B_german_S},

  {"dag",       SYM_daggar},
  {"ddag",      SYM_double_daggar},
  {"paragraph", SYM_paragraph},
  {"copyright", SYM_copyright},
  {"pounds",    SYM_pounds},
  {"section",   SYM_section},
  {"",          UCSCU_END_TABLE},
};

/* In the ASCII representations, the following characters will be
   replaced by the respective diacritical marks in the target character set

   ' acute
   ` grave
   ~ tilde
   - -
   ^ cicumflex
   @ circle
   / virgule
   = macron
   _ _
   " diaeresis
   u breve
   v chz   ( I don't know what this is called (upsidedown circumflex)
   * acuteacute
   , cedilla
   . tail  (backwards cedilla or ognok)

*/

#define UCSCU_END_TABLE 0
struct bit8_layout
{
  int  symbol;
  char *ASCII;
  char_8bit codes[11];
};

struct bit8_layout bit8_table[] =
{
/*                                                                        / IBM-PC CSs       \ */
/*                         ASCII       IBM-PC8  DEC-MNCS Roman-8  Epson-8  850 852 860 863 865 DN  QUME*/
 {A_acute                 ,"A\b'"     ,{ 0      ,192/**/,224/**/,192/**/,181,181,134, 0 , 0 , 0 , 0 },},
 {a_acute                 ,"a\b'"     ,{160/**/,225/**/,196/**/,196/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {A_breve                 ,"A\bu"     ,{ 0      , 0      , 0      , 0      , 0 ,198, 0 , 0 , 0 , 0 , 0 },},
 {a_breve                 ,"a\bu"     ,{ 0      , 0      , 0      , 0      , 0 ,199, 0 , 0 , 0 , 0 , 0 },},
 {A_tail                  ,"A\b."     ,{ 0      , 0      , 0      , 0      , 0 ,164, 0 , 0 , 0 , 0 , 0 },},
 {a_tail                  ,"a\b."     ,{ 0      , 0      , 0      , 0      , 0 ,165, 0 , 0 , 0 , 0 , 0 },},
 {A_circle                ,"A\b@"     ,{143/**/,197/**/,208/**/,177/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {a_circle                ,"a\b@"     ,{134/**/,229/**/,212/**/, 0 /*a*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {A_circumflex            ,"A\b^"     ,{ 0      ,194/**/,162/**/,192/**/,182,182,143,132, 0 , 0 , 0 },},
 {a_circumflex            ,"a\b^"     ,{131/**/,226/**/,192/**/,196/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {A_diaeresis             ,"A\b\""    ,{142/**/,196/**/,216/**/,209/**/, 0 , 0 , 0 , 0 , 0 , 0 ,177},},
 {a_diaeresis             ,"a\b\""    ,{132/**/,228/**/,204/**/,214/**/, 0 , 0 , 0 , 0 , 0 , 0 ,182},},
 {A_E                     ,"AE"       ,{146/**/,198/**/,211/**/, 0 /*A*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {a_e                     ,"ae"       ,{145/**/,230/**/,215/**/, 0 /*a*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {A_grave                 ,"A\b`"     ,{ 0      ,193/**/,161/**/,192/**/,183,183,145,142, 0 , 0 , 0 },},
 {a_grave                 ,"a\b`"     ,{133/**/,224/**/,200/**/,193/**/, 0 , 0 , 0 , 0 , 0 , 0 ,161},},
 {A_macron                ,"A\b="     ,{ 0      , 0      , 0      ,192/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {a_macron                ,"a\b="     ,{ 0      , 0      , 0      ,196/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {A_tilde                 ,"A\b~"     ,{ 0      ,195/**/,225/**/,192/**/, 0 , 0 ,142, 0 , 0 ,170, 0 },},
 {a_tilde                 ,"a\b~"     ,{ 0      ,227/**/,226/**/,196/**/, 0 , 0 ,132, 0 , 0 ,169, 0 },},
 {a_underlined            ,"a\b_"     ,{166/**/,170/**/,249/**/, 0 /*a*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {B_german_S              ,"B"        ,{225/**/,223     , 0      ,217     ,225,225, 0 , 0 , 0 , 0 ,185},},

 {C_acute                 ,"C\b'"     ,{ 0      , 0      , 0      , 0      , 0 ,143, 0 , 0 , 0 , 0 , 0 },},
 {c_acute                 ,"c\b'"     ,{ 0      , 0      , 0      , 0      , 0 ,134, 0 , 0 , 0 , 0 , 0 },},
 {C_backwards             ," "        ,{ 0      , 0      , 0      ,189     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {C_chz                   ,"C\b^"     ,{ 0      , 0      , 0      , 0      , 0 ,172, 0 , 0 , 0 , 0 , 0 },},
 {c_chz                   ,"c\b^"     ,{ 0      , 0      , 0      , 0      , 0 ,159, 0 , 0 , 0 , 0 , 0 },},
 {C_cedilla               ,"C\b,"     ,{128/**/,199/**/,180/**/,194/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {c_cedilla               ,"c\b,"     ,{135/**/,231/**/,181/**/, 0 /*c*/, 0 , 0 , 0 , 0 , 0 , 0 ,162},},

 {D_bar                   ,"D\b-"     ,{ 0      ,240     ,227/**/, 0      ,209,209, 0 , 0 , 0 , 0 , 0 },},
 {d_bar                   ,"d\b-"     ,{ 0      , 0      ,228/**/, 0      ,208,208, 0 , 0 , 0 , 0 , 0 },},
 {D_chz                   ,"D\b^"     ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {d_chz                   ,"d\b^"     ,{ 0      , 0      , 0      , 0      , 0 ,212, 0 , 0 , 0 , 0 , 0 },},

 {E_acute                 ,"E\b'"    ,{144/**/,201/**/,220/**/,202/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {e_acute                 ,"e\b'"    ,{130/**/,233/**/,197/**/,219/**/, 0 , 0 , 0 , 0 , 0 , 0 ,187},},
 {E_breve                 ,"E\bu"    ,{ 0      ,202/**/,164/**/,202/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {e_breve                 ,"e\bu"    ,{ 0      ,234     , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {E_chz                   ,"E\b^"    ,{ 0      ,202/**/,164/**/,202/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {e_chz                   ,"e\b^"    ,{ 0      ,234     , 0      , 0      , 0 ,216, 0 , 0 , 0 , 0 , 0 },},
 {E_tail                  ,"E\b."    ,{ 0      , 0      , 0      , 0      , 0 ,168, 0 , 0 , 0 , 0 , 0 },},
 {e_tail                  ,"e\b."    ,{ 0      , 0      , 0      , 0      , 0 ,169, 0 , 0 , 0 , 0 , 0 },},
 {E_circumflex            ,"E\b^"    ,{ 0      ,202/**/,164/**/,202/**/,210, 0 ,138,146, 0 , 0 , 0 },},
 {e_circumflex            ,"e\b^"    ,{136/**/,234/**/,193/**/,218/**/, 0 , 0 ,137, 0 , 0 , 0 , 0 },},
 {E_diaeresis             ,"E\b\""   ,{ 0      ,203/**/,165/**/,202/**/,211,211, 0 ,148, 0 , 0 , 0 },},
 {e_diaeresis             ,"e\b\""   ,{137/**/,235/**/,205/**/, 0 /*e*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {E_grave                 ,"E\b`"    ,{ 0      ,200/**/,163/**/,202/**/,212, 0 ,146,145, 0 , 0 , 0 },},
 {e_grave                 ,"e\b`"    ,{138/**/,232/**/,201/**/,221/**/, 0 , 0 , 0 , 0 , 0 , 0 ,189},},
 {E_macron                ,"E\b="    ,{ 0      , 0      , 0      ,202     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {e_macron                ,"E\b="    ,{ 0      , 0      , 0      ,218     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {I_acute                 ,"I\b'"    ,{ 0      ,205/**/,229/**/, 0 /*I*/,214,214, 0 , 0 , 0 , 0 , 0 },},
 {i_acute                 ,"i\b'"    ,{161/**/,237/**/,213/**/, 0 /*i*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {I_circumflex            ,"I\b^"    ,{ 0      ,206/**/,166/**/, 0 /*I*/,215,215, 0 ,168, 0 , 0 , 0 },},
 {i_circumflex            ,"i\b^"    ,{140/**/,238/**/,209/**/, 0 /*i*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {I_diaeresis             ,"I\b\""   ,{ 0      ,207/**/,167/**/, 0 /*I*/,216, 0 , 0 ,149, 0 , 0 , 0 },},
 {i_diaeresis             ,"i\b\""   ,{139/**/,239/**/,221/**/, 0 /*i*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {I_grave                 ,"I\b`"    ,{ 0      ,204/**/,230/**/, 0 /*I*/,222, 0 ,152, 0 , 0 , 0 , 0 },},
 {i_grave                 ,"i\b`"    ,{141/**/,236/**/,217/**/, 0 /*i*/, 0 , 0 ,139, 0 , 0 , 0 , 0 },},

 {I_P                     ,"I\bP"    ,{ 0      ,222     , 0      ,217     ,231, 0 , 0 , 0 , 0 , 0 , 0 },},
 {I_p                     ,"I\bp"    ,{ 0      ,254     , 0      ,217     ,232, 0 , 0 , 0 , 0 , 0 , 0 },},
 {i_j                     ,"ij"      ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {J_circumflex            ,"J\b^"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {j_circumflex            ,"j\b^"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {H_circumflex            ,"H\b^"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {h_circumflex            ,"h\b^"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {L_acute                 ,"L\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,145, 0 , 0 , 0 , 0 , 0 },},
 {l_acute                 ,"l\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,146, 0 , 0 , 0 , 0 , 0 },},
 {L_chz                   ,"L\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,149, 0 , 0 , 0 , 0 , 0 },},
 {l_chz                   ,"l\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,150, 0 , 0 , 0 , 0 , 0 },},
 {L_slash                 ,"L\b/"    ,{ 0      , 0      , 0      , 0      , 0 ,157, 0 , 0 , 0 , 0 , 0 },},
 {l_slash                 ,"l\b/"    ,{ 0      , 0      , 0      , 0      , 0 ,136, 0 , 0 , 0 , 0 , 0 },},
 {L_dot                   ,"L"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 ,158, 0 },},
 {l_dot                   ,"l"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 ,159, 0 },},

 {N_acute                 ,"N\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,227, 0 , 0 , 0 , 0 , 0 },},
 {n_acute                 ,"n\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,228, 0 , 0 , 0 , 0 , 0 },},
 {N_chz                   ,"N\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,213, 0 , 0 , 0 , 0 , 0 },},
 {n_chz                   ,"n\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,229, 0 , 0 , 0 , 0 , 0 },},
 {N_tilde                 ,"N\b~"    ,{165/**/,209/**/,182/**/,213/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {n_tilde                 ,"n\b~"    ,{164/**/,241/**/,183/**/,222/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},


 {O_acuteacute            ,"O\b*"    ,{ 0      , 0      , 0      , 0      , 0 ,138, 0 , 0 , 0 , 0 , 0 },},
 {o_acuteacute            ,"o\b*"    ,{ 0      , 0      , 0      , 0      , 0 ,139, 0 , 0 , 0 , 0 , 0 },},
 {O_acute                 ,"O\b'"    ,{ 0      ,211/**/,231/**/, 0 /*O*/,224,224,157, 0 , 0 , 0 , 0 },},
 {o_acute                 ,"o\b'"    ,{162/**/,243/**/,198/**/, 0 /*o*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {O_circumflex            ,"O\b^"    ,{ 0      ,212/**/,223/**/, 0 /*O*/,226,226,140,153, 0 , 0 , 0 },},
 {o_circumflex            ,"o\b^"    ,{147/**/,244/**/,194/**/, 0 /*o*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {O_diaeresis             ,"O\b\""   ,{153/**/,214/**/,218/**/,210/**/, 0 , 0 , 0 , 0 , 0 , 0 ,178},},
 {o_diaeresis             ,"o\b\""   ,{148/**/,246/**/,206/**/,215/**/, 0 , 0 , 0 , 0 , 0 , 0 ,183},},
 {O_E                     ,"OE"      ,{ 0      ,215     , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {o_e                     ,"oe"      ,{ 0      ,247     , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {O_grave                 ,"O\b`"    ,{ 0      ,210/**/,223/**/, 0 /*O*/,227, 0 ,169, 0 , 0 , 0 , 0 },},
 {o_grave                 ,"o\b`"    ,{149/**/,242/**/,202/**/, 0 /*o*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {O_slash                 ,"O\b/"    ,{237/**/,215/**/,210/**/, 0 /*O*/,157, 0 , 0 , 0 ,157,157, 0 },},
 {o_slash                 ,"o\b/"    ,{ 0      ,247/**/,214/**/,178/**/,155, 0 , 0 , 0 ,155,155, 0 },},
 {O_tilde                 ,"O\b~"    ,{ 0      ,213/**/,233/**/, 0 /*O*/,229, 0 ,153, 0 , 0 ,166, 0 },},
 {o_tilde                 ,"o\b~"    ,{ 0      ,245/**/,234/**/, 0 /*o*/,228, 0 ,148, 0 , 0 ,167, 0 },},
 {o_underlined            ,"o\b_"    ,{167/**/,186/**/,250/**/, 0 /*o*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {R_acute                 ,"R\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,232, 0 , 0 , 0 , 0 , 0 },},
 {r_acute                 ,"r\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,234, 0 , 0 , 0 , 0 , 0 },},
 {R_chz                   ,"R\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,252, 0 , 0 , 0 , 0 , 0 },},
 {r_chz                   ,"r\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,253, 0 , 0 , 0 , 0 , 0 },},

 {S_acute                 ,"S\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,151, 0 , 0 , 0 , 0 , 0 },},
 {s_acute                 ,"s\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,152, 0 , 0 , 0 , 0 , 0 },},
 {S_cedilla               ,"S\b,"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {s_cedilla               ,"s\b,"    ,{ 0      , 0      , 0      , 0      , 0 ,173, 0 , 0 , 0 , 0 , 0 },},
 {S_tail                  ,"S\b."    ,{ 0      , 0      , 0      , 0      , 0 ,184, 0 , 0 , 0 , 0 , 0 },},
 {s_tail                  ,"s\b."    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {S_circumflex            ,"S\b^"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {s_circumflex            ,"s\b^"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {S_breve                 ,"S\bu"    ,{ 0      , 0      ,235/**/, 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {s_breve                 ,"s\bu"    ,{ 0      , 0      ,236/**/, 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {S_chz                   ,"S\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,230, 0 , 0 , 0 , 0 , 0 },},
 {s_chz                   ,"s\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,231, 0 , 0 , 0 , 0 , 0 },},

 {T_cedilla               ,"T\b,"    ,{ 0      , 0      , 0      , 0      , 0 ,221, 0 , 0 , 0 , 0 , 0 },},
 {t_cedilla               ,"t\b,"    ,{ 0      , 0      , 0      , 0      , 0 ,238, 0 , 0 , 0 , 0 , 0 },},
 {T_chz                   ,"T\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,155, 0 , 0 , 0 , 0 , 0 },},
 {t_chz                   ,"t\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,156, 0 , 0 , 0 , 0 , 0 },},

 {U_acute                 ,"U\b'"    ,{ 0      ,218/**/,237/**/, 0 /*U*/,233,223,150,151, 0 , 0 , 0 },},
 {u_acute                 ,"u\b'"    ,{163/**/,250/**/,199/**/,220/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {U_acuteacute            ,"U\b*"    ,{ 0      , 0      , 0      , 0      , 0 ,235, 0 , 0 , 0 , 0 , 0 },},
 {u_acuteacute            ,"u\b*"    ,{ 0      , 0      , 0      , 0      , 0 ,251, 0 , 0 , 0 , 0 , 0 },},
 {U_breve                 ,"U\bu"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {u_breve                 ,"u\bu"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {U_circle                ,"U\b@"    ,{ 0      , 0      , 0      , 0      , 0 ,222, 0 , 0 , 0 , 0 , 0 },},
 {u_circle                ,"u\b@"    ,{ 0      , 0      , 0      , 0      , 0 ,133, 0 , 0 , 0 , 0 , 0 },},
 {U_circumflex            ,"U\b^"    ,{ 0      ,219/**/,174/**/, 0 /*U*/,234, 0 , 0 ,158, 0 , 0 , 0 },},
 {u_circumflex            ,"u\b^"    ,{150/**/,251/**/,195/**/, 0 /*u*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {U_diaeresis             ,"U\b\""   ,{154/**/,220/**/,219/**/,211/**/, 0 , 0 , 0 , 0 , 0 , 0 ,179},},
 {u_diaeresis             ,"u\b\""   ,{129/**/,252/**/,207/**/,216/**/, 0 , 0 , 0 , 0 , 0 , 0 ,184},},
 {U_grave                 ,"U\b`"    ,{ 0      ,217/**/,173/**/, 0 /*U*/,235, 0 ,157,157, 0 , 0 , 0 },},
 {u_grave                 ,"u\b`"    ,{151/**/,249/**/,203/**/, 0 /*u*/, 0 , 0 , 0 , 0 , 0 , 0 ,188},},

 {Y_acute                 ,"Y\b'"    ,{ 0      , 0      ,177     , 0      ,237,237, 0 , 0 , 0 , 0 , 0 },},
 {y_acute                 ,"y\b'"    ,{ 0      , 0      ,178     , 0      ,236,236, 0 , 0 , 0 , 0 , 0 },},
 {Y_diaeresis             ,"Y\b\""   ,{ 0      ,221/**/,238/**/, 0 /*Y*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {y_diaeresis             ,"y\b\""   ,{152/**/,253/**/,239/**/, 0 /*y*/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {Z_acute                 ,"Z\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,141, 0 , 0 , 0 , 0 , 0 },},
 {z_acute                 ,"z\b'"    ,{ 0      , 0      , 0      , 0      , 0 ,171, 0 , 0 , 0 , 0 , 0 },},
 {Z_chz                   ,"Z\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,166, 0 , 0 , 0 , 0 , 0 },},
 {z_chz                   ,"z\bv"    ,{ 0      , 0      , 0      , 0      , 0 ,167, 0 , 0 , 0 , 0 , 0 },},
 {Z_plus                  ,"Z\b+"    ,{ 0      , 0      , 0      , 0      , 0 ,189, 0 , 0 , 0 , 0 , 0 },},
 {z_plus                  ,"z\b+"    ,{ 0      , 0      , 0      , 0      , 0 ,190, 0 , 0 , 0 , 0 , 0 },},

/* Diacritical marks */

 {DM_macron               ," "       ,{ 0      ,175     ,175     , 0      ,238, 0 , 0 ,167, 0 , 0 , 0 },},
 {DM_center_bar           ,"-"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {DM_under_bar            ,"_"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {DM_circumflex           ,"^"       ,{'^'     , 0      ,170     , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {DM_breve                ," "       ,{ 0      , 0      , 0      , 0      , 0 ,244, 0 , 0 , 0 , 0 , 0 },},
 {DM_chz                  ," "       ,{ 0      , 0      , 0      , 0      , 0 ,243, 0 , 0 , 0 , 0 , 0 },},
 {DM_tilde                ,"~"       ,{'~'     , 0      ,172     , 0      , 0 , 0 , 0 , 0 , 0 , 0 ,181},},
 {DM_diaeresis            ,"\""      ,{ 0      , 0      ,171     , 0      ,247,249, 0 ,164, 0 , 0 ,190},},
 {DM_cedilla              ,","       ,{ 0      ,184     , 0      , 0      ,247,247, 0 ,165, 0 , 0 , 0 },},
 {DM_tail                 ,"."       ,{ 0      , 0      , 0      , 0      , 0 ,242, 0 , 0 , 0 , 0 , 0 },},
 {DM_acute                ,"'"       ,{ 0      ,180     ,168     ,199     ,239, 0 , 0 ,161, 0 , 0 ,160},},
 {DM_grave                ,"`"       ,{ 0      , 0      ,169     , 0      , 0 , 0 , 0 , 0 , 0 , 0 ,164},},
 {DM_virgule              ,"/"       ,{'/'     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {DM_parallels            ," "       ,{186/**/, 0      , 0      ,207     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {DM_circle               ," "       ,{248/**/, 0      ,179     , 0      , 0 ,248, 0 , 0 , 0 , 0 ,166},},
 {DM_plus                 ," "       ,{ 0      , 0      , 0      , 0      , 0 ,250, 0 , 0 , 0 , 0 , 0 },},
 {DM_acuteacute           ,"\""      ,{ 0      , 0      , 0      , 0      , 0 ,235, 0 , 0 , 0 , 0 , 0 },},

/* Text symbols */
 {SYM_daggar              ,"+"       ,{ 0      , 0      , 0      ,200     , 0 , 0 , 0 , 0 , 0 , 0 ,168},},
 {SYM_double_daggar       ,"++"      ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_paragraph           ,"]P"      ,{ 0      ,182/**/,224     , 0      ,244, 0 , 0 ,134, 0 , 0 ,175},},
 {SYM_section             ,"S\bo"    ,{ 21     ,167     ,189     ,201     ,245,245, 0 ,143, 0 , 0 ,169},},
 {SYM_big_asterisk        ,"*"       ,{ 15     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_house               ,"^\b_"    ,{127/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

/* Buisness */
 {SYM_copyright           ,"(c)"     ,{ 0      ,169/**/, 0      ,203     ,184, 0 , 0 , 0 , 0 , 0 ,171},},
 {SYM_registered          ,"(R)"     ,{ 0      ,174/**/, 0      , 0      ,169, 0 , 0 , 0 , 0 , 0 ,170},},
 {SYM_pounds              ,"#"       ,{156/**/,163     ,187     ,195     , 0 , 0 , 0 , 0 , 0 , 0 ,163},},
 {SYM_yen                 ,"Y\b-"    ,{157/**/,165     ,188     ,208     ,190, 0 , 0 , 0 , 0 , 0 ,176},},
 {SYM_cents               ,"c\b|"    ,{155/**/,162     ,191     ,212     ,169, 0 , 0 , 0 , 0 , 0 ,180},},
 {SYM_trade_mark          ,"(TM)"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 ,186},},
 {SYM_francs              ,"f"       ,{159/**/, 0      ,190     ,223     , 0 , 0 , 0 , 0 , 0 , 0 ,191},},
 {SYM_Tx                  ,"Tx"      ,{ 0      , 0      , 0      ,176     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_Pt                  ,"Pt"      ,{158/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_degrees_C           ,"C"       ,{ 0      , 0      , 0      ,180     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_degrees_F           ,"F"       ,{ 0      , 0      , 0      ,181     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_degrees             ,"'\b`"    ,{248/**/,176     ,179     ,198     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_X_circle            ,"X\bO"    ,{ 0      ,168     ,186     , 0      ,207,207, 0 ,152,175,175, 0 },},

/* Punctuation */
 {SYM_UD_question         ,"?"       ,{168/**/,191     ,185     , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_UD_exclamation      ,"!"       ,{ 0      ,161     ,184     , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_double_exclamation  ,"!!"      ,{ 19     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_french_begin_quote  ,"<<"      ,{174/**/,171     ,251     , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_french_end_quote    ,">>"      ,{175/**/,187     ,253     , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_bar_left            ,"-"       ,{169/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_bar_right           ,"-"       ,{170/**/,172     , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_split_vert_bar      ,"|"       ,{ 0      ,166     , 0      , 0      ,221, 0 , 0 ,160, 0 , 0 , 0 },},
 {SYM_vert_bar            ,"|"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

/* directional */

 {SYM_arrow_left          ,"<-"      ,{ 27     , 0      , 0      ,166     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_arrow_right         ,"->"      ,{ 26     , 0      , 0      ,167     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_arrow_leftright     ,"<->"     ,{ 29     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_arrow_down          ,"|\bv"    ,{ 25     , 0      , 0      ,165     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_arrow_up            ,"|\b^"    ,{ 24     , 0      , 0      ,164     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_arrow_updown        ,"^\b|\bv" ,{ 18     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_arrow_updown_under  ,"^\b|\bv\b_" ,{ 23     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_left       ,"<|"      ,{  0     , 0      , 0      ,172     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_right      ,"|>"      ,{  0     , 0      , 0      ,171     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_up         ,"^\b_"    ,{  0     , 0      , 0      ,169     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_down       ,"v\b-"    ,{  0     , 0      , 0      ,170     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_left_solid ,"<\b+|"   ,{ 17     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_right_solid,"|+\b>"   ,{ 16     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_up_solid   ,"^\b_\b+" ,{ 30     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_triangle_down_solid ,"v\b-\b+" ,{ 31     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {TRIANGLE_ULC            ," "       ,{ 0      , 0      , 0      ,251     , 0 , 0 , 0 , 0 , 0 , 0 ,251},},
 {TRIANGLE_LLC            ," "       ,{ 0      , 0      , 0      ,254     , 0 , 0 , 0 , 0 , 0 , 0 ,254},},
 {TRIANGLE_URC            ," "       ,{ 0      , 0      , 0      ,253     , 0 , 0 , 0 , 0 , 0 , 0 ,253},},
 {TRIANGLE_LRC            ," "       ,{ 0      , 0      , 0      ,252     , 0 , 0 , 0 , 0 , 0 , 0 ,252},},

/* Graphics */

 {SYM_bar                 ,"-"       ,{ 0      ,173/**/,246/**/, 0      ,240,240, 0 , 0 , 0 , 0 , 0 },},
 {SYM_diamond_open        ,"^\bv"    ,{ 0      , 0      , 0      ,173     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_square              ,"#"       ,{ 0      , 0      , 0      ,175     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_filled_square       ,"#\b+"    ,{254/**/, 0      ,252     , 0      , 0 ,254, 0 , 0 , 0 , 0 , 0 },},
 {SYM_small_circle        ,"o"       ,{  9     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_small_circle_rvrsd  ,"o"       ,{  8     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_circle              ,"O"       ,{ 0      , 0      , 0      ,168     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_filled_circle       ,"O\b+"    ,{  7     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_largedot_rvrsd      ,"."       ,{  8     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_largedot            ,"."       ,{  7     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_cdot                ,"."       ,{249/**/,183     ,242     , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_cpoint              ,"."       ,{250/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {SYM_smiley_white        ,"O"       ,{  1     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_smiley_black        ,"O"       ,{  2     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_heart               ," "       ,{  3     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_diamond             ,"^\bv\b+" ,{  4     , 0      , 0      ,174     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_club                ," "       ,{  5     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_spade               ," "       ,{  6     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_male                ,"o\b'"    ,{ 11     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_female              ,"o\b+"    ,{ 12     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

/* 12  Quarter Block graphics
   34                         */

 {QBG_1,                   " "       ,{ 0      , 0      , 0      ,225     , 0 , 0 , 0 , 0 , 0 , 0 ,225},},
 {QBG_2,                   " "       ,{ 0      , 0      , 0      ,227     , 0 , 0 , 0 , 0 , 0 , 0 ,227},},
 {QBG_3,                   " "       ,{ 0      , 0      , 0      ,226     , 0 , 0 , 0 , 0 , 0 , 0 ,226},},
 {QBG_4,                   " "       ,{ 0      , 0      , 0      ,228     , 0 , 0 , 0 , 0 , 0 , 0 ,228},},
 {QBG_12,                  " "       ,{223/**/, 0      , 0      ,231     , 0 , 0 , 0 , 0 , 0 , 0 ,231},},
 {QBG_13,                  " "       ,{221/**/, 0      , 0      ,233     , 0 , 0 , 0 , 0 , 0 , 0 ,233},},
 {QBG_14,                  " "       ,{ 0      , 0      , 0      ,229     , 0 , 0 , 0 , 0 , 0 , 0 ,229},},
 {QBG_23,                  " "       ,{ 0      , 0      , 0      ,230     , 0 , 0 , 0 , 0 , 0 , 0 ,230},},
 {QBG_24,                  " "       ,{222/**/, 0      , 0      ,234     , 0 , 0 , 0 , 0 , 0 , 0 ,234},},
 {QBG_34,                  " "       ,{220/**/, 0      , 0      ,232     , 0 , 0 , 0 , 0 , 0 , 0 ,232},},
 {QBG_123,                 " "       ,{ 0      , 0      , 0      ,235     , 0 , 0 , 0 , 0 , 0 , 0 ,235},},
 {QBG_134,                 " "       ,{ 0      , 0      , 0      ,237     , 0 , 0 , 0 , 0 , 0 , 0 ,237},},
 {QBG_124,                 " "       ,{ 0      , 0      , 0      ,236     , 0 , 0 , 0 , 0 , 0 , 0 ,236},},
 {QBG_234,                 " "       ,{ 0      , 0      , 0      ,238     , 0 , 0 , 0 , 0 , 0 , 0 ,238},},
 {QBG_1234,                " "       ,{219/**/, 0      , 0      ,239     , 0 , 0 , 0 , 0 , 0 , 0 ,239},},

/* circle quarters */

 {CQ_1,                    " "       ,{ 0      , 0      , 0      ,163/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {CQ_2,                    " "       ,{ 0      , 0      , 0      ,161/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {CQ_3,                    " "       ,{ 0      , 0      , 0      ,162/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {CQ_4,                    " "       ,{ 0      , 0      , 0      ,160/**/, 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

/* checker board */

 {CB_low,                  " "       ,{176/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {CB_med,                  " "       ,{177/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {CB_high,                 " "       ,{178/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},


/* Line graphics */
/* V   vertical
   H   Horizontal
   T   Tee
   B   Bar
   C   Corner
   I   Intersection
   L   Left
   R   Right
   U   Up
   D   Down
   s   single
   d   double
*/

 {LG_sVB,                  "|"       ,{179/**/, 0      , 0      ,245     , 0 , 0 , 0 , 0 , 0 , 0 ,245},},
 {LG_sLT,                  "+"       ,{180/**/, 0      , 0      ,249     , 0 , 0 , 0 , 0 , 0 , 0 ,249},},
 {LG_dLsT,                 "+"       ,{181/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sLdT,                 "+"       ,{182/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sLdDC,                "+"       ,{183/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dLsDC,                "+"       ,{184/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dLT,                  "+"       ,{185/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dVB,                  "|"       ,{186/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dLDC,                 "+"       ,{187/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dLUC,                 "+"       ,{188/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sLdUC,                "+"       ,{189/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dLsUC,                "+"       ,{190/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sLDC,                 "+"       ,{191/**/, 0      , 0      ,242     , 0 , 0 , 0 , 0 , 0 , 0 ,242},},
 {LG_sUsRC,                "+"       ,{192/**/, 0      , 0      ,246     , 0 , 0 , 0 , 0 , 0 , 0 ,246},},
 {LG_sUT,                  "+"       ,{193/**/, 0      , 0      ,248     , 0 , 0 , 0 , 0 , 0 , 0 ,248},},
 {LG_sDT,                  "+"       ,{194/**/, 0      , 0      ,243     , 0 , 0 , 0 , 0 , 0 , 0 ,243},},
 {LG_sRT,                  "+"       ,{195/**/, 0      , 0      ,244     , 0 , 0 , 0 , 0 , 0 , 0 ,244},},
 {LG_sHB,                  "-"       ,{196/**/, 0      , 0      ,241     , 0 , 0 , 0 , 0 , 0 , 0 ,241},},
 {LG_sUDLRI,               "+"       ,{197/**/, 0      , 0      ,250     , 0 , 0 , 0 , 0 , 0 , 0 ,250},},
 {LG_dRsT,                 "+"       ,{198/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sRdT,                 "+"       ,{199/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dURC,                 "+"       ,{200/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dDRC,                 "+"       ,{201/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dUT,                  "+"       ,{202/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dDT,                  "+"       ,{203/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dRT,                  "+"       ,{204/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dHB,                  "-"       ,{205/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dUDLRI,               "+"       ,{206/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sUdT,                 "+"       ,{207/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dUsT,                 "+"       ,{208/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sDdT,                 "+"       ,{209/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dDsT,                 "+"       ,{210/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dUsRC,                "+"       ,{211/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sUdRC,                "+"       ,{212/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sDdRC,                "+"       ,{213/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dDsRC,                "+"       ,{214/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_dUDsLRI,              "+"       ,{215/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sUDdLRI,              "+"       ,{216/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {LG_sULC,                 "+"       ,{217/**/, 0      , 0      ,247     , 0 , 0 , 0 , 0 , 0 , 0 ,247},},
 {LG_sDRC,                 "+"       ,{218/**/, 0      , 0      ,240     , 0 , 0 , 0 , 0 , 0 , 0 ,240},},

/* Math */

 {SYM_one_half            ,"1/2"     ,{171/**/,189     ,248     ,206     , 0 , 0 , 0 , 0 , 0 , 0 ,174},},
 {SYM_one_quarter         ,"1/4"     ,{172/**/,188     ,247     ,204     , 0 , 0 , 0 , 0 , 0 , 0 ,172},},
 {SYM_three_quarters      ,"3/4"     ,{ 0      ,190     ,245     , 0      ,243, 0 , 0 ,173, 0 , 0 ,173},},
 {SYM_exponent_1          ,"1"       ,{ 0      ,185     , 0      , 0      ,251, 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_exponent_2          ,"2"       ,{253/**/,178     , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_exponent_3          ,"3"       ,{ 0      ,179     , 0      , 0      ,252, 0 , 0 ,166, 0 ,174, 0 },},
 {SYM_exponent_n          ,"n"       ,{252/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_GTE                 ,">\b_"    ,{242/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_LTE                 ,"<\b_"    ,{243/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_plus_minus          ,"+\b_"    ,{241/**/,177     ,254     ,188     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_infinity            ,"infinity",{236/**/, 0      , 0      ,186     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_equivelence         ,"=\b_"    ,{240/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_intersection        ," "       ,{239/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_union               ,"U"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_approximate         ,"~\b="    ,{247/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_root                ,"^/"      ,{251/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_times               ,"x"       ,{ 0      , 0      , 0      ,190     ,158,158, 0 , 0 , 0 , 0 , 0 },},
 {SYM_overline_x          ,"x\b-"    ,{ 0      , 0      , 0      ,205     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_division            ,"-\b:"    ,{246/**/, 0      , 0      ,191     , 0 ,246, 0 , 0 , 0 , 0 , 0 },},

/* Music */

 {SYM_quaver             ,".\b|\b`"  ,{ 13     , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {SYM_quaverquaver       ,".\b|=.\b:",{ 14/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

/* Greek */

 {GRK_alpha               ," "       ,{224/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_beta                ,"b"       ,{225/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_gamma               ," "       ,{231/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_delta               ,"d"       ,{235/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_epsilon             ," "       ,{238/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_zeta                ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_eta                 ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_theta               ,"o\b-"    ,{233/**/, 0      , 0      ,179     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_iota                ,"i"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_kappa               ,"k"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_lambda              ,"l"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_mu                  ," "       ,{230/**/,181     ,243     ,197     , 0 , 0 , 0 , 0 , 0 , 0 ,165},},
 {GRK_nu                  ,"v"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_xi                  ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_pi                  ," "       ,{210/**/, 0      , 0      ,187     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_omicron             ,"o"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_rho                 ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_sigma               ,"o\b'"    ,{229/**/, 0      , 0      ,186     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_tau                 ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_upsilon             ,"u"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_phi                 ,"o\b/"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_chi                 ,"x"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_psi                 ,"v\b/"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_omega               ,"w"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {GRK_Alpha               ,"A"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Beta                ,"B"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Gamma               ," "       ,{226/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Delta               ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Epsilon             ,"E"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Zeta                ,"Z"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Eta                 ,"H"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Theta               ,"O\b-"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Iota                ,"I"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Kappa               ,"K"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Lambda              ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Mu                  ,"M"       ,{ 0      , 0      , 0      , 0      ,230, 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Nu                  ,"N"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Xi                  ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Pi                  ," "       ,{227/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Omicron             ,"O"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Rho                 ,"P\b-"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Sigma               ,">\b_\b~" ,{228/**/, 0      , 0      ,184     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Tau                 ,"T"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Upsilon             ,"I\b~"    ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Phi                 ,"0\bI"    ,{232/**/, 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Chi                 ,"X"       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Psi                 ," "       ,{ 0      , 0      , 0      , 0      , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {GRK_Omega               ,"O\b_"    ,{234/**/, 0      , 0      ,182     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},

 {GRK_UDOmega             ,"O\b~"    ,{234/**/, 0      , 0      ,183     , 0 , 0 , 0 , 0 , 0 , 0 , 0 },},
 {UCSCU_END_TABLE         ,""        ,{0       ,0       ,0       ,0       ,0  ,0  ,0  ,0  ,0  ,0  ,0  },}
};



#if defined(__STDC__) || defined(__cplusplus)
int code_used_in(int code,int CS)
#else
int code_used_in(code,CS)
int code;
int CS;
#endif
{
  int i;
  int used = 0;
  for (i = 0; bit8_table[i].symbol; i++)
     used = (bit8_table[i].codes[CS] == code) ? 1 : used;
  return(used);
};

#if defined(__STDC__) || defined(__cplusplus)
void copy_CS(int source,int target)
#else
void copy_CS(source,target)
int source;
int target;
#endif
{
  int i;
  for (i = 0; bit8_table[i].symbol; i++)
     if (bit8_table[i].codes[target] == 0)
        if (!code_used_in(bit8_table[i].codes[source],target))
           bit8_table[i].codes[target] = bit8_table[i].codes[source];
};

/* A null string indicates the character is not available in the character set */
/* see if the character can be drawn with a multiple character sequence. */

struct bit7_layout
{
  int  symbol;
  char_8bit codes[13];
};


struct bit7_layout bit7_table[] =
{
/*                   USA UK  GRM DUT FRN SWD NRW ITL SPN PRT FIN BEL SWS*/
 {A_diaeresis       ,{0  ,0  ,'[',0  ,0  ,0  ,'[',0  ,0  ,0  ,'[',0  ,0  },},
 {a_diaeresis       ,{0  ,0  ,'{',0  ,0  ,'{',0  ,0  ,0  ,0  ,'{',0  ,'{'},},
 {A_circle          ,{0  ,0  ,0  ,']',0  ,0  ,']',0  ,']',0  ,']',0  ,0  },},
 {a_circle          ,{0  ,0  ,0  ,'}',0  ,'}','}',0  ,0  ,0  ,'}',0  ,0  },},
 {A_E               ,{0  ,0  ,0  ,'[',0  ,0  ,0  ,'[',0  ,0  ,0  ,0  ,0  },},
 {a_e               ,{0  ,0  ,0  ,'{',0  ,0  ,'{',0  ,0  ,0  ,0  ,0  ,0  },},
 {a_grave           ,{0  ,0  ,0  ,0  ,'@',0  ,0  ,'{',0  ,0  ,0  ,'@','@'},},
 {A_tilde           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'[',0  ,0  ,0  },},
 {a_tilde           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'{',0  ,0  ,0  },},

 {B_german_S        ,{0  ,0  ,'~',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},

 {C_cedilla         ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,92 ,0  ,0  ,92 },},
 {c_cedilla         ,{0  ,0  ,0  ,0  ,92 ,0  ,0  ,92 ,'}','|',0  ,92 ,0  },},

 {E_acute           ,{0  ,0  ,0  ,0  ,0  ,'@',0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {e_acute           ,{0  ,0  ,0  ,0  ,'{','`',0  ,']',0  ,0  ,'`','{','['},},
 {e_circumflex      ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,']'},},
 {e_grave           ,{0  ,0  ,0  ,0  ,'}',0  ,0  ,'}',0  ,0  ,0  ,'}','_'},},

 {i_circumflex      ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'^'},},
 {i_grave           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,'~',0  ,0  ,0  ,0  ,0  },},

 {N_tilde           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,92 ,0  ,0  ,0  ,0  },},
 {n_tilde           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'|',0  ,0  ,0  ,0  },},

 {o_circumflex      ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'`'},},
 {O_diaeresis       ,{0  ,0  ,92 ,0  ,0  ,92 ,0  ,0  ,0  ,0  ,92 ,0  ,0  },},
 {o_diaeresis       ,{0  ,0  ,'|',0  ,0  ,'|',0  ,0  ,0  ,0  ,'|',0  ,'|'},},
 {o_grave           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,'|',0  ,0  ,0  ,0  ,0  },},
 {O_slash           ,{0  ,0  ,0  ,92 ,0  ,92 ,92 ,0  ,0  ,0  ,0  ,0  ,0  },},
 {o_slash           ,{0  ,0  ,0  ,92 ,0  ,'|','|',0  ,0  ,0  ,0  ,0  ,0  },},
 {O_tilde           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,']',0  ,0  ,0  },},
 {o_tilde           ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'}',0  ,0  ,0  },},

 {i_circumflex      ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'~'},},
 {U_diaeresis       ,{0  ,0  ,']',0  ,0  ,'^',0  ,0  ,0  ,0  ,'^',0  ,0  },},
 {u_diaeresis       ,{0  ,0  ,'}',0  ,0  ,'~',0  ,0  ,0  ,0  ,'~',0  ,'}'},},
 {u_grave           ,{0  ,0  ,0  ,0  ,'|',0  ,0  ,'`',0  ,0  ,0  ,'|','#'},},

/* Symbols which differ from ASCII representation */
/*                    USA UK  GRM DUT FRN SWD NRW ITL SPN PRT FIN BEL SWS */
 {SYM_pounds        ,{0  ,'#',0  ,'#',0  ,0  ,0  ,'#','#',0  ,0  ,'#',0  },},
 {SYM_X_circle      ,{0  ,0  ,0  ,0  ,0  ,'$',0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_section       ,{0  ,0  ,'@',0  ,0  ,0  ,0  ,'@','@',0  ,0  ,'}',0  },},
 {SYM_cdot          ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_vert_bar      ,{0  ,0  ,0  ,']',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_one_half      ,{0  ,0  ,0  ,92 ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_three_quarters,{0  ,0  ,0  ,'@',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {i_j               ,{0  ,0  ,0  ,'[',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_francs        ,{0  ,0  ,0  ,'|',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_one_quarter   ,{0  ,0  ,0  ,'}',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {SYM_UD_question   ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,']',0  ,0  ,0  ,0  },},
 {SYM_UD_exclamation,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'[',0  ,0  ,0  ,0  },},


/* Diacritical marks that differ from ASCII representation */

/*                   USA UK  GRM DUT FRN SWD NRW ITL SPN PRT FIN BEL SWS */
 {DM_circumflex    ,{0  ,0  ,0  ,'^',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {DM_tilde         ,{0  ,0  ,0  ,'~',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {DM_diaeresis     ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,'~',0  },},
 {DM_acute         ,{0  ,0  ,0  ,'~',0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },},
 {DM_circle        ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,'[','{',0  ,0  ,'[',0  },},
 {UCSCU_END_TABLE  ,{0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  },}
};


#if defined(__STDC__) || defined(__cplusplus)
int lookup_char_8bit(char_8bit code,int cs_8bit)
#else
int lookup_char_8bit(code,cs_8bit)
char_8bit code;
int cs_8bit;
#endif
{
  int i;

  for (i = 0; bit8_table[i].symbol ;i++)
  {
    if (bit8_table[i].codes[cs_8bit] == code)
       return(bit8_table[i].symbol);
  };
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
int lookup_char_7bit(char_8bit code,int cs_7bit)
#else
int lookup_char_7bit(code,cs_7bit)
char_8bit code;
int cs_7bit;
#endif
{
  int i;

  for (i = 0; bit7_table[i].symbol ;i++)
    if (bit7_table[i].codes[cs_7bit] == code)
       return(bit7_table[i].symbol);
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
char *lookup_code_ASCII(int symbol)
#else
char *lookup_code_ASCII(symbol)
int symbol;
#endif
{
  int i;
  for (i = 0; bit8_table[i].symbol; i++)
     if (bit8_table[i].symbol == symbol)
        return(bit8_table[i].ASCII);
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
char *lookup_code_TeX(int symbol)
#else
char *lookup_code_TeX(symbol)
int symbol;
#endif
{
  int i;
  for (i = 0; UCSCU_TeX_table[i].symbol; i++)
     if (bit8_table[i].symbol == symbol)
        return(UCSCU_TeX_table[i].TeX);
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
char *UCSCU_lookup_code_HTML(int symbol)
#else
char *UCSCU_lookup_code_HTML(symbol)
int symbol;
#endif
{
  int i;
  for (i = 0; UCSCU_HTML_table[i].symbol; i++)
     if (bit8_table[i].symbol == symbol)
        return(UCSCU_HTML_table[i].HTML);
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
char_8bit lookup_code_7bit(int symbol,int CS)
#else
char_8bit lookup_code_7bit(symbol,CS)
int symbol;
int CS;
#endif
{
  int i;

  for (i = 0;bit7_table[i].symbol != UCSCU_END_TABLE;i++)
     if (bit7_table[i].symbol == symbol)
        return(bit7_table[i].codes[CS]);
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
char_8bit lookup_code_8bit(int symbol,int CS)
#else
char_8bit lookup_code_8bit(symbol,CS)
int symbol;
int CS;
#endif
{
  int i;

  if (CS == CS_NONE) return(0);
  if (symbol < UCSCU_END_TABLE) return(0);
  for (i = 0;bit8_table[i].symbol;i++)
     if (bit8_table[i].symbol == symbol)
        return(bit8_table[i].codes[CS]);
  return(0);
};

#if defined(__STDC__) || defined(__cplusplus)
int lookup_diacritical_mark(char_8bit ch,int bit7_cs,int bit8_cs)
#else
int lookup_diacritical_mark(ch,bit7_cs,bit8_cs)
char_8bit ch;
int bit7_cs;
int bit8_cs;
#endif
{
   int DC_mark = 0;

   switch (ch)
   {
    case 39  : DC_mark = DM_acute       ; break;
    case '`' : DC_mark = DM_grave       ; break;
    case '~' : DC_mark = DM_tilde       ; break;
    case '^' : DC_mark = DM_circumflex  ; break;
    case '@' : DC_mark = DM_circle      ; break;
    case '/' : DC_mark = DM_virgule     ; break;
    case '=' : DC_mark = DM_macron      ; break;
    case '-' : DC_mark = DM_center_bar  ; break;
    case '_' : DC_mark = DM_under_bar   ; break;
    case '"' : DC_mark = DM_diaeresis   ; break;
    case 'u' : DC_mark = DM_breve       ; break;
    case 'v' : DC_mark = DM_chz         ; break;
    case '*' : DC_mark = DM_parallels   ; break;
    case ',' : DC_mark = DM_cedilla     ; break;
    case '.' : DC_mark = DM_tail        ; break;
    case '+' : DC_mark = DM_plus        ; break;
   };

   if (DC_mark)
     if (UCSCU_output_mode.prefer_7bit)                                                      /* Then look up 7-bit code first */
     {
        if ((ch = lookup_code_7bit(DC_mark,bit7_cs)) == 0)     /*   if no 7bit representation */
           ch = lookup_code_8bit(DC_mark,bit8_cs);             /*   if no 8bit representation */
     }
     else                                                                         /* otherwise lookup 8-bit code first */
        if ((ch = lookup_code_8bit(DC_mark,bit8_cs)) == 0)
           ch = lookup_code_7bit(DC_mark,bit7_cs);

   if (ch)
      return(ch);
   else
   {
     char *ASCII_rep = lookup_code_ASCII(DC_mark);
     if (ASCII_rep)
        return(ASCII_rep[0]);
     else
        return(' ');  /* If no ASCII representation of the diacritical mark,
                         then print nothing, but since BS already entered*/
   };
};

#if defined(__STDC__) || defined(__cplusplus)
char_8bit *get_string_representation(int code, char *str,int backspacing,int graphics,int is_graphic,char_8bit *buffer)
#else
char_8bit *get_string_representation(code,str,backspacing,graphics,is_graphic,buffer)
int code;
char *str;
int backspacing;
int graphics;
int is_graphic;
char_8bit *buffer;
#endif
{
   int i;
   int buf_pos = 0;
   buffer[0] = 0;
   if (is_graphic)
      if (graphics)
        buffer[0] = ((graphics==GR_MARK) ? UNKNOWN_MARKER : ' ');
      else
        strcpy((char *)buffer,(const_char *)str);
   else
      if (((code <= z_plus) && (backspacing == BS_NOT_CAP)) ||
          (backspacing == BS_NEVER))
         buffer[0] = str[0];
      else
         for (i = 0; i < strlen(str);i++)
           if ((str[i] == 8) && (backspacing == BS_OMIT))
           {  /* do nothing: Omit the backspacing */}
           else
              buffer[buf_pos++] = ((i>0)?
                lookup_diacritical_mark(str[i],UCSCU_output_option.bit7_CS,UCSCU_output_option.bit8_CS) :
                str[i]);
   return(buffer);
};

#if defined(__STDC__) || defined(__cplusplus)
char_8bit *get_simple_line_graphic(int code,char_8bit *buffer)
#else
char_8bit *get_simple_line_graphic(code,buffer)
int code;
char_8bit *buffer;
#endif
{
  int i;

  switch (code)
  { case LG_dVB   :                                 code = LG_sVB; break;
    case LG_dLsT  : case LG_sLdT  : case LG_dLT   : code = LG_sLT; break;
    case LG_sLdDC : case LG_dLsDC : case LG_dLDC  : code = LG_sLDC; break;
    case LG_dURC  : case LG_dUsRC : case LG_sUdRC : code = LG_sUsRC; break;
    case LG_dUT   : case LG_sUdT  : case LG_dUsT  : code = LG_sUT; break;
    case LG_dDT   : case LG_sDdT  : case LG_dDsT  : code = LG_sDT; break;
    case LG_dRsT  : case LG_sRdT  : case LG_dRT   : code = LG_sRT; break;
    case LG_dHB   :                                 code = LG_sHB; break;
    case LG_dUDLRI: case LG_dUDsLRI :case LG_sUDdLRI : code = LG_sUDLRI; break;
    case LG_dLUC  : case LG_sLdUC : case LG_dLsUC : code = LG_sULC; break;
    case LG_dDRC  : case LG_sDdRC : case LG_dDsRC : code = LG_sDRC; break;
  };

  for (i = 0; bit8_table[i].symbol != UCSCU_END_TABLE; i++)
    if (bit8_table[i].symbol == code)
    {
       if ((UCSCU_output_option.bit8_CS >= 0) && (bit8_table[i].codes[UCSCU_output_option.bit8_CS]))
          buffer[0] = (bit8_table[i].codes[UCSCU_output_option.bit8_CS]);
       else
          strcpy((char *)buffer,(const_char *)bit8_table[i].ASCII);
       return(buffer);
    };
  return(NULL);
};

/*  Currently we aren't handling multi character representations of
    accented characters on input.

    For example  e' could be taken to mean  e_acute, but so could 'e
    this could cause problems with words such as:

     re'elle  or ree'lle

     or quoted text

     'This is a quote'

    Also, many scandinavians use double lettering for ASCII representations
    of accented characters,  Ie  aa for a_circle.

    This requires more context parsing than I care to deal with at this
    time, since I don't know all possible situations.


#if defined(__STDC__) || defined(__cplusplus)
int check_diacritical_mark(char letter,int mark)
#else
int check_diacritical_mark(letter,mark)
char letter;
int mark;
#endif
{
  return(0);

};
*/

#define is_8bit(x)              (x & 0x80)
#define is_diacritical_mark(x)  ((x >= DM_macron) && (x <= DM_acuteacute))
#define is_line_graphics(x)     ((x >= LG_sVB) && (x <= LG_sDRC))


#if defined(__STDC__) || defined(__cplusplus)
int in_char(void)
#else
int in_char()
#endif
{
  int in;
  in = fgetc(UCSCU_input_option.file);
  return(feof(UCSCU_input_option.file) ? 0 : in);
};

#if defined(__STDC__) || defined(__cplusplus)
int in_symbol(void)
#else
int in_symbol()
#endif
{
  static int at_eof = 0;
  static char_8bit lookahead = 0;
  char_8bit curr = 0;
  int curr_code = 0;
/* Currently not handling ASCII representations of diacritical markings
  int diacritical_code;
  int potential_diacritical_marking = 0;
*/

  if (lookahead)
     curr = lookahead;
  if (at_eof)
     return(0);
  else
    if (!curr)
    {
      curr = fgetc(UCSCU_input_option.file);
      if (feof(UCSCU_input_option.file))
      {		/* fgetc may have returned EOF (-1) */
	at_eof = 1;
	lookahead = 0;
      };
    };


  if (feof(UCSCU_input_option.file))
  {
    at_eof = 1;
    lookahead = 0;
  }
  else
  {
    lookahead = fgetc(UCSCU_input_option.file);
    if (feof(UCSCU_input_option.file))
    {		/* fgetc may have returned EOF (-1) */
      at_eof = 1;
      lookahead = 0;
    }
  };


  if (lookahead == 8)  /* gobble up backspaces */
  {
     lookahead = fgetc(UCSCU_input_option.file);
     if (feof(UCSCU_input_option.file))
     {		/* fgetc may have returned EOF (-1) */
       at_eof = 1;
       lookahead = 0;
     }
   /*potential_diacritical_marking = 1; */
  };

  /* space will cause problems with symbols that have no ASCII representation*/
  if (curr == ' ') return(' ');

  if (is_8bit(curr))
     curr_code = lookup_char_8bit(curr,UCSCU_input_option.bit8_CS);
  else
/*  { */
     curr_code = lookup_char_7bit(curr,UCSCU_input_option.bit7_CS);
/* Could do input of ASCII diacritical representations
     if (curr_code)
       if (is_diacritical_mark(curr_code))
       {
         if ((diacritical_code = check_diacritical_mark(lookahead,curr_code)) != 0)
         {
            curr_code = diacritical_code;
            lookahead = fgetc(UCSCU_input_option.file);
         }
       }
       else
         if ((diacritical_code = check_diacritical_mark(curr,lookup_7bit(lookahead,UCSCU_input_option.cs_7bit))) != 0)
         {
            curr_code = diacritical_code;
            lookahead = fgetc(input.file);
         };
  };
*/
  return(curr_code ? curr_code : curr);
};
#ifndef UCSCU_STANDALONE
#if defined(__STDC__) || defined(__cplusplus)
int str_in_symbol(char *inbuffer, int *buff_pos)
#else
int str_in_symbol(inbuffer,buff_pos)
char *inbuffer;
int *buff_pos;
#endif
{
  char curr;
  int curr_code = 0;

  if (inbuffer[*buff_pos] == '0') return(0);

  curr = inbuffer[*buff_pos];
  *buff_pos += 1;

  if (curr == ' ')
     return(' ');

  if (is_8bit(curr))
     curr_code = lookup_char_8bit(curr,UCSCU_input_option.bit8_CS);
  else
     curr_code = lookup_char_7bit(curr,UCSCU_input_option.bit7_CS);
  return(curr_code ? curr_code : curr);
};
#endif


#if defined(__STDC__) || defined(__cplusplus)
char_8bit *UCSCU_char(int cs_code)
#else
char_8bit *UCSCU_char(cs_code)
int cs_code;
#endif
{
   static char_8bit buffer[20];
   char *outstr = 0;
   char_8bit outchar = 0;
   buffer[0] = 0;
   buffer[1] = 0;

   if (UCSCU_output_option.bit7_CS == CS_TeX)                  /* If generating TeX then we only want 7 bit ASCII */
   {                                                     /* (note that can't do graphics in TeX) */
      if ((outstr = (char_8bit *)lookup_code_TeX(cs_code)) != 0)      /* If TeX symbol or Accented character */
         sprintf((char *)buffer,"\\%s",outstr);                  /*   Output TeX representation */
      else                                               /*   (note that TeX strings are stored without \ */
         if ((outstr = (char_8bit *)lookup_code_ASCII(cs_code)) == 0) /* If code as ASCII representation */
            buffer[0] = cs_code;                         /*   output ASCII representation (Shouldn't have to do this, but I don't know all the TeX symbols and accents */
         else
            strcpy((char *)buffer,(const_char *)outstr);                       /* Otherwise it's a 7-bit ascii character */
   }
   else
   {
     if (UCSCU_output_mode.prefer_7bit)                                                      /* Then look up 7-bit code first */
     {
        if ((outchar = lookup_code_7bit(cs_code,UCSCU_output_option.bit7_CS)) == 0)     /*   if no 7bit representation */
           outchar = lookup_code_8bit(cs_code,UCSCU_output_option.bit8_CS);             /*   if no 8bit representation */
     }
     else                                                                         /* otherwise lookup 8-bit code first */
        if ((outchar = lookup_code_8bit(cs_code,UCSCU_output_option.bit8_CS)) == 0)
           outchar = lookup_code_7bit(cs_code,UCSCU_output_option.bit7_CS);

     if (outchar)                                              /* If a single char representation is found */
        buffer[0] = outchar;                                   /* output character */
     else
        if (is_line_graphics(cs_code) &&                       /* if it is a line graphic */
           !lookup_code_8bit(cs_code,UCSCU_output_option.bit8_CS))   /* but there is no character */
              strcpy((char *)buffer,                                   /* try to output single line character or +-| */
                 (const_char *)get_simple_line_graphic(cs_code,buffer));
        else
           if ((outstr = (char_8bit *)lookup_code_ASCII(cs_code)) == 0)   /* if no ascii representation (symbol not in table*/
              buffer[0] = cs_code;                           /* then this must be a printable 7-bit ascii code */
           else                                              /* otherwise output the string representation */
              strcpy((char *)buffer,
               (const_char *)get_string_representation(cs_code,outstr,UCSCU_output_mode.backspacing,
                  UCSCU_output_mode.graphics,is_line_graphics(cs_code),buffer));
   };
   return(buffer);
};

char_8bit direct_conversion[256][10];


/* Currently we are only accepting character (7 or 8 bit) input,
   so we can always do direct conversion.

   Eventually we will want to be able to input character sequences
   which are ascii/text representations of accented characters and symbols
   (such as the ASCII representation we currenly output or TeX
   representation. In this case we could not do direct_conversion.

*/
#if defined(__STDC__) || defined(__cplusplus)
int setup_direct_conversion(void)
#else
int setup_direct_conversion()
#endif
{
  char_8bit curr = 0;
  int curr_code = 0;
  char_8bit *outstr;
  char_8bit outchar = 0;

  { int i;
    for (i = 0; i < 255; i++)
    {
      direct_conversion[i][0] = i;
      direct_conversion[i][1] = 0;
    };
  }

  for (curr = 1; curr < 255; curr++)
  {
    if (is_8bit(curr))
       curr_code = lookup_char_8bit(curr,UCSCU_input_option.bit8_CS);
    else
       curr_code = lookup_char_7bit(curr,UCSCU_input_option.bit7_CS);

    curr_code = (curr_code ? curr_code : curr);

     if (UCSCU_output_option.bit7_CS == CS_TeX)                  /* If generating TeX then we only want 7 bit ASCII */
     {                                                     /* (note that can't do graphics in TeX) */
        if ((outstr = lookup_code_TeX(curr_code)) != 0)      /* If TeX symbol or Accented character */
           sprintf(direct_conversion[curr],"\\%s",outstr);      /*   Output TeX representation */
        else                                               /*   (note that TeX strings are stored without \ */
           if ((outstr = (char_8bit *)lookup_code_ASCII(curr_code)) == 0) /* If code as ASCII representation */
              direct_conversion[curr][0] = curr_code;
           else
              sprintf(direct_conversion[curr],"%s",outstr);     /* Otherwise it's a 7-bit ascii character */
     }
     else
     {
       if (UCSCU_output_mode.prefer_7bit)                                                      /* Then look up 7-bit code first */
       {
          if ((outchar = lookup_code_7bit(curr_code,UCSCU_output_option.bit7_CS)) == 0)     /*   if no 7bit representation */
             outchar = lookup_code_8bit(curr_code,UCSCU_output_option.bit8_CS);             /*   if no 8bit representation */
       }
       else                                                                         /* otherwise lookup 8-bit code first */
          if ((outchar = lookup_code_8bit(curr_code,UCSCU_output_option.bit8_CS)) == 0)
             outchar = lookup_code_7bit(curr_code,UCSCU_output_option.bit7_CS);

       if (outchar)                                              /* If a single char representation is found */
          direct_conversion[curr][0] = outchar;
       else
          if (is_line_graphics(curr_code) &&                       /* if it is a line graphic */
              !lookup_code_8bit(curr_code,UCSCU_output_option.bit8_CS))  /* but there is no character */
                   get_simple_line_graphic(curr_code,direct_conversion[curr]);             /* try to output single line character or +-| */
          else
             if ((outstr = (char_8bit *)lookup_code_ASCII(curr_code)) == 0)   /* if no ascii representation (symbol not in table*/
                direct_conversion[curr][0] = curr_code;            /* then this must be a printable 7-bit ascii code */
             else                                              /* otherwise output the string representation */
                get_string_representation(curr_code,outstr,UCSCU_output_mode.backspacing,UCSCU_output_mode.graphics,is_line_graphics(curr_code),direct_conversion[curr]);
     };
  };
  return(1);
};

/* Converts the character string.*/
#ifndef UCSCU_STANDALONE
#if defined(__STDC__) || defined(__cplusplus)
char *UCSCU_string(char *instr,char *outstr)
#else
char *UCSCU_string(instr,outstr)
char *instr;
char *outstr;
#endif
{
  int i;
  int curr_char;
  outstr[0] = 0;

  for (i = 0; i < strlen(instr); i++)
  {
    curr_char = UCSCU_do_direct_conversion ? instr[i] :
       str_in_symbol(instr,&i);
    strcat(outstr,(const_char *)UCSCU_do_direct_conversion ?
           direct_conversion[curr_char]: UCSCU_char(curr_char));
  };
  return(outstr);
};
#endif

#if defined(__STDC__) || defined(__cplusplus)
void show_7bit_options(void)
#else
void show_7bit_options()
#endif
{
   printf(" 1. English\n");
   printf(" 2. German\n");
   printf(" 3. Dutch\n");
   printf(" 4. French\n");
   printf(" 5. Swedish\n");
   printf(" 6. Norwegian/Danish\n");
   printf(" 7. Italian\n");
   printf(" 8. Spanish\n");
   printf(" 9. Portuguese\n");
   printf("10. Finnish\n");
   printf("11. Belgian (French/Flemish)\n");
   printf("12. Swiss\n");
   printf("13. TeX/LaTeX (output only)\n\n");
};

#if defined(__STDC__) || defined(__cplusplus)
void show_8bit_options(void)
#else
void show_8bit_options()
#endif
{
   printf(" 0. No 8 bit characters.\n");
   printf(" 1. PC8     For IBM-PC and most IBM-PC printers.\n");
   printf(" 2. MNCS    For multi-national character set on most DEC machines.\n");
   printf("    AMIGA   or the Amiga default character set.\n");
   printf("    ECMA-94 or the ECMA-94 (Latin 1) character set.\n");
   printf(" 3. ROMAN8  For most new printers, including the HP series and compatibles.\n");
   printf(" 4. EPSON8  For most older printers and the Epson series.\n");
   printf(" 5. PC850   MS-DOS PC Multilingual (Latin 1).\n");
   printf(" 6. PC852   MS-DOS PC Slavic (Latin 2).\n");
   printf(" 7. PC860   Portugal MS-DOS code page.\n");
   printf(" 8. PC863   Canadian-French MS-DOS code page.\n");
   printf(" 9. PC865   Nordic (Norway/Denmark) MS-DOS code page.\n");
   printf("10. PCDN    Denmark/Norway HP laser printers.\n");
   printf("11. PRINTER Most other printers with limited 8-bit characters:\n");
   printf("            Toshiba-1340, Qume-Sprint-11, Diablo-630.\n\n");
};


static char input_filename[256];
static char output_filename[256];

void UCSCU_prompt_input_options()
{


  UCSCU_input_option.bit7_CS = -1;
  while ((UCSCU_input_option.bit7_CS <0) || (UCSCU_input_option.bit7_CS > 11))
  {
   printf("\nSelect the 7-Bit ASCII/ISO character set of input file.\n");
   show_7bit_options();
   scanf("%d",&UCSCU_input_option.bit7_CS);
   UCSCU_input_option.bit7_CS -= 1;   /* Because the program ranges 0-8 not 1-9 */
  };

  UCSCU_input_option.bit8_CS = -2;
  while ((UCSCU_input_option.bit8_CS < -1) || (UCSCU_input_option.bit8_CS > 10))
  {
   printf("\nSelect the 8-bit character set of the input file.\n");
   show_8bit_options();
   scanf("%d",&UCSCU_input_option.bit8_CS);
   UCSCU_input_option.bit8_CS -= 1;
  };


};

void UCSCU_prompt_output_options()
{

  UCSCU_output_option.bit7_CS = -1;
  while ((UCSCU_output_option.bit7_CS <0) || (UCSCU_output_option.bit7_CS > 12))
  {
   printf("\nSelect the 7-Bit ASCII/ISO character set of output file.\n");
   show_7bit_options();
   scanf("%d",&UCSCU_output_option.bit7_CS);
   UCSCU_output_option.bit7_CS -= 1;   /* Because the program ranges 0-8 not 1-9 */
  };

  UCSCU_output_option.bit8_CS = -2;
  while ((UCSCU_output_option.bit8_CS < -1) || (UCSCU_output_option.bit8_CS > 10))
  {
   printf("\nSelect the 8-bit character set of the output file.\n");
   show_8bit_options();
   scanf("%d",&UCSCU_output_option.bit8_CS);
   UCSCU_output_option.bit8_CS -= 1;
  };


  UCSCU_output_mode.prefer_7bit = -1;
  while ((UCSCU_output_mode.prefer_7bit < 0) || (UCSCU_output_mode.prefer_7bit > 1))
  {
    printf("\nWhen outputing an accented letter or symbol, or using diacritical marks:\n");
    printf("If the character appears both in the 7-bit and the 8-bit character set,\n");
    printf("should the character from the 7-bit set be used instead of the 8-bit character?\n\n");
    printf("Usually Yes when the output is intended for a printer or line terminal.\n");
    printf("Usually No when the output is intended for a PC or workstation.\n");
    printf("\nEnter 1 for yes, 0 for no.\n\n");
    scanf("%d",&UCSCU_output_mode.prefer_7bit);
  };

  UCSCU_output_mode.backspacing = -1;
  while ((UCSCU_output_mode.backspacing < 0) || (UCSCU_output_mode.backspacing > 3))
  {
    printf("\nWhen an appropriate 7 or 8 bit accented letter is not available the\n");
    printf("respective unaccented letter will be used, additional diacritical marks\n");
    printf("can also be optionally added:\n");
    printf(" 1. Add backspace character and diacritical mark (accent) I.e. A bs '\n");
    printf("    Choose this option for output intended for a printer\n");
    printf(" 2. Only print the unaccented letter.\n");
    printf("    Choose this option for output line terminals or where adding\n");
    printf("    additional characters would mess up formatted output.\n");
    printf(" 3. Backspace only on lower case characters.\n");
    printf("    Choose this option for older 9 pin printers where there is no\n");
    printf("    space above capital characters to place the diacritical mark.\n");
    printf(" 4. Add the diacritical mark, but not the backspace: I.e. A'\n");
    printf("    Choose this option for output intended for text display where formatting\n");
    printf("    and character positioning is not important.\n\n");
    scanf("%d",&UCSCU_output_mode.backspacing);
    UCSCU_output_mode.backspacing -= 1;
  };


  UCSCU_output_mode.graphics = -1;
  while ((UCSCU_output_mode.graphics < 0) || (UCSCU_output_mode.graphics > 2))
  {
    printf("\nWhen a graphics character or special symbol is not available in the\n");
    printf("target character set, the converter can attempt to draw the character\n");
    printf("by using characters that are available by backspacing and typing over\n");
    printf("characters.  The quality of the results will vary depending on the device:\n\n");
    printf(" 1. Try to draw the character using multiple characters.\n");
    printf("    This option should only be used for output intended for printers.\n");
    printf(" 2. Mark the graphic using a single question mark.\n");
    printf("    Use this option where it would be helpful to see where graphics characters\n");
    printf("    would be positioned.  Note that line drawing characters will be replaced\n");
    printf("    with plus for intersections, tees and corners, minus and bar for lines.\n");
    printf(" 3. No graphics - use space.\n");
    printf("    Use this option if drawing replacement graphics is unacceptable.\n");
    scanf("%d",&UCSCU_output_mode.graphics);
    UCSCU_output_mode.graphics -=1;
  };

};

void UCSCU_prompt_options()
{
  printf("\nEnter the name of the input text file\n");
  gets(input_filename);
  UCSCU_input_option.filename = input_filename;

  UCSCU_prompt_input_options();

  printf("\nEnter the name of the output text file\n");
  scanf("%s",output_filename);
  UCSCU_output_option.filename = output_filename;


  UCSCU_prompt_output_options();
};

#if defined(__STDC__) || defined(__cplusplus)
void UCSCU_show_usage(void)
#else
void UCSCU_show_usage()
#endif
{
#ifdef UCSCU_STANDALONE
  printf("\nNon-interactive usage:\n");
  printf("    ucscu option INPUT inname 7bit 8bit OUTPUT outname 7bit 8bit\n");
  printf("\n inname and outname are the input and output text files.\n");
  printf("inname and or outname may be a - hypen to indicate stdin or stdout.\n");
  printf("\nA 7-bit and an 8-bit character set must be choosen for both files:\n");
#endif
  printf("The 7-Bit ASCII/ISO character set may be one of:\n");
  printf("    ENGLISH  FINNISH  SWISS  FRENCH   PORTUGUESE\n");
  printf("    ITALIAN  SPANISH  DUTCH  SWEDISH\n");
  printf("    NORDIC  (Norwegian and Danish\n");
  printf("    BELGIAN (French and Flemish)\n");
  printf("    GERMAN  (Also Austrian)\n");
  printf("    TEX     (Output TeX accent format)\n");
  printf("The 8-bit international character set may be one of:\n");
  printf("    DEC ECMA94 AMIGA ROMAN EPSON QUME TOSHIBA DIABLO\n");
  printf("    PC8 PC850 PC852 PC860 PC863 PC865 PC8DN\n");
  printf("    or NONE to for 7-bit ISO or ASCII representations.\n");
  printf("Back spacing options:  BS_NORMAL BS_NEVER BS_NOT_CAP BS_OMIT\n");
  printf("Graphics draw options: GR_NORMAL GR_MARK  GR_NONE\n");
  printf("7-Bit selection preference : PREFER7BIT\n");
#ifdef UCSCU_STANDALONE
  printf("\nInteractive usage: ucscu\n\n");
  printf("Try running the interactive mode for additional documentation\n");
#endif
};


#if defined(__STDC__) || defined(__cplusplus)
void UCSCU_parse_command_line(int argc,char *argv[])
#else
void UCSCU_parse_command_line(argc,argv)
int argc;
char *argv[];
#endif
{
  int arg;
  int IO_option = 0;

  for (arg = 1; arg < argc; arg++)
  {
    if      (strcmp(argv[arg],"ENGLISH")   == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_ENGLISH7;        else UCSCU_output_option.bit7_CS = CS_ENGLISH7;       }
    else if (strcmp(argv[arg],"GERMAN")    == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_GERMAN7;         else UCSCU_output_option.bit7_CS = CS_GERMAN7;        }
    else if (strcmp(argv[arg],"DUTCH")     == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_DUTCH7;          else UCSCU_output_option.bit7_CS = CS_DUTCH7;         }
    else if (strcmp(argv[arg],"FRENCH")    == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_FRENCH7;         else UCSCU_output_option.bit7_CS = CS_FRENCH7;        }
    else if (strcmp(argv[arg],"SWEDISH")   == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_SWEDISH7;        else UCSCU_output_option.bit7_CS = CS_SWEDISH7;       }
    else if (strcmp(argv[arg],"NORDIC")    == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_NORDIC7;         else UCSCU_output_option.bit7_CS = CS_NORDIC7;        }
    else if (strcmp(argv[arg],"ITALIAN")   == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_ITALIAN7;        else UCSCU_output_option.bit7_CS = CS_ITALIAN7;       }
    else if (strcmp(argv[arg],"SPANISH")   == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_SPANISH7;        else UCSCU_output_option.bit7_CS = CS_SPANISH7;       }
    else if (strcmp(argv[arg],"PORTUGUESE")== 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_PORTUGUESE7;     else UCSCU_output_option.bit7_CS = CS_PORTUGUESE7;    }
    else if (strcmp(argv[arg],"FINNISH")   == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_FINNISH7;        else UCSCU_output_option.bit7_CS = CS_FINNISH7;       }
    else if (strcmp(argv[arg],"BELGIAN")   == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_BELGIAN7;        else UCSCU_output_option.bit7_CS = CS_BELGIAN7;       }
    else if (strcmp(argv[arg],"SWISS")     == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_SWISS7;          else UCSCU_output_option.bit7_CS = CS_SWISS7;         }
    else if (strcmp(argv[arg],"TEX")       == 0) {if (IO_option)UCSCU_input_option.bit7_CS = CS_TeX;             else UCSCU_output_option.bit7_CS = CS_TeX;            }

    else if (strcmp(argv[arg],"NONE")      == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_NONE;            else UCSCU_output_option.bit8_CS = CS_NONE;           }
    else if (strcmp(argv[arg],"DEC")       == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_DEC8;            else UCSCU_output_option.bit8_CS = CS_DEC8;           }
    else if (strcmp(argv[arg],"ECMA94")    == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_ECMA94;          else UCSCU_output_option.bit8_CS = CS_ECMA94;         }
    else if (strcmp(argv[arg],"AMIGA")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_AMIGA;           else UCSCU_output_option.bit8_CS = CS_AMIGA;          }
    else if (strcmp(argv[arg],"ROMAN")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_ROMAN8;          else UCSCU_output_option.bit8_CS = CS_ROMAN8;         }
    else if (strcmp(argv[arg],"EPSON")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_EPSON8;          else UCSCU_output_option.bit8_CS = CS_EPSON8;         }
    else if (strcmp(argv[arg],"QUME")      == 0)
      {if (IO_option)UCSCU_input_option.bit8_CS = CS_QUME_SPRINT_11;  else UCSCU_output_option.bit8_CS = CS_QUME_SPRINT_11; }
    else if (strcmp(argv[arg],"TOSHIBA")   == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_TOSHIBA_1340;    else UCSCU_output_option.bit8_CS = CS_TOSHIBA_1340;   }
    else if (strcmp(argv[arg],"DIABLO")    == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_DIABLO_630;      else UCSCU_output_option.bit8_CS = CS_DIABLO_630;     }
    else if (strcmp(argv[arg],"PC8")       == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC8;             else UCSCU_output_option.bit8_CS = CS_PC8;            }
    else if (strcmp(argv[arg],"PC850")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC850;           else UCSCU_output_option.bit8_CS = CS_PC850;          }
    else if (strcmp(argv[arg],"PC852")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC852;           else UCSCU_output_option.bit8_CS = CS_PC852;          }
    else if (strcmp(argv[arg],"PC860")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC860;           else UCSCU_output_option.bit8_CS = CS_PC860;          }
    else if (strcmp(argv[arg],"PC863")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC863;           else UCSCU_output_option.bit8_CS = CS_PC863;          }
    else if (strcmp(argv[arg],"PC865")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC865;           else UCSCU_output_option.bit8_CS = CS_PC865;          }
    else if (strcmp(argv[arg],"PC8DN")     == 0) {if (IO_option)UCSCU_input_option.bit8_CS = CS_PC8DN;           else UCSCU_output_option.bit8_CS = CS_PC8DN;          }

    else if (strcmp(argv[arg],"BS_NORMAL") == 0) UCSCU_output_mode.backspacing = BS_NORMAL;
    else if (strcmp(argv[arg],"BS_NEVER")  == 0) UCSCU_output_mode.backspacing = BS_NEVER;
    else if (strcmp(argv[arg],"BS_NOT_CAP")== 0) UCSCU_output_mode.backspacing = BS_NOT_CAP;
    else if (strcmp(argv[arg],"BS_OMIT")   == 0) UCSCU_output_mode.backspacing = BS_OMIT;

    else if (strcmp(argv[arg],"GR_NORMAL") == 0) UCSCU_output_mode.graphics = GR_NORMAL;
    else if (strcmp(argv[arg],"GR_MARK")   == 0) UCSCU_output_mode.graphics = GR_MARK;
    else if (strcmp(argv[arg],"GR_NONE")   == 0) UCSCU_output_mode.graphics = GR_NONE;

    else if (strcmp(argv[arg],"PREFER7BIT")== 0) UCSCU_output_mode.prefer_7bit = 1;

    else if (strcmp(argv[arg],"OUTPUT")    == 0) IO_option = 0;
    else if (strcmp(argv[arg],"INPUT")     == 0) IO_option = 1;
#ifdef UCSCU_STANDALONE
    else
       if (IO_option)
         UCSCU_input_option.filename=argv[arg];
       else
         UCSCU_output_option.filename=argv[arg];
#endif
  };
};
#ifdef UCSCU_STANDALONE
#if defined(__STDC__) || defined(__cplusplus)
void UCSCU_init(void)
#else
void UCSCU_init()
#endif
#else
#if defined(__STDC__) || defined(__cplusplus)
void UCSCU_init(int input_bit7_CS,
                int input_bit8_CS,
                int output_bit7_CS,
                int output_bit8_CS,
                int outmode_quiet,
                int outmode_prefer7_bit,
                int outmode_backspacing,
                int outmode_graphics)
#else
void UCSCU_init
(input_bit7_CS,input_bit8_CS,output_bit7_CS,output_bit8_CS,
 outmode_quiet,outmode_prefer7_bit,outmode_backspacing,outmode_graphics)
int input_bit7_CS;
int input_bit8_CS;
int output_bit7_CS;
int output_bit8_CS;
int outmode_quiet;
int outmode_prefer7_bit;
int outmode_backspacing;
int outmode_graphics;
#endif
#endif
{
#ifndef UCSCU_STANDALONE
  UCSCU_input_option.bit7_CS = input_bit7_CS;
  UCSCU_input_option.bit8_CS = input_bit8_CS;
  UCSCU_output_option.bit7_CS = output_bit7_CS;
  UCSCU_output_option.bit8_CS = output_bit8_CS;

  UCSCU_output_mode.quiet           = outmode_quiet;
  UCSCU_output_mode.prefer_7bit     = outmode_prefer7_bit;
  UCSCU_output_mode.backspacing     = outmode_backspacing;
  UCSCU_output_mode.graphics        = outmode_graphics;
#endif

  if ((UCSCU_input_option.bit8_CS == CS_PC850) || (UCSCU_output_option.bit8_CS == CS_PC850))
     copy_CS(CS_PC8,CS_PC850);
  if ((UCSCU_input_option.bit8_CS == CS_PC852) || (UCSCU_output_option.bit8_CS == CS_PC852))
     copy_CS(CS_PC8,CS_PC852);
  if ((UCSCU_input_option.bit8_CS == CS_PC860) || (UCSCU_output_option.bit8_CS == CS_PC860))
     copy_CS(CS_PC8,CS_PC860);
  if ((UCSCU_input_option.bit8_CS == CS_PC863) || (UCSCU_output_option.bit8_CS == CS_PC863))
     copy_CS(CS_PC8,CS_PC863);
  if ((UCSCU_input_option.bit8_CS == CS_PC865) || (UCSCU_output_option.bit8_CS == CS_PC865))
     copy_CS(CS_PC8,CS_PC865);
  if ((UCSCU_input_option.bit8_CS == CS_PC8DN) || (UCSCU_output_option.bit8_CS == CS_PC8DN))
     copy_CS(CS_PC8,CS_PC8DN);

  UCSCU_do_direct_conversion = setup_direct_conversion();

#ifdef UCSCU_STANDALONE
  if (!UCSCU_output_mode.quiet) printf("Processing file\n");
#endif
};

#ifdef UCSCU_STANDALONE

#ifdef VMS
#if defined(__STDC__) || defined(__cplusplus)
main(int argc, char *argv[])
#else
main(argc,argv)
int argc;
char *argv[];
#endif
#else
#if defined(__STDC__) || defined(__cplusplus)
int main(int argc, char *argv[])
#else
void main(argc,argv)
int argc;
char *argv[];
#endif
#endif

#else
#if defined(__STDC__) || defined(__cplusplus)
/* Returns 0 if successfull
           1 if failed
*/
int UCSCU_file(char *input_filename, char *output_filename)
#else
int UCSCU_file(input_filename,output_filename)
char *input_filename;
char *output_filename;
#endif
#endif
{

  int curr_char;

#ifdef UCSCU_STANDALONE

  if (argc == 1)
     UCSCU_prompt_options();
  else
    if (argc < 3)
    { UCSCU_show_usage();
      exit(UCSCU_FAILED);
    }
    else
    { UCSCU_output_mode.quiet = 1;
      UCSCU_parse_command_line(argc,argv);
    };
#else
    strcpy(UCSCU_input_option.filename,input_filename);
    strcpy(UCSCU_output_option.filename,output_filename);
#endif

  if (strcmp(UCSCU_input_option.filename,UCSCU_output_option.filename) == 0)
#ifdef UCSCU_STANDALONE
     if (strcmp(UCSCU_input_option.filename,"-") != 0)
     {
       printf("Cannot use same name for input and output file\n");
       exit(UCSCU_FAILED);
     };
#else
     return(UCSCU_FAILED);
#endif

  UCSCU_input_option.file =
#ifdef UCSCU_STANDALONE
      (strcmp(UCSCU_input_option.filename,"-") == 0) ? stdin :
#endif
      fopen(UCSCU_input_option.filename,READMODE);

  if (UCSCU_input_option.file == 0)
#ifdef UCSCU_STANDALONE
  {
    printf("Unable to open input file %s\n",UCSCU_input_option.filename);
    exit(UCSCU_FAILED);
  };
#else
    return(UCSCU_FAILED);
#endif

  UCSCU_output_option.file =
#ifdef UCSCU_STANDALONE
      (strcmp(UCSCU_output_option.filename,"-") == 0) ? stdout :
#endif
      fopen(UCSCU_output_option.filename,WRITEMODE);

  if (UCSCU_output_option.file == 0)
#ifdef UCSCU_STANDALONE
  {
    printf("Unable to open output file %s\n",UCSCU_output_option.filename);
    exit(UCSCU_FAILED);
  };
  if (UCSCU_output_option.file == stdout) UCSCU_output_mode.quiet = 1;
  if (!UCSCU_output_mode.quiet) printf("Setting up PC character sets\n");
  UCSCU_init();
#else
  return(UCSCU_FAILED);
#endif

  curr_char = UCSCU_do_direct_conversion ? in_char() : in_symbol();
  while (curr_char)
  {
      fprintf(UCSCU_output_option.file,"%s",UCSCU_do_direct_conversion ?
          direct_conversion[curr_char] : UCSCU_char(curr_char));

#ifdef UCSCU_STANDALONE
      if (!UCSCU_output_mode.quiet && (curr_char == '\n'))
      {
         printf("\r%d   ",UCSCU_line_count++);
         fflush(stdout);
      };
      curr_char = UCSCU_do_direct_conversion ? in_char() : in_symbol();
#else
      UCSCU_line_count += 1;
#endif
  };

  if (UCSCU_input_option.file != stdin)
     fclose(UCSCU_input_option.file);
  if (UCSCU_output_option.file != stdout)
     fclose(UCSCU_output_option.file);

#ifdef UCSCU_STANDALONE
#if defined(__STDC__) || defined(__cplusplus)
  return(UCSCU_SUCCEEDED);
#else
  exit(UCSCU_SUCCEEDED);
#endif
#else
  return(UCSCU_SUCCEEDED);
#endif
};

