/* pnmtocgm - convert a portable anymap to an ANSI/ISO CGM metafile. 
**
** Copyright April 1991, Carnegie Mellon University.  Author Joel Welling.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include <stdio.h>
#include <pnm.h>

#ifdef USE_STDLIB_H
#include <stdlib.h>
#else

/* Some externals */
#ifdef VOID_STAR
extern void *malloc();
#else
extern char *malloc();
#endif
extern void free();

#endif /* USE_STDLIB_H */

/* Rather than include a great list of files, define the format ID's
 * used by the pbm, pgm, and ppm packages.
 */
#define PBM_FORMAT  'P' * 256 + '1'
#define RPBM_FORMAT 'P' * 256 + '4'
#define PGM_FORMAT  'P' * 256 + '2'
#define RPGM_FORMAT 'P' * 256 + '5'
#define PPM_FORMAT  'P' * 256 + '3'
#define RPPM_FORMAT 'P' * 256 + '6'

/*  This module recognizes what type of machine it's on by the presence
of the symbol VMS, unix, CRAY, or ardent.  The following makes the machine
assignment less ambiguous.
*/
#if ( unix && ( !CRAY && !ardent $$ !__hpux ) )
#define USE_UNIX
#endif

/* Include defs files that allow linkage to Fortran on various systems */
#ifdef USE_UNIX
#include "unix_defs.h"
#endif
#ifdef CRAY
#include "unicos_defs.h"
#endif
#ifdef ardent
#include "unicos_defs.h"  /* these are also appropriate on Ardent Titan */
#endif
#ifdef _IBMR2
/* include nothing */
#endif
#ifdef __hpux
/* include nothing */
#endif

/* Struct to hold file information */
typedef struct header_struct {
  FILE *file;
  int cols;
  int rows;
  xelval maxval;
  int format;
} pnm_header;

/* Flag to produce verbose output */
static int verboseflag= 0;

static void open_output(outfname)
char *outfname;
/* This routine opens the output file. */
{
  int ierr= 0, i256= 256;
  
  wrcopn(outfname,&ierr);
  if (ierr) pm_error("Error opening file <%s> for output!\n",outfname);
  wrmxci(&i256,&ierr);
  if (ierr) pm_error("Error writing to file!\n");
}

static void close_output()
/* This routine closes the output file. */
{
  int ierr= 0;

  wrtend(&ierr);
  if (ierr) pm_error("Error closing output!\n");
}

static void begin_direct_page(in_hdr)
pnm_header *in_hdr;
/* This routine begins a direct color page */
{
  int ierr= 0, one= 1;
  float red, green, blue;

  /* Black background, for lack of anything better */
  red= 0.0;
  green= 0.0;
  blue= 0.0;

  wrbegp(&ierr);
  if (ierr) pm_error("Error writing begin page!\n");
  wrtcsm(&one,&ierr);
  if (ierr) pm_error("Error writing color selection mode!\n");
  wrbgdc(&red, &green, &blue, &ierr);
  if (ierr) pm_error("Error writing background color!\n");
  wrbgpb(&ierr);
  if (ierr) pm_error("Error writing begin picture body!\n");
}

static void handle_color_table(in_hdr)
pnm_header *in_hdr;
/* This routine makes up a fake gray color table for the image.
 */
{
  int tablesize, i, zero= 0, ierr= 0;
  float *rarray, *garray, *barray;
  float gray, graystep;

  tablesize= in_hdr->maxval + 1;

  if ( !(rarray= (float *)malloc( tablesize*sizeof(float) )) ) 
    pm_error("unable to allocate %d floats for color table reds!\n",
	     tablesize);
  if ( !(garray= (float *)malloc( tablesize*sizeof(float) )) ) 
    pm_error("unable to allocate %d floats for color table greens!\n",
	     tablesize);
  if ( !(barray= (float *)malloc( tablesize*sizeof(float) )) ) 
    pm_error("unable to allocate %d floats for color table blues!\n",
	     tablesize);

  /* Transcribe the color map into the float arrays */
  gray= 0.0;
  graystep= 1.0/((float)(tablesize-1));
  for (i=0; i<tablesize; i++) {
    rarray[i]= garray[i]= barray[i]= gray;
    gray += graystep;
  }

  /* Actually write the map */
  wrctbl(rarray, garray, barray, &zero, &tablesize, &ierr);
  if (ierr) pm_error("Error writing color map!\n");

  /* Clean up */
  free( (char *)rarray );
  free( (char *)garray );
  free( (char *)barray );
}

static void begin_indexed_page(in_hdr)
pnm_header *in_hdr;
/* This routine begins an indexed color page */
{
  int ierr= 0;
  float red, green, blue;

  /* Black background, for lack of anything better */
  red= 0.0;
  green= 0.0;
  blue= 0.0;

  /* Preliminary CGM stuff */
  wrbegp(&ierr);
  if (ierr) pm_error("Error writing begin page!\n");
  wrbgdc(&red, &green, &blue, &ierr);
  if (ierr) pm_error("Error writing background color!\n");
  wrbgpb(&ierr);
  if (ierr) pm_error("Error writing begin picture body!\n");

  /* Produce a color table */
  handle_color_table(in_hdr);
}

static void handle_monochrome_table(in_hdr)
pnm_header *in_hdr;
/* This routine makes up a two element color table. */
{
  int tablesize, i, zero= 0, ierr= 0;
  float rarray[2], garray[2], barray[2];

  tablesize= 2;

  rarray[0]= garray[0]= barray[0]= 1.0;
  rarray[1]= garray[1]= barray[1]= 0.0;

  /* Actually write the map */
  wrctbl(rarray, garray, barray, &zero, &tablesize, &ierr);
  if (ierr) pm_error("Error writing color map!\n");
}

static void begin_monochrome_page(in_hdr)
pnm_header *in_hdr;
/* This routine begins a monochrome color page */
{
  int ierr= 0;
  float red, green, blue;

  /* white background */
  red= 1.0;
  green= 1.0;
  blue= 1.0;

  /* Preliminary CGM stuff */
  wrbegp(&ierr);
  if (ierr) pm_error("Error writing begin page!\n");
  wrbgdc(&red, &green, &blue, &ierr);
  if (ierr) pm_error("Error writing background color!\n");
  wrbgpb(&ierr);
  if (ierr) pm_error("Error writing begin picture body!\n");

  /* Produce a color table */
  handle_monochrome_table(in_hdr);
}

static void end_page()
/* This routine ends a page. */
{
  int ierr= 0;

  wrendp(&ierr);
  if (ierr) pm_error("Error ending output page!\n");
}

static void center_image(in_hdr, px, py, qx, qy, rx, ry)
pnm_header *in_hdr;
float *px, *py, *qx, *qy, *rx, *ry;
/* This routine centers the image in DrawCGM's unit square coordinates. */
{
  float xrange, yrange;

  xrange= (float)(in_hdr->cols);
  yrange= (float)(in_hdr->rows);
  if (xrange>=yrange) {
    *px= 0.0;
    *qx= 1.0;
    *rx= *qx;
    *py= 0.5 * (1.0 - yrange/xrange);
    *qy= 1.0 - *py;
    *ry= *py;
  }
  else {
    *py= 0.0;
    *qy= 1.0;
    *ry= *py;
    *px= 0.5 * (1.0 - xrange/yrange);
    *qx= 1.0 - *px;
    *rx= *qx;
  }
}

static void process_indexed_page(in_hdr)
pnm_header *in_hdr;
/* This routine handles one page of a file, assuming indexed color. */
{
  int x, y;
  static int *iarray= (int *)0; /* Stores the CGM integer image */
  int *iptr;
  gray *thisrow, *thispixel;
  static int nx, ny;            /* dimensions of iarray */
  float px, py, qx, qy, rx, ry; /* CGM array boundaries */
  int ierr= 0;

  /* Be verbose if requested */
  if (verboseflag) fprintf(stderr,"Found indexed color (pgm) image\n");

  /* Begin an output page, including the color table */
  begin_indexed_page(in_hdr);
  
  /* Calculate array boundaries */
  center_image(in_hdr, &px, &py, &qx, &qy, &rx, &ry);

  /* Allocate the CGM image array, if it's not already there */
  if (!iarray || 
      !(nx == in_hdr->cols) ||
      !(ny == in_hdr->rows)) {
    if (iarray) free( (char *)iarray );
    nx= in_hdr->cols;
    ny= in_hdr->rows;
    if ( !(iarray= (int *)malloc( nx*ny*sizeof(int) )) ) 
      pm_error("cannot allocate %d ints!\n",nx*ny);
  }

  /* Allocate the row of pgm grays */
  thisrow= pgm_allocrow( in_hdr->cols );

  /* Copy the image to the output file. */
  for ( y=0; y < in_hdr->rows; y++ ) {
    pgm_readpgmrow( in_hdr->file, thisrow, in_hdr->cols, in_hdr->maxval,
		   in_hdr->format );
    iptr= iarray + (in_hdr->rows - (y+1))*in_hdr->cols;
    thispixel= thisrow;
    for ( x=0; x < in_hdr->cols; x++ ) *iptr++= (int) *thispixel++;
    }
  
  /* Write the CGM cell array */
  wrtcla( iarray, &nx, &ny, &px, &py, &qx, &qy, &rx, &ry, &ierr );
  if (ierr) pm_error("Error writing indexed cell array!\n");

  /* End the page */
  end_page();

  /* Clean up */
  pgm_freerow( thisrow );
}

static void process_direct_page(in_hdr)
pnm_header *in_hdr;
/* This routine handles one page of a file, assuming direct color. */
{
  int x, y;
  static float *rarray= (float *)0; /* red CGM image */
  static float *garray= (float *)0; /* green CGM image */
  static float *barray= (float *)0; /* blue CGM image */
  pixel *thisrow, *thispixel;
  float *rptr, *gptr, *bptr;        /* CGM image arrays */
  static int nx, ny;                /* dimensions of image */
  float px, py, qx, qy, rx, ry;     /* CGM array boundaries */
  int ierr= 0;
  
  /* Be verbose if requested */
  if (verboseflag) fprintf(stderr,"Found direct color (ppm) image\n");
  
  /* Begin an output page */
  begin_direct_page(in_hdr);
  
  /* Calculate array boundaries */
  center_image(in_hdr, &px, &py, &qx, &qy, &rx, &ry);
  
  /* Allocate the CGM image array, if it's not already there */
  if (!rarray || 
      !(nx == in_hdr->cols) ||
      !(ny == in_hdr->rows)) {
    if (rarray) free( (char *)rarray );
    if (garray) free( (char *)garray );
    if (barray) free( (char *)barray );
    nx= in_hdr->cols;
    ny= in_hdr->rows;
    if ( !(rarray= (float *)malloc( nx*ny*sizeof(float) )) ) 
      pm_error("cannot allocate %d floats for red!\n",nx*ny);
    if ( !(garray= (float *)malloc( nx*ny*sizeof(float) )) ) 
      pm_error("cannot allocate %d floats for green!\n",nx*ny);
    if ( !(barray= (float *)malloc( nx*ny*sizeof(float) )) ) 
      pm_error("cannot allocate %d floats for blue!\n",nx*ny);
  }

  /* Allocate space for the pgm row */
  thisrow= ppm_allocrow( in_hdr->cols );

  /* Read the input image and copy it to the output file. */
  for ( y = 0; y < in_hdr->rows; y++ )
    {
      ppm_readppmrow( in_hdr->file, thisrow, in_hdr->cols, in_hdr->maxval,
		     in_hdr->format );
      rptr= rarray + (in_hdr->rows - (y+1))*in_hdr->cols;
      gptr= garray + (in_hdr->rows - (y+1))*in_hdr->cols;
      bptr= barray + (in_hdr->rows - (y+1))*in_hdr->cols;
      thispixel= thisrow;
      for (x=0; x<nx; x++) {
	*rptr++= (float)PPM_GETR(*thispixel)/(float)in_hdr->maxval;
	*gptr++= (float)PPM_GETG(*thispixel)/(float)in_hdr->maxval;
	*bptr++= (float)PPM_GETB(*thispixel)/(float)in_hdr->maxval;
	thispixel++;
      }
    }
  
  /* Write the CGM cell array */
  wcladc( rarray, garray, barray, 
	 &nx, &ny, &px, &py, &qx, &qy, &rx, &ry, &ierr );
  if (ierr) pm_error("Error writing direct color cell array!\n");

  /* End the page */
  end_page();

  /* Clean up */
  ppm_freerow( thisrow );
}

static void process_monochrome_page(in_hdr)
pnm_header *in_hdr;
/* This routine handles one page of a file, assuming a monochrome image. */
{
  int x, y;
  static int *iarray= (int *)0; /* Stores the CGM integer image */
  int *iptr;
  bit *thisrow, *thispixel;
  static int nx, ny;            /* dimensions of iarray */
  float px, py, qx, qy, rx, ry; /* CGM array boundaries */
  int ierr= 0, one= 1;

  /* Be verbose if requested */
  if (verboseflag) fprintf(stderr,"Found monochrome color (pbm) image\n");

  /* Begin an output page, including the color table */
  begin_monochrome_page(in_hdr);
  
  /* Calculate array boundaries */
  center_image(in_hdr, &px, &py, &qx, &qy, &rx, &ry);

  /* Allocate the CGM image array, if it's not already there */
  if (!iarray || 
      !(nx == in_hdr->cols) ||
      !(ny == in_hdr->rows)) {
    if (iarray) free( (char *)iarray );
    nx= in_hdr->cols;
    ny= in_hdr->rows;
    if ( !(iarray= (int *)malloc( nx*ny*sizeof(int) )) ) 
      pm_error("cannot allocate %d ints!\n",nx*ny);
  }

  /* Allocate the row of pgm grays */
  thisrow= pbm_allocrow( in_hdr->cols );

  /* Copy the image to the output file. */
  for ( y=0; y < in_hdr->rows; y++ ) {
    pbm_readpbmrow( in_hdr->file, thisrow, in_hdr->cols, in_hdr->format );
    iptr= iarray + (in_hdr->rows - (y+1))*in_hdr->cols;
    thispixel= thisrow;
    for ( x=0; x < in_hdr->cols; x++ ) *iptr++= (*thispixel++ != PBM_WHITE);
    }
  
  /* Write the CGM cell array, in packed list mode for precision 1 */
  wrgcla( iarray, &nx, &ny, &px, &py, &qx, &qy, &rx, &ry, 
	 &one, &one, &ierr );
  if (ierr) pm_error("Error writing indexed cell array!\n");

  /* End the page */
  end_page();

  /* Clean up */
  pbm_freerow( thisrow );
}

main( argc, argv )
int argc;
char **argv;
{
  char *infname = NULL;
  char *outfname = NULL;
  int  debugflag= 0;
  FILE *outfile;
  pnm_header in_hdr;
  xel **image;
  int ierr= 0;
  int pnm_cnt= 0;
  int argn= 1;
  char usage[]= "[-d] [-v] [-o cgmfilename] [pnmfile] [pnmfile] ...";
  
  /* Set the global program flag */
  pm_progname= argv[0];

  /* Parse the argument list */
  while ( argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0' )
    {
      if ( !strcmp(argv[argn],"-d") ) debugflag= 1;
      else if ( !strcmp(argv[argn],"-v") ) verboseflag= 1;
      else if ( !strcmp(argv[argn],"-o") ) outfname= argv[++argn];
      else pm_usage( usage );
      argn++;
    }
  
  /* Turn on debugging if requested. */
  if (debugflag) {
    tgldbg(&ierr);
    if (ierr) pm_message("couldn't set debugging!\n");
  }

  /* Read images from the input file until the end of file is encountered. */
  while ( argn < argc ) {

    /* Open the input file. */
    if ( strcmp(argv[argn],"-") ) infname= argv[argn];
    else infname= NULL;
    in_hdr.file = pm_openr( infname ? infname : "-" );

    /* Load an image */
    pnm_readpnminit( in_hdr.file, &in_hdr.cols, &in_hdr.rows, 
		       &in_hdr.maxval, &in_hdr.format );
    
    /* Open the CGM output file (first frame only) */
    if (pnm_cnt==0) open_output(outfname ? outfname : "-");
    
    /* Count the input images. */
    pnm_cnt++;
    
    /* Be verbose if requested */
    if (verboseflag) fprintf(stderr,"image %d: ",pnm_cnt);

    /* Based on the format of the input image, select an output mode
     * for this picture.
     */
    switch (in_hdr.format) {
    case PPM_FORMAT:        /* ascii ppm */
    case RPPM_FORMAT:       /* raw ppm */
      process_direct_page(&in_hdr); break;
    case PGM_FORMAT:        /* ascii pgm */
    case RPGM_FORMAT:       /* raw pgm */
      process_indexed_page(&in_hdr); break;
    case PBM_FORMAT:        /* ascii pbm */
    case RPBM_FORMAT:       /* raw pbm */
      process_monochrome_page(&in_hdr); break;
    default: fprintf(stderr,"%d is an invalid image format number!\n",
			in_hdr.format);
    }

    if (infname) pm_close( in_hdr.file );
    argn++;
  }
  
  /* Close the CGM output file, if one was written. */
  if (pnm_cnt > 0) close_output();
  else pm_error("No input images found!\n");
}

