/*
 *	lbp.c -- printer dependent routines
 *		 This file is for Canon New LaserShot (LIPS3)
 *
 *	Written by H. Nagahara on 9/5/90 15:51:47.
 *	Copyright (C) 1990 by ASCII Corporation.  All rights reserved.
 *		SCCS memo: lbp.c 5.1
 */
#include	<stdio.h>
#include	<string.h>
#include	"config.h"
#include	"font.h"
#include	"drive.h"
#include	"convert.h"
#define		_LBP_
#include	"lbp.h"

#if MSDOS
# include	<stdlib.h>
int	lprintf( char *, ... );
int	lputchar( int );
#else
#define	lprintf		printf
#define	lputchar	putchar
#endif

#ifdef LBPDEBUG
static char	*(bitp[]) = {
    "        ",
    "       #",
    "      # ",
    "      ##",
    "     #  ",
    "     # #",
    "     ## ",
    "     ###",
    "    #   ",
    "    #  #",
    "    # # ",
    "    # ##",
    "    ##  ",
    "    ## #",
    "    ### ",
    "    ####",
    "   #    ",
    "   #   #",
    "   #  # ",
    "   #  ##",
    "   # #  ",
    "   # # #",
    "   # ## ",
    "   # ###",
    "   ##   ",
    "   ##  #",
    "   ## # ",
    "   ## ##",
    "   ###  ",
    "   ### #",
    "   #### ",
    "   #####",
    "  #     ",
    "  #    #",
    "  #   # ",
    "  #   ##",
    "  #  #  ",
    "  #  # #",
    "  #  ## ",
    "  #  ###",
    "  # #   ",
    "  # #  #",
    "  # # # ",
    "  # # ##",
    "  # ##  ",
    "  # ## #",
    "  # ### ",
    "  # ####",
    "  ##    ",
    "  ##   #",
    "  ##  # ",
    "  ##  ##",
    "  ## #  ",
    "  ## # #",
    "  ## ## ",
    "  ## ###",
    "  ###   ",
    "  ###  #",
    "  ### # ",
    "  ### ##",
    "  ####  ",
    "  #### #",
    "  ##### ",
    "  ######",
    " #      ",
    " #     #",
    " #    # ",
    " #    ##",
    " #   #  ",
    " #   # #",
    " #   ## ",
    " #   ###",
    " #  #   ",
    " #  #  #",
    " #  # # ",
    " #  # ##",
    " #  ##  ",
    " #  ## #",
    " #  ### ",
    " #  ####",
    " # #    ",
    " # #   #",
    " # #  # ",
    " # #  ##",
    " # # #  ",
    " # # # #",
    " # # ## ",
    " # # ###",
    " # ##   ",
    " # ##  #",
    " # ## # ",
    " # ## ##",
    " # ###  ",
    " # ### #",
    " # #### ",
    " # #####",
    " ##     ",
    " ##    #",
    " ##   # ",
    " ##   ##",
    " ##  #  ",
    " ##  # #",
    " ##  ## ",
    " ##  ###",
    " ## #   ",
    " ## #  #",
    " ## # # ",
    " ## # ##",
    " ## ##  ",
    " ## ## #",
    " ## ### ",
    " ## ####",
    " ###    ",
    " ###   #",
    " ###  # ",
    " ###  ##",
    " ### #  ",
    " ### # #",
    " ### ## ",
    " ### ###",
    " ####   ",
    " ####  #",
    " #### # ",
    " #### ##",
    " #####  ",
    " ##### #",
    " ###### ",
    " #######",
    "#       ",
    "#      #",
    "#     # ",
    "#     ##",
    "#    #  ",
    "#    # #",
    "#    ## ",
    "#    ###",
    "#   #   ",
    "#   #  #",
    "#   # # ",
    "#   # ##",
    "#   ##  ",
    "#   ## #",
    "#   ### ",
    "#   ####",
    "#  #    ",
    "#  #   #",
    "#  #  # ",
    "#  #  ##",
    "#  # #  ",
    "#  # # #",
    "#  # ## ",
    "#  # ###",
    "#  ##   ",
    "#  ##  #",
    "#  ## # ",
    "#  ## ##",
    "#  ###  ",
    "#  ### #",
    "#  #### ",
    "#  #####",
    "# #     ",
    "# #    #",
    "# #   # ",
    "# #   ##",
    "# #  #  ",
    "# #  # #",
    "# #  ## ",
    "# #  ###",
    "# # #   ",
    "# # #  #",
    "# # # # ",
    "# # # ##",
    "# # ##  ",
    "# # ## #",
    "# # ### ",
    "# # ####",
    "# ##    ",
    "# ##   #",
    "# ##  # ",
    "# ##  ##",
    "# ## #  ",
    "# ## # #",
    "# ## ## ",
    "# ## ###",
    "# ###   ",
    "# ###  #",
    "# ### # ",
    "# ### ##",
    "# ####  ",
    "# #### #",
    "# ##### ",
    "# ######",
    "##      ",
    "##     #",
    "##    # ",
    "##    ##",
    "##   #  ",
    "##   # #",
    "##   ## ",
    "##   ###",
    "##  #   ",
    "##  #  #",
    "##  # # ",
    "##  # ##",
    "##  ##  ",
    "##  ## #",
    "##  ### ",
    "##  ####",
    "## #    ",
    "## #   #",
    "## #  # ",
    "## #  ##",
    "## # #  ",
    "## # # #",
    "## # ## ",
    "## # ###",
    "## ##   ",
    "## ##  #",
    "## ## # ",
    "## ## ##",
    "## ###  ",
    "## ### #",
    "## #### ",
    "## #####",
    "###     ",
    "###    #",
    "###   # ",
    "###   ##",
    "###  #  ",
    "###  # #",
    "###  ## ",
    "###  ###",
    "### #   ",
    "### #  #",
    "### # # ",
    "### # ##",
    "### ##  ",
    "### ## #",
    "### ### ",
    "### ####",
    "####    ",
    "####   #",
    "####  # ",
    "####  ##",
    "#### #  ",
    "#### # #",
    "#### ## ",
    "#### ###",
    "#####   ",
    "#####  #",
    "##### # ",
    "##### ##",
    "######  ",
    "###### #",
    "####### ",
    "########"
    };
#endif	/* LBPDEBUG */

/* display a message to printer console */
void
LBP_sendMSG( s )
char	*s;
{
    lprintf( "\033P1y%s\033\\", s );
}

/* send StartOfJob seq. to printer */
void
LBP_sendSOJ( print )
int	print;		/* true if printing job */
{
    int		base;

    lprintf( "\033%@" );		/* start of text-mode */
    lprintf( "\033P31;300;2;JTeX Out\033\\" );	/* start of JOB */
    lprintf( "\033<" );		/* soft reset */

    if( print ) {
	LBP_sendMSG( "DVI printing" );
	lprintf( "\033[11h" );	/* unit size mode for moving CAP */
	lprintf( "\033[7 I" );	/* select unit (dot = 1/300 inch) */
	/* page format selection */
	switch( papersize ) {
	  case A4:
	    base = 14;
	    break;
	  case A5:
	    base = 16;
	    break;
	  case HAGAKI:
	    base = 18;
	    break;
	  case B4:
	    base = 24;
	    break;
	  case B5:
	    base = 26;
	    break;
	  case LETTER:
	    base = 30;
	    break;
	  case LIGAL:
	    base = 32;
	    break;
	  case USER:
	    base = 80;
	    break;
	  case UNKNOWN:
	    base = 0;
	}
	if( landscape )
	    base++;
	if( papersize != USER )
	    lprintf( "\033[%d;;p", base );
	else
	    lprintf( "\033[%d;%d;%dp", base, (int)(mm2dot(paperwidth) + 0.5),
		    (int)(mm2dot(paperlength) + 0.5) );
	
	lprintf( "\033[?1l" );	/* no auto-wrap */
	lprintf( "\033[?2h" );	/* no auto-new-page */
	lprintf( "\033[?3h" );	/* no auto-move of CAP */
    }
}

/* send EndOfJob seq. to printer */
void
LBP_sendEOJ()
{
    LBP_sendMSG( "" );
    lprintf( "\033P0J\033\\" );	/* end of JOB */
}


/* select Japanese font */
void
LBP_selectJfont( f )
struct font	*f;
{
    uint4	default_width;
    double	pitch_in_meter;
    int		pitch;
    int		point;
    int		face;

    lputchar( SHIFTOUT );			/* select G1 */
    lprintf( "\033$)%c", JIS_GSET - 0x80 );		/* select JIS83/78 */
    default_width = f->info->cc.klist->cinfo->width;
					/* default width is in element 0 */
    pitch_in_meter = ((double)default_width / 1048576.0)
	* (f->scale * scalepoint) * WIDTH_ADJ;
    pitch = (int)(INCH_IN_METER / pitch_in_meter * 100.0 + 0.5);
    lprintf( "\033[?%d K", pitch );		/* select char pitch */	
    point = (int)(m2dot(f->scale * scalepoint) * HIGHT_ADJ + 0.5);
    lprintf( "\033[%d C", point );		/* select char size */
    lprintf( "\033[23m" );			/* select char style */
    lprintf( "\033[22m" );			/* select stroke waght */
    if( strncmp( f->info->name, "min", 3 ) == 0 )
	face = mincho_syotai;
    else if( strncmp( f->info->name, "goth", 4 ) == 0 )
	face = gothic_syotai;
    else {
	fprintf( stderr, "%s: unknown Japanese font %s\n", progname, f->info->name );
	exit( 2 );
    }
    lprintf( "\033[%d;y", face );		/* select syotai */
    lputchar( SHIFTIN );				/* return to G0 */
}
	
/* assign font # to Japanese font for later selection */
void
LBP_assignJfont( no, f )
int		no;	/* font # to assign */  
struct font	*f;	/* font to assign */
{
    /* select Japanese font from scalable font */
    LBP_selectJfont( f );
    lprintf( "\033[%d;2 D", no );
}

/* select alphabet font */
void
LBP_selectAfont( f )
struct font	*f;
{
    lprintf( "\033Pz%s\033\\", f->prn->name );
}

/* assign font # to alphabet font for later selection */
void
LBP_assignAfont( no, f )
int		no;	/* font # to assign */  
struct font	*f;	/* font to assign */
{
    /* select Japanese font from scalable font */
    LBP_selectAfont( f );
    lprintf( "\033[%d;1 D", no );
}

#define	dotpitch(c)	m2dot(((f->info->cc.width[(int)(c) - f->info->begin] / 1048576.0) * (f->scale * scalepoint)))

/* download a font set */
int
LBP_aFont( f, map )
struct font	*f;	/* font info */
struct grif	*map;	/* font bitmap data */
{
    register struct grif	*cur;
    register int		nchar, width, hight, depth;
    int4			bytes;
    int				direction;
    int				pitch;
    int				csize;
    register int		cc;
    register int		dmy;
    register int		i;
    register uint1		*p;
    int				gset;
    char			msgbuf[64];
    char			*cp;

    /* gather download info. */
    nchar = 0;
    bytes = 0;
    width = 0;
    hight = 0;
    depth = 0;
    for( cc = 0; cc < (f->info->end - f->info->begin + 1); cc++ ) {
	cur = map + cc;
	if( cur->bitmap == NULL )
	    continue;
	nchar++;
	dmy = cur->width;
	if( cur->hrefp < 0 )
	    dmy -= cur->hrefp;
	if( width < dmy )
	    width = dmy;
	if( hight < cur->vrefp )
	    hight = cur->vrefp;
	if( depth < (dmy = (int)cur->hight - (int)cur->vrefp) )
	    depth = dmy;
	bytes += (cur->width + 7)/8 * cur->hight + 11;
		/* prolog for every char has 11 bytes */
    }
    f->prn->datacount = bytes;
    if( (width > MAXDOWNWIDTH) || ((hight + depth) > MAXDOWNHIGHT) )
	return FALSE;	/* cannot download! */
    if( (maxdownsize >= 0) && ((downloadcount + bytes) > maxdownsize) )
	return FALSE;	/* download size over! */

    /* setup parameters */
    downloadcount += bytes;
    direction = landscape ? 1 : 0;
    gset = ++using_gset;
    if( (f->info->begin <= 0x20) && (f->info->end >= 0x20 ) )
	pitch = (int)(dotpitch( 0x20 ) + 0.5);
    else
	pitch = m2dot(((f->info->cc.width[0] / 1048576.0) * (f->scale * scalepoint)));
    csize = (int)(m2point( f->scale * scalepoint ) * 100.0 + 0.5);

    /* download prolog */
    strcpy( msgbuf, f->prn->name );
    cp = strchr( msgbuf, '.' );
    if( cp != NULL )
	*cp = '\0';
    LBP_sendMSG( msgbuf );
    lprintf( "\033P;;;;;;x%s\033\\", f->prn->name );
    lprintf( "\033[%ld;%d;0;", bytes, nchar );	/* bytes, nchar, default-mark */
    lprintf( "%d;%d;9;", direction, gset );	/* orientation, gr-set, kerning */
    lprintf( "%d;%d;0;0;", pitch, csize );	/* pitch, size, style, stroke */
    lprintf( "200;%d;%d;%d;", width, hight + depth, depth );
						/* typeface, cel(w), cel(h), base */
    lprintf( "2;0;1;0;1" );	/* u-line, code len, level, base flag, control */
    lprintf( ".p" );

    /* data download loop */
    for( cc = 0; cc < (f->info->end - f->info->begin + 1); cc++ ) {
	cur = map + cc;
	if( cur->bitmap == NULL )
	    continue;
	lputchar( (int)cur->code );		/* char code */
	dmy = (int)(m2dot(((f->info->cc.width[cc] / 1048576.0) * (f->scale * scalepoint))) + 0.5);
	lputchar( dmy >> 8 );
	lputchar( dmy & 0xff );
	lputchar( (int)(cur->width >> 8) );	/* width */
	lputchar( (int)(cur->width & 0xff) );
	dmy =  -cur->hrefp;		/* horizontal offset */
	lputchar( dmy >> 8 );
	lputchar( dmy & 0xff );
	lputchar( (int)(cur->hight >> 8) );	/* hight */
	lputchar( (int)(cur->hight & 0xff) );
	dmy = cur->hight - cur->vrefp;	/* virtical offset */
	lputchar( (dmy >> 8) & 0xff );
	lputchar( dmy & 0xff );
#ifdef	LBPDEBUG
	fprintf( stderr, "\n[%X] (%d,%d):(%d,%d)", cur->code, cur->width, cur->hight,cur->hrefp, cur->vrefp );
#endif
	for( i = 0, p = cur->bitmap;
	     i < ((cur->width + 7)/8 * cur->hight);
	     i++, p++ ) {
	    lputchar( (int)*p );
#ifdef	LBPDEBUG
	    if( (i % ((cur->width + 7)/8)) == 0 )
		fprintf( stderr, "\n" );
	    fprintf( stderr, "%s", bitp[*p] );
#endif
	}
    }

    /* set downloaded font info into fontlist */
    f->prn->graphset = gset;
    f->prn->download = TRUE;

    return TRUE;
}

/* move CAP */
void
LBP_move( x, y )
int4	x, y;
{
    register long	v, h;

    h = (int)(m2dot( (double)x * scalepoint )) + h_base;
    v = (int)(m2dot( (double)y * scalepoint )) + v_base;
    lprintf( "\033[%ld;%ldf", v, h );
}

#ifdef	SQUARE
/* write frame */
void
LBP_square( h0, v0, h1, v1, pattern )
int4	h0, v0;		/* start point */
int4	h1, v1;		/* end point */
int	pattern;	/* pattern # */
{
    LBP_move( h0, v0 );		/* move to start point */
    lprintf( "\033[%d;9;s", pattern );	/* start of overlay */
    LBP_move( h0 + h1, v0 - v1 );	/* move to end point */
    lprintf( "\033[9r" );	/* end of overlay */
}
#endif	/* SQUARE */

/* write rectangle */
void
LBP_rectangle( h0, v0, h1, v1 )
int4	h0, v0;		/* start point */
int4	h1, v1;		/* end point */
{
    register int4	move_x, move_y;

    LBP_move( h0, v0 );		/* move to start point */
    lprintf( "\033[3;9;s" );	/* start of overlay */
    move_x = (int4)(m2dot( (double)h1 * scalepoint ) );
    if( move_x > 0 )
	lprintf( "\033[%lda", move_x );		/* move right */
    else if( move_x < 0 )
	lprintf( "\033[%ldj", (int4)(- move_x) );	/* move left */
    move_y = (int4)(m2dot( (double)v1 * scalepoint ) );
    if( move_y > 0 )
	lprintf( "\033[%ldk", move_y );		/* move down */
    else if( move_y < 0 )
	lprintf( "\033[%lde", (int4)(- move_y) );
    lprintf( "\033[9r" );	/* end of overlay */
}
    
/* print Japanese char as raster image */
void
LBP_raster( x0, y0, cdata )
int4		x0, y0;		/* write positon */
struct fdir	*cdata;		/* font image data */
{
    register int	i;
    int			len;
    int4		rh, rv, doth, dotv;

    /* move to start position */
    rh = (int4)(m2dot( (double)x0 * scalepoint ) + 0.5) + h_base;
    rv = (int4)(m2dot( (double)y0 * scalepoint ) + 0.5) + v_base;
    doth = rh - (int4)cdata->h_offset;
    dotv = rv - (int4)cdata->v_offset;
    lprintf( "\033[%ld;%ldf", dotv, doth );

    /* send laster image */
    len = ((cdata->width + 7) / 8) * cdata->hight;
    lprintf( "\033[%d;%d;300;0;.r", len, (int)((cdata->width +7) / 8) );
    for( i = 0; i < len; i++ )
	lputchar( (int)(*(cdata->bitmap + i)) );
}

/* print char as raster image for non-downloaded font */
void
LBP_rasterAchar( x0, y0, f, c )
int4		x0, y0;
struct font	*f;	/* font */
uint4		c;	/* character code */
{
    register struct grif	*grifdata;
    register int		i;
    int				len;
    register int4		rh, rv;
    int4			doth, dotv;

#ifdef	DEBUG
    if( f->gdata.pattern == NULL ) {
	fprintf( stderr, "%s: internal error.  no pattern of %s.\n",
		 progname, f->info->name );
	return;
    }
#endif

    /* move to start position */
    rh = (int4)(m2dot( (double)x0 * scalepoint ) + 0.5) + h_base;
    rv = (int4)(m2dot( (double)y0 * scalepoint ) + 0.5) + v_base;
    grifdata = f->gdata.pattern + (c - f->info->begin);
    doth = rh - (int4)grifdata->hrefp;
    dotv = rv - (int4)grifdata->vrefp;
    lprintf( "\033[%ld;%ldf", dotv, doth );

    /* send laster image */
    len = ((grifdata->width + 7) / 8) * grifdata->hight;
    lprintf( "\033[%d;%d;300;0;.r", len, (int)((grifdata->width +7) / 8) );
    for( i = 0; i < len; i++ )
	lputchar( (int)(*(grifdata->bitmap + i)) );
}

/* print a char */
void
LBP_putchar( x0, y0, f, c )
int4		x0, y0;
struct font	*f;	/* font */
uint4		c;	/* char code */
{
    register struct jfm	*base;
    register int	idx;
    register int	top, bottom;
    int			found;
    int4		chardepth, charnglue;
    uint4		charwidth, charhight;
    int4		cd, cg;
#ifdef	SQUARE
    int			cw, ch;
#endif

    if( (c < ' ') || (c == 0x7f) ) {	/* alphabet font */
	LBP_move( x0, y0 );
	lprintf( "\033[1.v%c", (int)c );
    }
    else if( c > 255 ) {		/* Japanese font */
	/* get character's metric data */
	base = f->info->cc.klist;
	charwidth = base->cinfo->width;	/* default metrics */
	charhight = base->cinfo->hight;
	chardepth = base->cinfo->depth;
	charnglue = base->cinfo->nglue;
	top = 0;
	bottom = f->info->cnum - 1;
	if( (base + bottom)->kcode > (uint2)c ) {
	    /* code is in range of the table --> do search */
	    found = FALSE;
	    while( (bottom - top) > 1 ) {
		idx = (top + bottom) / 2;
		if( (base + idx)->kcode == (uint2)c ) {
		    found = TRUE;
		    break;
		}
		else if( (base + idx)->kcode < (uint2)c )
		    top = idx;
		else
		    bottom = idx;
	    }
	    if( found ) {
#ifdef	SQUARE
		charwidth = (base + idx)->cinfo->width;
		charhight = (base + idx)->cinfo->hight;
#endif
		chardepth = (base + idx)->cinfo->depth;
		charnglue = (base + idx)->cinfo->nglue;
	    }
	}

	cg = (int4)((double)charnglue / 1048576.0 * (double)f->scale + 0.5);
#ifdef	SQUARE
	cw = (int)((double)charwidth / 1048576.0 * (double)f->scale + 0.5);
	ch = (int)((double)charhight / 1048576.0 * (double)f->scale + 0.5);
	cd = (int)((double)chardepth / 1048576.0 * (double)f->scale + 0.5);
	LBP_square( x0, y0, (int4)cw, (int4)ch, 1 );
	LBP_square( x0, y0, (int4)cw, (int4)-cd, 2 );
#endif	/* SQUARE */
	cd = (int4)((double)chardepth / 1048576.0 * DEPTH_ADJ *(double)f->scale + 0.5);
	LBP_move( x0 - cg, y0 + cd );
	lputchar( (int)((c >> 8) | 0x80) );
	lputchar( (int)((c & 0xff) | 0x80) );
    }
    else {				/* alphabet font */
	LBP_move( x0, y0 );
	lputchar( (int)c );
    }
}

/* select harJapanese font as its # */
void
LBP_numJfont( fno )
int	fno;	/* assigned font # */
{
    lprintf( "\033[%d;2;1%%v", fno );
}

/* select alphabet font as its # */
void
LBP_numAfont( fno )
int	fno;	/* assigned font # */
{
    lprintf( "\033[%d;1;1%%v", fno );
}

/* flush page */
void
LBP_flushpage()
{
    lputchar( 0x0c );
}
