/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/*
 * Here we have some procedures, which form the interface
 * between lex and yacc. We have two yaccs and lexs in our case,
 * one reading in PFD files, the other one AFM files.
 *
 * In the Makefile, lex and yacc related tokens are arranged in the following
 * way:
 * yy...: afm file stuff
 * YY...: pfd file stuff
 * Yy...: defenc file stuff
 *
 * Observe, that yywrap() and yyerror() are the only two cases, where
 * this mapping to upper case letters for the pfd lexical analyser
 * does not take place!
 */

#include <stdio.h>
#include "defs.h"
#include "pfd2tfm.h"
#include "extfil.h"

extern  char * ProgName;
extern  char * StrcpyAlloc();

/* Which lexical analyser are we using. */
int     WhichLex = NO_LEX;

/* Flag to indicate when the lexical analyser is done. */
int     LexDone = FALSE;

/* Debugging flag for the lexical analyser: prints symbolic tokens. */
/* Set by SetupLex */
int     LexDebug;

/* Token counter */
int     LexTokenCount = 0;

/* Token number for last token read, for error messages */
int	LastTokenRead;

/* Character last returned by LCGetChar */
char    LexLastChar;

/* Maximum of consecutive unputc's by the lexical analyser. */
#define UNPUTMAX     4

/* Hold unput characters here */
char    UnputChar[UNPUTMAX];

/* How many characters are in the 'unput' array. */
int UnputCount;

int GetCharEof; /* TRUE if LCGetChar has read the last character in the file. */

int LexLineNumber;

/* File pointer of file, from which the input is read */
EX_FILES_P ExLexFile;

char LCGetChar();

/* How many lines do you want to write out in case of an error
 * (this provides the textual context of where the error occured).
 */
#define SAVE_LINES 7
char SaveLine[SAVE_LINES][256]; /* [line#][character index], does not
				   include \n */
int SaveLineNumber;     /* line number of current line */
int SaveLineCharIndex;  /* character index within that line */

/*
 * MYlex
 * *****
 * Yacc calls this procedure, which calls the lexical analyzer itself.
 *
 * RET: the next token.
 */
int
Mylex()
{
  int     ret;
  
  /* Get token */
  switch (WhichLex) {
    case AFM_LEX:
      ret = yylex_afm();
      break;
    case PFD_LEX:
      ret = YYlex_pfd();
      break;
    case DEFENC_LEX:
      ret = Yylex_defenc();
      break;
    default:
      Fatal ("Mylex(): default");
    }

  LastTokenRead = ret;
  LexTokenCount++;
  if (LexDebug) {
    PrintToken (stderr, ret);
    printf ("\n");
  }
  return (ret);
}   /* MYlex() */

/*
 * SetupLex
 * ********
 * Set up the lexical analyser. 
 *
 * file:     pointer to already open file.
 * lex:      controls which lexical analyser is used.
 * token_p:  print tokens (TRUE)
 */
SetupLex (ex_fp, lex, token_p)
     EX_FILES_P ex_fp;
     int lex;
     int token_p;
{
  extern FILE * yyin;
  extern FILE * YYin;
  int i;

  ExLexFile = ex_fp;
  yyin = ex_fp->ef_filep;
  YYin = ex_fp->ef_filep;
  WhichLex = lex;
  LexLineNumber = 1;
  LexDone = FALSE;
  LexTokenCount = 0;
  LastTokenRead = -1;
  UnputCount = 0;
  GetCharEof = FALSE;
  LexDebug = token_p;
  SaveLineNumber = 1;
  SaveLineCharIndex = 0;
  for (i=0; i<SAVE_LINES; i++)
    SaveLine[i][0] = '\0';
}

/*
 * LCUnputChar(c)
 * **************
 * Unput operation.
 *
 * c: the character to unput.
 */
unput(c) char c;{LCUnputChar(c);}
LCUnputChar(c)
     char        c;
{
  if (UnputCount == UNPUTMAX)
    Fatal ("LCUnputChar(): no more unputs!");
  UnputChar[UnputCount++] = c;
}

/*
 * LCGetChar
 * *********
 * Get character from file (unless there are leftover
 * characters due to unputs). This procedure is called
 * by the lexical analysers. 
 */
char
input()
{
  return LCGetChar();
}

char
LCGetChar()
{
  int ret;

  /* 'Left over' character? */
  if (UnputCount != 0)
    return (LexLastChar = UnputChar[--UnputCount]);

  if (GetCharEof) {
    LexDone = TRUE;
    return (0);
  }

  /* Get character from file */
  if ((ret = getc(ExLexFile->ef_filep)) == '\n') {
    LexLineNumber++;
    SaveLine[SaveLineNumber%SAVE_LINES][SaveLineCharIndex] = '\0';
    SaveLineCharIndex = 0;
    SaveLineNumber++;
  } else {
    SaveLine[SaveLineNumber%SAVE_LINES][SaveLineCharIndex++] = ret;
  }
  if (ret == EOF) {
    GetCharEof = TRUE;
    return (0);
  }

  return (LexLastChar = ret);
}   /* LCGetChar() */

/*
 * PrintToken
 * **********
 * Prints either token types - AFM or PFD files
 *
 * fp: where output goes.
 * token: the token to print.
 */
PrintToken(fp, token)
     FILE * fp;
     int token;
{
  fprintf (fp, "%4d: ", LexTokenCount);
  switch (WhichLex) {
    case AFM_LEX:
      PrintTokenAfm(fp, token);
      break;
    case PFD_LEX:
      PrintTokenPfd(fp, token);
      break;
    case DEFENC_LEX:
      fprintf (stderr, "No token (defenc-lex)");
      break;
    default:
      Fatal ("PrintToken(): illegal lexical analyser");
    }
}

/* Observe, that yywrap() and yyerror() are the only two cases, where
 * this mapping to upper case letters for the pfd lexical analyser
 * does not take place! */

/*
 * yyerror
 * *******
 * s: error message
 */
yyerror(s)
    char *s;
{
  int c, i;

  fflush(stdout);
  fflush(stderr);

  fprintf (stderr, "%s: yacc error: %s\n", ProgName, s);
  fprintf (stderr, "Last token read: ");
  PrintToken(stderr, LastTokenRead);
  fprintf (stderr, ", line %d in \"%s\"\n", LexLineNumber, ExLexFile->ef_fn);
  fprintf (stderr, "Now showing couple of the last lines read:\n");

  /* Now dump the last c lines to give the user an idea of where the
   * error occured. */
  c = (SaveLineNumber >= SAVE_LINES) ? SAVE_LINES : SaveLineNumber;
  c--; /* number of complete lines in the buffer */
  for (i=c; i>=1; i--)
    fprintf (stderr, "%3d: %s\n", 
	     SaveLineNumber - i,
	     SaveLine[(SaveLineNumber - i + SAVE_LINES) % SAVE_LINES]);
  if (SaveLineCharIndex > 0) {
    fprintf (stderr, "%3d: ", SaveLineNumber);
    for (i=0; i<SaveLineCharIndex; i++)
      putc(SaveLine[SaveLineNumber%SAVE_LINES][i], stderr);
    fprintf (stderr, "\n");
  }

  Fatal ("yyerror(): Compiler Error");
}

/*
 * yywrap/YYwrap
 * *************
 * This procedure is called at the end by the pfd/afm lexical analyser.
 */
yywrap(){  return (1); }
YYwrap(){  return (1); }
