/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Procedures to read from the dvi file: postamble, preamble, font defs. */

#include <stdio.h>
#include "units.h"
#include "dvitps.h"
#include "extfil.h"
#include "fontd.h"
#include "dvi-com.h"
#include "defs.h"

extern double sqrt();
extern char *StrcpyAlloc();
extern char *Malloc();
extern void ReportFontFileTypes();
extern int DviNum;
extern int DviDen;
extern int DviMag;
extern int HConv;
extern FE_P Fonts[MAX_FONTS];
extern int LastPP;
extern int FirstPP;
extern int Resolution;
extern int Verbose;
extern FFP FontFilePriorities[4];
extern void ReadFontDef();

EX_FILES ExDvi; /* Currently active .dvi file. */

int FontsFindError; /* Something went wrong, when the program tried
		       to find pixel or pdr files. */
int ChecksumError;  /* Checksum errors. */

int CurFontNumber;

int DviPostAmbleOffset; /* Offset of the postamble within dvi file. */


/*
 * FindDviPostAmbleOffset
 * **********************
 * Find the postamble pointer into the current dvi file.
 */
void
FindDviPostAmbleOffset()
{
  int     i;

  FExSeek (&ExDvi, 0, FSEEK_END);   /* Goto end of file. */
  DviPostAmbleOffset = FExTell (&ExDvi) - 4;
  FExSeek (&ExDvi, DviPostAmbleOffset, FSEEK_ABS);

  while (TRUE) {
    FExSeek (&ExDvi, --DviPostAmbleOffset, FSEEK_ABS);
    if (((i = NoSignExtend(EX_FP(ExDvi), 1)) != 223) &&
	(i != DVI_FILE_ID))
      Fatal ("FindDviPostAmbleOffset(): Bad end of DVI file.");
    if (i == DVI_FILE_ID)
      break;
  }
  FExSeek (&ExDvi, DviPostAmbleOffset - 4, FSEEK_ABS);
  DviPostAmbleOffset = NoSignExtend(EX_FP(ExDvi), 4);
  FExSeek (&ExDvi, DviPostAmbleOffset, FSEEK_ABS);
}

/*
 * GetFontDefs
 * ***********
 * Get all the font definitions out of the postamble of the dvi file.
 * Routine does not reposition dvi file after use.
 */
void
GetFontDefs()
{
  int command;
  int i;

  FontsFindError = FALSE;
  ChecksumError  = FALSE;

#ifdef DEBUG
  fprintf (stderr, "%% GetFontDefs()\n");
#endif
  while (((command = NoSignExtend(EX_FP(ExDvi), 1)) >= DVI_FNT_DEF1) &&
	 (command <= DVI_FNT_DEF4))
    ReadFontDef (NoSignExtend(EX_FP(ExDvi), command-DVI_FNT_DEF1+1));

  if (command != DVI_POST_POST)
    Fatal ("GetFontDefs(): no DVI_POST_POST after the font definitions.");

  if (FontsFindError) {
    for (i=0; i<=CF_NONE; i++) {
      if (FontFilePriorities[i].ffp_p == -1)
	continue;
    }
    ReportFontFileTypes();
    Fatal ("GetFontDefs(): Unresolvable font files");
  }

  if (ChecksumError)
    Warning ("GetFontDefs(): Checksum error(s)");
#ifdef DEBUG
  fprintf (stderr, "%% GetFontDefs(): done\n");
#endif
}

/*
 * ReadDviPostAmble 
 * ****************
 * Read in the postamble of the dvi file. Sets as side
 * effect LastPP, the offset of the last page within the dvi file.
 * No parameters or such: ExDvi is expected to define the current
 * dvi file.
 */
void
ReadDviPostAmble()
{
#ifdef DEBUG
  fprintf (stderr, "%% ReadDviPostAmble():\n");
#endif
  FindDviPostAmbleOffset ();
  if (NoSignExtend (EX_FP(ExDvi), 1) != DVI_POST)
    Fatal ("ReadDviPostAmble(): DVI_POST missing at head of postamble");
  LastPP = NoSignExtend(EX_FP(ExDvi), 4);
  NoSignExtend(EX_FP(ExDvi), 4); /* num: already in preamble */
  NoSignExtend(EX_FP(ExDvi), 4); /* den */
  NoSignExtend(EX_FP(ExDvi), 4); /* mag */
  NoSignExtend(EX_FP(ExDvi), 4); /* height-plus-depth of tallest page */
  NoSignExtend(EX_FP(ExDvi), 4); /* width of widest page */
  if (NoSignExtend(EX_FP(ExDvi), 2) >= MAX_STACK_SIZE)
    Fatal ("ReadDviPostAmble(): Stack size is too small");
  NoSignExtend(EX_FP(ExDvi), 2); /* Number of pages in DVI file */
  GetFontDefs (); /* Finally get all font definitions */
}

/*
 * ReadDviPreAmble
 * ***************
 * Read the preamble of the dvi file. As side effect get the
 * offset of the first page in the dvi file.
 *
 * FirstPP, a pointer to the first page, is set as side effect
 * of this procedure.
 */
void
ReadDviPreAmble()
{
  char buffer[256];

  /* Dvi file initial stuff */
  if (NoSignExtend (EX_FP(ExDvi), 1) != DVI_PRE)
    Fatal ("ReadDviPreAmble(): No preamble. Is this a DVI file?");
  if (SignExtend (EX_FP(ExDvi), 1) != DVI_FILE_ID)
    Fatal ("ReadDviPreAmble(): Wrong dvi file id");
  DviNum = NoSignExtend(EX_FP(ExDvi), 4);
  DviDen = NoSignExtend(EX_FP(ExDvi), 4);
  /* Value later used in ConversionFactors(). */
  DviMag = NoSignExtend(EX_FP(ExDvi), 4);

  /* This is the text stored in every dvi file which identifies
     the origin of the dvi file. */
  ReadIntoBuffer (EX_FP(ExDvi), buffer, 1, 256);
#ifdef DEBUG
  fprintf (stderr, "%% ReadDviPreAmble(): \"%s\"\n", buffer);
#endif
  FirstPP = FExTell (&ExDvi);
}
