/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

#include <stdio.h>
#if SYS_V == 1
#include <string.h>
#else
#include <strings.h>
#endif
#include "tfm.h"
#include "extfil.h"

#define TRUE 1
#define FALSE 0

extern char *Malloc();
extern char *LocateFileWithPath();
extern char *HandleFileNameExtension();

TFM_S_P ReadTfmFile();

/*
 * ReadTfmFileUsePath
 * ******************
 * Read in a tfm file.
 *
 * fn: file name of the tfm file. Extension "tfm" optional.
 * path1: use this path to locate the tfm file. If NULL use path2.
 * path2: second path to use to locate the tfm file.
 * RETURN: pointer to a data structure TFM_S allocated by this procedure
 *         which contains the tfm file data.
 */
TFM_S_P
ReadTfmFileUsePath(fn, path1, path2)
     char *fn;
     char *path1;
     char *path2;
{
  char *fn_tfm; /* Store file name with extension .tfm here. */
  char *fn_expanded; /* Store here the "expanded tfm file name". */

  fn_tfm = HandleFileNameExtension(1, fn, "tfm");
  fn_expanded = LocateFileWithPath(fn_tfm, path1, path2, FALSE);
  if (fn_expanded == NULL)
    Fatal2("ReadTfmFileUsePath(): could not locate tfm file \"%s\"", fn);
  return (ReadTfmFile(fn_expanded));
}

/*
 * ReadTfmFile
 * ***********
 * Read in a tfm file.
 *
 * fn: file name of the tfm file. File extension may be present or absent.
 * RETURN: pointer to a data structure TFM_S allocated by this procedure
 *         which contains the tfm file data.
 */
TFM_S_P
ReadTfmFile(fn)
     char *fn;
{
  EX_FILES ex_f;

  int i;
  int code;

  /* The following variables are loaded from the tfm file. */ 
  double d_designsize;	/* as floating point number		*/

  /* The char_info, width, height and depth table table of a TFM file
   * The tfm width information is saved in the tfm_width field of the
   * font_entry structure of the current font. */
  int char_info[MAX_CHAR_CODE_IN_TFM_FILE];

  /* Width table from the tfm file, also as floating point number. */
  int width[MAX_CHAR_CODE_IN_TFM_FILE];

  double d_font_space;

  /* Here we store the result of our loading. */
  TFM_S_P tfm;

  /* Allocate space for the tfm file and clear the character definitions. */
  tfm = (TFM_S_P) Malloc (sizeof (TFM_S));
  for (i=0; i<MAX_CHAR_CODE_IN_TFM_FILE; i++)
    tfm->tfm_c_def[i] = FALSE;

  /* Open tfm file. */
  FExOpen(&ex_f, EFT_READ, EFQ_NO_STDIN, fn, "tfm");

  /* Get the header first. */
  tfm->tfm_lf = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_lh = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_bc = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_ec = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_nw = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_nh = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_nd = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_ni = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_nl = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_nk = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_ne = NoSignExtend (EX_FP(ex_f), 2);
  tfm->tfm_np = NoSignExtend (EX_FP(ex_f), 2);
    
#ifdef	DEBUG
  fprintf (stderr, "%% lf = %d\n", tfm->tfm_lf);
  fprintf (stderr, "%% lh = %d\n", tfm->tfm_lh);
  fprintf (stderr, "%% bc = %d\n", tfm->tfm_bc);
  fprintf (stderr, "%% ec = %d\n", tfm->tfm_ec);
  fprintf (stderr, "%% nw = %d\n", tfm->tfm_nw);
  fprintf (stderr, "%% nh = %d\n", tfm->tfm_nh);
  fprintf (stderr, "%% nd = %d\n", tfm->tfm_nd);
  fprintf (stderr, "%% ni = %d\n", tfm->tfm_ni);
  fprintf (stderr, "%% nl = %d\n", tfm->tfm_nl);
  fprintf (stderr, "%% nk = %d\n", tfm->tfm_nk);
  fprintf (stderr, "%% ne = %d\n", tfm->tfm_ne);
  fprintf (stderr, "%% np = %d\n", tfm->tfm_np);
#endif

  if (tfm->tfm_ec >= MAX_CHAR_CODE_IN_TFM_FILE)
    Fatal ("ReadTfmFileInfo(): MAX_CHAR_CODE_IN_TFM_FILE too small.");

  /* These are trivial checks (see description of tfm files)
   * just to make sure that everything is ok with reading in. */
  if (tfm->tfm_lf !=
      (6 + tfm->tfm_lh + (tfm->tfm_ec - tfm->tfm_bc + 1) +
       tfm->tfm_nw + tfm->tfm_nh + tfm->tfm_nd + tfm->tfm_ni + tfm->tfm_nl +
       tfm->tfm_nk + tfm->tfm_ne + tfm->tfm_np))
    Fatal ("ReadTfmFileInfo(): sum test failed.");
  if ((tfm->tfm_nh>16) || (tfm->tfm_nd>>16) || (tfm->tfm_ni>64))
    Fatal ("ReadTfmFileInfo(): error in array lengths.");

  /* We now get the first two header words (the rest of the header, if
   * there is such information) will be skipped. Get checksum first, then
   * design size. */
  tfm->tfm_cs = NoSignExtend (EX_FP(ex_f), 4); /* header[0] */
  tfm->tfm_designsize = NoSignExtend (EX_FP(ex_f), 4); /* header[1] */
  d_designsize = FIXES_TO_FLOAT(tfm->tfm_designsize);
  tfm->tfm_designsize_sp = (int) (d_designsize * 65536.0);
#ifdef	DEBUG
  fprintf (stderr, "%% checksum = %d\n", tfm->tfm_cs);
  fprintf (stderr, "%% designsize = %5.2lfpt\n", d_designsize);
#endif

  /* Skip rest of header, to the beginning of char_info. */
  FExSeek (&ex_f, (tfm->tfm_lh-2)*4, FSEEK_REL);
     
  /* Now reading char_info, the character information. */
  for (i=tfm->tfm_bc; i<= tfm->tfm_ec; i++)	{
    char_info[i] = NoSignExtend (EX_FP(ex_f), 4);
#ifdef	DEBUG
    fprintf (stderr, "ch: %3d, w_i: %3d, h_i: %3d, i_i: %3d, re: %3d\n",
	    i,
	    GET_WIDTH_INDEX(char_info[i]),
	    GET_HEIGHT_INDEX(char_info[i]),
	    GET_ITALIC_INDEX(char_info[i]),
	    GET_REMAINDER(char_info[i]));
#endif
  }
  
  /* Now reading width table of the tfm file. */
#ifdef	DEBUG
  fprintf (stderr, "\nWidth Table:\n\n");
#endif
  for (i=0; i<=tfm->tfm_nw-1; i++)	{
    width[i] = NoSignExtend (EX_FP(ex_f), 4);
#ifdef	DEBUG
    fprintf (stderr, "i: %3d, width: %3d, d_width: %5.3lfpt\n",
	    i, width[i], FIXES_TO_FLOAT(width[i])*d_designsize);
#endif
  }

  /* Skip to the end of the file, where you find param.
   * Read param[2], which contains the amount of space of the font.
   * See [547] in Volume B of Computers & Typesetting (TeX Program).
   * We skip everything from the beginning, and then one more byte
   * of the param file, to access the italics stuff. */
  FExSeek (&ex_f, (6 + tfm->tfm_lh + (tfm->tfm_ec - tfm->tfm_bc + 1) +
		   tfm->tfm_nw + tfm->tfm_nh +
		   tfm->tfm_nd + tfm->tfm_ni +
		   tfm->tfm_nl + tfm->tfm_nk +
		   tfm->tfm_ne + 1)*4, FSEEK_ABS);
  tfm->tfm_space = NoSignExtend(EX_FP(ex_f), 4);
  d_font_space = tfm->tfm_space;

  /* Load the tfm structure width information. */
#ifdef	DEBUG
  fprintf (stderr, "\nCharacter Width relationship\n\n");
#endif

  for (code=tfm->tfm_bc; code<= tfm->tfm_ec; code++)	{
    if (GET_WIDTH_INDEX(char_info[code]) != 0)	{
      /* Character is defined. */
      tfm->tfm_c_def[code] = TRUE;
      tfm->tfm_c_width[code] =
	width[GET_WIDTH_INDEX(char_info[code])];
      tfm->tfm_c_wsp[code] = FIXES_TO_FLOAT(tfm->tfm_c_width[code]) * tfm->tfm_designsize_sp;
#ifdef	DEBUG
      fprintf (stderr, "%% code = '%3o, width = %10d, %8.4lfpt\n",
	       code, tfm->tfm_c_width[code],
	       FIXES_TO_FLOAT(tfm->tfm_c_width[code]));
#endif
    } else
      tfm->tfm_c_def[code] = FALSE;
  } /* for */

  /* Close the .tfm file now, return a pointer to the data structure
     just filled. */
  FExClose(&ex_f);
  return (tfm);
}

