/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/*
 * This file contains some useful procedures for file handling.
 */

#include <stdio.h>
#if SYS_V == 1
#include <string.h>
#else
#include <strings.h>
#endif

#if SYS_V == 1
#define index  strchr
#define rindex strrchr
#endif

#define TRUE 1
#define FALSE 0

extern char * Malloc();
extern FILE * fopen();
extern char * getenv();
extern char * StrcpyAlloc();
extern char * StrncpyAlloc();
extern void Fatal();
extern void Fatal2();
extern void Fatal3();

extern char * ProgNameS; /* Name of program (short, base name, not
			    full path name) */

/*
 * BytesIntoBuffer
 * ***************
 * Load a bunch of bytes, from a file, into a buffer.
 *
 * fp: file
 * cp: where to store
 * n: number of bytes to get.
 */
void
BytesIntoBuffer(fp, cp, n)
    register FILE *fp;
    register char *cp;
    register int n;
{
  while (n--)
    *cp++ = getc(fp);
}

/*
 * FileNameNormalization
 * *********************
 * Given a file name fn, remove any redundant "./" from
 * this file name. Return the normalized file name
 * and also update the argument itself.
 */
char *
FileNameNormalization(fn)
     char *fn;
{
  int ln;
  int i, k;

  if ((ln = Strlen(fn)) == 0)
    Fatal ("FileNameNormalization: empty file name.");
  if (ln < 2)
    return(fn);

  /* Search for every occurence of "./" and throw it out. */
  for (i=0;;) {
    /* If at the end: return. */
    if (! *(fn+i))
      return(fn);
    /* Is there a "./" sequence? */
    if (*(fn+i) == '.' && *(fn+i+1) == '/') {
      /* Yes, remove it. */
      k = i;
      while (*(fn+k)) {
	*(fn+k) = *(fn+k+2);
	k++;
      }
    } else
      i++; /* No, look at the next character. */
  }
}

/* 
 * GetBytes
 * ********
 * Get a bunch of bytes from a file. Append '\0' -- that's useful for strings.
 * If n==0 then we get an empty string.
 *
 * fp: file pointer
 * n: number of bytes to get
 * RET: where the whole stuff is stored. Space allocated by this routine.
 */
char *
GetBytes(fp, n)
     register FILE *fp;
     register int n;
{
  char * ret;
  char * ret2;

  ret2 = ret = Malloc(n+1);
  *(ret+n) = '\0';
  while (n--)
    *ret++ = getc(fp);
  return (ret2);
}

/*
 * GetEnv
 * ******
 * Given a environment variable, check whether it is set and if
 * so return it, otherwise take the default and return it.
 *
 * env: the environment variable to test.
 * def: default to be used if not set.
 * RET: whatever string to use.
 */
char*
GetEnv (env, def)
     char * env;
     char * def;
{  
  char * ret;
  if ((ret=getenv(env)) == NULL)
    return (def);
  else
    return (ret);
}

/*
 * NoSignExtend
 * ************
 * Get a bunch of bytes from a file, no sign extension.
 *
 * fp: file pointer
 * n:  number of bytes to read
 */
int
NoSignExtend(fp, n)
     register FILE *fp;
     register int n;
{
  register int x;	/* number being constructed */

  x = 0;
  while (n--)  {
    x <<= 8;
    x |= getc(fp);
  }
  return(x);
}

/*
 * ReadIntoBuffer
 * **************
 * Read information into a buffer. This routine follows the
 * way of how variable length items are typically stored in
 * TeX and Metafont related file types. At the end of the buffer
 * a \0 is appended.
 *
 * file:   file from where to read.
 * buffer: where to read to.
 * ll:     length of the length code in the file (1 through 4).
 * maxl:   length of buffer (read at most that many bytes).
 *
 * RET:	   number of bytes actually read in. That is one less than
 *         actually allocated because of the trailing \0 which is appended.
 */
int
ReadIntoBuffer (file, buffer, ll, maxl)
     FILE *file;
     char *buffer;
     int ll;
     int maxl;
{
  int length;
  int i;

  if (ll<1 || ll>4)
    Fatal ("ReadIntoBuffer(): illegal length code, not in range [1..4]");
  /* When checking for length: must be able to append '\0' */
  if ((length=NoSignExtend (file, ll)) < 0)
    Fatal ("ReadIntoBuffer(): length negative");
  if (length-1 >= maxl)
    Fatal ("ReadIntoBuffer(): provided buffer too small");
  for (i=0; i<length; i++)
    *(buffer+i) = NoSignExtend(file, 1);
  *(buffer+length) = '\0';

  return (length);
}

/*
 * ReadLineIntoBuffer
 * ******************
 * Read a line into a buffer. Return the number of characters
 * read in (the \n does not count and has been stripped off).
 * There is a \0 at the end of the line). Return EOF if done.
 * Generate a fatal error if length of buffer provided
 * is insufficient.
 *
 * f: file from where to read.
 * buffer: buffer address where result is to be stored.
 * length: maximum number of characters which can be stored in 'buffer'.
 * RET: EOF if done, otherwise the number of characters read in.
 */
int
ReadLineIntoBuffer (f, buffer, length)
     FILE * f;
     char * buffer;
     int length;
{
  int i;
  int c;

  i = 0;
  buffer[0] = '\0'; /* In case it's an empty line. */
  while ((c=getc(f)) != EOF) {
    buffer[i] = c;
    if (c == '\n') {
      buffer[i] = '\0';
      return (i);
    }
    if (++i >= length-1)
      Fatal ("ReadLineIntoBuffer(): buffer too small.");
  }
  return (EOF);
}

/*
 * ReadQuotedStringIntoBuffer
 * **************************
 * Read a quoted string "abc def" into a buffer.
 * Don't return quotes, replace multiple spaces, newlines
 * and returns by one single space.
 *
 * f: file to read from
 * buffer: where to store result
 * length: maximum length of buffer. Generates fatal error
 *         if exceeded.
 * RET: EOF if done, otherwise the number of
 *          characters stored in the buffer.
 */
int
ReadQuotedStringIntoBuffer(f, buffer, length)
     FILE *f;
     char buffer[1];
     int length;
{
  int c; /* character being read in */
  int i; /* index into buffer */
  int state; /* 0: skipping initial space, and ",
		1: looking for first chararcter after opening quote.
		2: reading characters, simple white space so far
		3: if another white space character, ignore
		*/
  buffer[0] = '\0';
  state = 0;
  i = 0;

  while ((c=getc(f)) != EOF) {
    switch_again:
    switch (state) {
      case 0:
        if (c == ' ' || c == '\t' || c == '\n')
	  continue;
	if (c != '"')
	  Fatal ("ReadQuotedStringIntoBuffer(): expected quote, not found.");
	state = 1;
	break;

      case 1:
	if (c == ' ' || c == '\t' || c == '\n')
	  continue;
	if (c == '"') { /* input was "" */
	  buffer[0] = '\0';
	  return(0);
	}
	state = 2;
	buffer[i++] = c;
	break;

      case 2:
	if (c == ' ' || c == '\t' || c == '\n') {
	  buffer[i++] = ' ';
	  if (i-1 >= length)
	    Fatal ("ReadQuotedStringIntoBuffer(): buffer length insufficient.");
	  state = 3; /* ignore future white space */
	  continue;
	}
	if (c == '"') { /* Terminating double quote */
	  buffer[i] = '\0';
	  return (i);
	}
	  
	buffer[i++] = c;
	if (i-1 >= length)
	  Fatal ("ReadQuotedStringIntoBuffer(): buffer length insufficient.");
	break;

      case 3:
	if (c == ' ' || c == '\t' || c == '\n')
	  continue;
	state = 2; /* Not a space, revert to state two and scan the character again */
	goto switch_again;
    }
  }
  buffer[i] = '\0';
  return (EOF); /* Premature end! */
}

/*
 * SignExtend
 * **********
 * Read a bunch of bytes from a file, perform sign extension.
 * fp: file to read from
 * n: number of bytes to read
 */
int
SignExtend (fp, n)
     register FILE *fp;
     register int n;
{
  int n1;         /* Number of bytes */
  register int x; /* Number being constructed */

  x = getc(fp);   /* Get first (high-order) byte */
  n1 = n--;
  while (n--)  {
    x <<= 8;
    x |= getc(fp);
  }

  /* NOTE: This code assumes that the right-shift is an arithmetic, rather
   * than logical, shift which will propagate the sign bit right.  According
   * to Kernighan and Ritchie, this is compiler dependent!		*/

  x<<=32-8*n1;
  x>>=32-8*n1;  /* sign extend */

#ifdef DEBUG
  fprintf (stderr, "%% \tSignExtend(fp,%d)=%X\n",n1,x);
#endif
  return(x);
}

/*
 * WriteBytes
 * **********
 * Write n bytes (n=1, 2, 4) to a file.
 *
 * file:     file to be used for output.
 * val:      the value to be written.
 * length:   how many bytes (1, 2, 3, 4).
 */
void
WriteBytes (file, val, length)
     FILE * file;
     int val;
     int length;
{
#ifdef DEBUG
  fprintf (stderr, "%% WriteBytes(): %d: %d\n", length, val);
#endif
  switch (length)	{
    case 1:
      putc (val, file);
      break;
    case 2:
      putc ((val>>8) & 0xff, file);
      putc (val&0xFF, file);
      break;
    case 3:
      putc ((val>>16) & 0xFF, file);
      putc ((val>>8) & 0xFF,  file);
      putc (val & 0xFF,       file);
      break;
    case 4:
      putc ((val>>24) & 0xFF, file);
      putc ((val>>16) & 0xFF, file);
      putc ((val>>8) & 0xFF,  file);
      putc (val & 0xFF,       file);
      break;
    default:
      Fatal ("WriteBytes(): illegal length");
    }
}


/*
 * WriteMultipleChars
 * ******************
 * Write to file f character c n times.
 *
 * f: FILE* for output.
 * c: character to repeat.
 * n: how many times;
 */
void
WriteMultipleChars(f, c, n)
     FILE *f;
     int c;
     int n;
{
  int i;

  for (i=1; i<=n; i++)
    fputc(c, f);
}

/*
 * WriteString
 * ***********
 * Write a string to a file, in the format as it is used
 * usually in the context of \TeX: first write the length (le bytes)
 * and then write the string, without the trailing \0 from C.
 *
 * file: file to be used for output.
 * le:   length (1 to 4) of the encoding of the string length itself.
 * stl:  string length
 * st:   the string to be written out.
 */
void
WriteString (f, le, stl, st)
     FILE *f;
     int le;
     char *st;
     int stl;
{
  int i;

  /* Write out the length of the string itself, in the appropriate length itself. */
  WriteBytes (f, Strlen(st), le);
  /* Write the string itself. */
  for (i=0; i<stl; i++)
    WriteBytes (f, *(st+i), 1);
}

