/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/*
 * This file contains some useful procedures for file handling.
 */

#include <stdio.h>
#if SYS_V == 1
#include <string.h>
#else
#include <strings.h>
#endif
#include <sys/types.h>
#include <sys/file.h>
#include "extfil.h"

#define TRUE 1
#define FALSE 0

#if SYS_V == 1
#define index  strchr
#define rindex strrchr
#endif

extern char *ProgNameS; /* Name of program (short, base name, not
			   full path name). */
extern char *TmpDirForFiles;
extern int TmpFileNameCounter;

extern char *StrcpyAlloc();
extern char *StrncpyAlloc();
extern void Fatal();
extern void Fatal2();
extern void Fatal3();

/*
 * AppendFileToOpenFile
 * ********************
 * Open a file (given by its file name) and append
 * the content of this file to an already open file.
 *
 * ef_out: file to which is appended, already open.
 * fn_in: file name of the file which is being read in
 *        (not yet open).
 */
void
AppendFileToOpenFile (ef_out, fn_in)
     EX_FILES_P ef_out;
     char *fn_in;
{
  int c;
  EX_FILES ex_in;

  FExOpen (&ex_in, EFT_READ, 0, fn_in, "");
  while ((c=getc(EX_FP(ex_in))) != EOF)
    putc(c, ef_out->ef_filep);

  FExClose(&ex_in);
}

/*
 * AreTwoFilesIdentical
 * ********************
 * Compare two files and return TRUE or FALSE depending on whether
 * the two files are identical or not.
 *
 * fn1, fn2: file name of two existant files. 
 * RET: TRUE if the two files are identical.
 *      FALSE if they are differnt.
 */
int
AreTwoFilesIdentical(fn1, fn2)
     char *fn1;
     char *fn2;
{
  EX_FILES ex_f1;
  EX_FILES ex_f2;
  int ret;

  FExOpen(&ex_f1, EFT_READ, NULL, fn1, NULL);
  FExOpen(&ex_f2, EFT_READ, NULL, fn2, NULL);
  ret = AreTwoFilesIdenticalExFp(&ex_f1, &ex_f2);
  FExClose(&ex_f1);
  FExClose(&ex_f2);
  return ret;
}

/*
 * AreTwoFilesIdenticalExFp
 * ************************
 * Compare two files and return TRUE or FALSE depending on whether
 * the two files are identical or not.
 *
 * ex_f1, ex_f2: the two file pointers.
 * RET: TRUE if the two files are identical.
 *      FALSE if they are differnt.
 */
int
AreTwoFilesIdenticalExFp(ex_f1, ex_f2)
     EX_FILES_P ex_f1, ex_f2;
{
  int c1, c2;
  for (;;) {
    c1 = getc(EX_FP(*ex_f1));
    c2 = getc(EX_FP(*ex_f2));
    if (c1 == EOF && c2 == EOF)
      return TRUE;
    if (c1 != c2 ||
	(c1 == EOF && c2 != EOF) ||
	(c1 != EOF && c2 == EOF))
      return FALSE;
  }
}


/*
 * FileCopy
 * ********
 * Copy one file to another one. The two files are given by their
 * file names.
 *
 * fn_in:  input file name.
 * fn_out: output file name (can be "-" for stdout).
 */
void
FileCopy(fn_in, fn_out)
     char *fn_in;
     char *fn_out;
{
  EX_FILES ef_in, ef_out;
  int c;

  FExOpen(&ef_in,  EFT_READ,  NULL, fn_in, NULL);
  FExOpen(&ef_out, EFT_WRITE, NULL, fn_out, NULL);

  while ((c=getc(EX_FP(ef_in))) != EOF)
    putc(c, EX_FP(ef_out));

  FExClose(&ef_in);
  FExClose(&ef_out);
}

/*
 * LocateFileWithPath
 * ******************
 * Given the name of a file and two paths, try to locate the file.
 * The second path is used only, when the first path is NULL.
 * A legal path must look as follows: "xx:yy:zz".
 * Return "expanded", i.e. the complete file name. Return NULL if
 * the file could not be found.
 *
 * filename: file name: if it starts with '/', then don't use the path.
 *           Use the filename directly.
 * path1:   first path.
 * path2:   second path. Used only, when path1 == NULL.
 *          Both paths can not be empty at the same time.
 * show: (TRUE/FALSE): show every file which this procedure tried to open.
 *       Mark the file name when printed out by '*' if the file could be located.
 * RETURN
 *      the expanded file name.
 *
 */
char *
LocateFileWithPath(fn, path1, path2, show)
     char * fn;
     char * path1;
     char * path2;
     int show;
{
  char t[256];  /* Temporary file name buffer */
  char p[512];  /* Temporary path buffer */
  char * ptr, *ptr2;

  if (Strlen(path1) == 0 && Strlen(path2) == 0)
    Fatal ("LocateFileWithPath(): path1 and path2 are empty.");

  /* If file name starts with '/', no path is used. */
  if (fn[0] == '/') {
    strcpy (t, fn);
    if (access(t, R_OK) == 0)
      return(StrcpyAlloc(fn));
    else
      return (NULL);
  }

  /* Search using path. First decide which path to use! */
  strcpy (p, Strlen(path1) != 0 ? path1:path2);

#ifdef DEBUG
  fprintf (stderr, "%% LocateFileWithPath(): path being used: \"%s\"\n", p);
#endif

  ptr = p;

  /* Run through the path */
  while (*ptr != '\0') {
    /* ptr points at the first character of a directory in the path */
      /* ptr2 points at the next ':' or at '\0' */
    if ((ptr2 = index(ptr, ':')) == NULL)
      ptr2 = p+Strlen(p);
    t[0] = '\0';
    strncat (t, ptr, ptr2-ptr);
    strcat (t, "/");
    strcat (t, fn);
    if (show)
      fprintf (stderr, "\t%s", t);

    /* Now see whether such file exists. */
    if (access(t, R_OK) == 0) {
      if (show)
	fprintf (stderr, " <found>\n");
      return (StrcpyAlloc(t));
    } else {
      if (show)
	fprintf (stderr, "\n");
    }

    /* Not found, try next directory, if there is any left. */
    ptr = (*ptr2 == '\0' ? ptr2 : ptr2 + 1);
  } /* while */
  return (NULL); /* not found */
}


/*
 * FSeek
 * *****
 * Like fseek, but generates an error right away, if it fails.
 *
 * fp: File pointer.
 * offset: the offset of the seek operation.
 * direction: one of the FSEEK_ codes.
 */
void
FSeek(fp, offset, direction)
     FILE * fp;
     int offset;
     int direction;
{
  long l;

  l = offset;
  if (fseek(fp, l, direction) == -1) {
    Fatal ("FSeek(): failed");
  }
}

/*
 * FTell
 * *****
 * Like ftell, but generate an error, if it fails.
 *
 * fp: file pointer
 * RET: relative position, in file.
 */
int
FTell(fp)
     FILE * fp;
{
  int l;

  if ((l=ftell(fp)) < 0)
    Fatal ("FTell(): error.");
  return (l);
}

/*
 * GenerateTmpFileName
 * *******************
 * Generate a temporary file name and return it.
 *
 * ext: extension to be used for the file name. If empty use "tmp".
 *      Specify the extension without period.
 * RETURN: the generated file name.
 */
char*
GenerateTmpFileName(ext)
     char *ext;
{
  char buffer[256];

  if (Strlen(ext) == 0)
    ext = "tmp";
  if (index(ext, '.') != NULL)
    Fatal ("GenerateTmpFileName(): extension should have no period.");

  /* Increment the counter, generate the name, return it. */
  TmpFileNameCounter++;
  sprintf (buffer, "%s/%s-%d-%d.%s", TmpDirForFiles, ProgNameS,
	   getpid(), TmpFileNameCounter, ext);
  return(StrcpyAlloc(buffer));
}

/*
 * HandleFileNameExtension
 * ***********************
 * This routine handles some file extension business. If a file
 * extension is missing or is not the right one various actions
 * are performed. If the file name is "-" (for stdin or stdout)
 * this file should not have been called in the first place. Therefore
 * a fatal error will be generated in this case.
 *
 * control: 0: strip the extension off (RET). If present
 *             and not what required, generate fatal error.
 *          1: add the extension on to it (RET). If present
 *             and not what required, generate fatal error.
 *          2: same as 0, but if extension error, return NULL,
 *             not a fatal error.
 *          3: same as 1, but if extension error, return NULL,
 *             not a fatal error.
 * fn:    file name to be looked at.
 * ext:   file extension (excluding the '.').
 * RET:   return the generated file name.
 */
char*
HandleFileNameExtension (control, fn, ext)
     int control;
     char *ext;
     char *fn;
{
  char buffer[256];
  char *p_dot; /* Position of dot. */
  char *p_slash; /* Position of a '/' in a file namne. */

  if (Strlen(fn) == 0)
    Fatal ("HandleFileNameExtension(): empty file name.");
  if (Strcmp(fn, "-") == 0)
    Fatal ("HandleFileNameExtension(): file name is \"-\"");
  if (Strlen(ext) == 0)
    Fatal ("HandleFileNameExtension(): empty file extension.");

  strcpy (buffer, fn);

  p_dot = rindex(buffer, '.');
  p_slash = rindex(buffer, '/');

  /* Is there an extension? */
  if ((p_dot != NULL && p_slash == NULL) ||
      (p_dot != NULL && p_slash != NULL && p_dot > p_slash)) {
    /* Yes, see what we have to do. */
    if (strncmp(p_dot+1, ext, Strlen(ext)) != 0) { /* Extension does not match. */
      switch (control) {
        case 0:
        case 1:
	  Fatal3 ("HandleFileNameExtension(): illegal extension in \"%s\", expected \".%s\"",
		  fn, ext);
        case 2:
	case 3:
          return (NULL);
	  break;
	default:
	  Fatal("HandleFileNameExtension(): illegal case");
	}
    } /* fi */
    *p_dot = '\0'; /* Buffer now contains name without extension
		      which was just chopped off. */
  }

  /* buffer now contains file name without extension. */
  switch (control) {
    case 0:
    case 2:
      return (StrcpyAlloc(buffer));
    case 1:
    case 3:
      strcat (buffer, ".");
      strcat (buffer, ext);
      return (StrcpyAlloc(buffer));
    default:
      Fatal("HandleFileNameExtension(): illegal case");
  }
  return (0); /* lint */
}

