/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Decoding of arguments for the dvitps driver. */

#include <stdio.h>
#include <ctype.h>
#include "defs.h"
#include "dvitps.h"
#include "extfil.h"
#include "emit.h"

extern int FileBusinessVerbose;
extern int FileCachingVerbose;
extern char *StrcpyAlloc();
extern void Fatal();
extern void Fatal2();
extern int UsualOrientation;
extern char *ProgName;
extern char *ProgNameS;
extern char *optarg;
extern int optind;
extern int Resolution;
extern int WidthVectors;
extern double FudgeX;
extern double FudgeY;
extern int TpicIsEnabled;
extern int OrderOfOutput;
extern int AllMovementsAbsolute;
extern char * PrologueFilesPath;
extern char * dvitps_cap_dir;
extern int MaxPixelsWarning;
extern int ValidCapabilities;
extern char *DefaultPaperSpec;
extern int MagCommandLine; /* The magnification given on the command line, -m option.
			      Set to -1 if none is specified. */

/* Macro READ_ARG reads in all arguments for DecodeArgs(). Parameters are:
 * str: a "%d" or such to determine the format of the argument to be read in.
 * where: pointer to a variable where the value read in will be stored.
 */
#define READ_ARG(str, where)\
  if (sscanf(optarg, str, where) != 1)\
    Fatal2 ("DecodeArgs: option \"%c\": invalid argument", c);

int NumberOfCopies; /* Number of copies to print. */
int ManualFeed; /* Is manual feed in effect? */
char *PaperType; /* A string to specify the paper type. Sent to the PS printer. */
int QTestingCode; /* -Q option's numerical argument */

/* Offsets are given in inches, defaults are zero. */
double PageHOffset, PageVOffset;

/* Three levels of verbosity: V_QUIET, V_SOME, V_A_LOT. Output goes
 * to stderr. */
int Verbose = V_A_LOT;

/* Sometimes it's useful to ignore all \specials. */
int IgnoreSpecials;

/* Maximum number of arguments. */
#define MAX_NUM_ARGS 30

/* All Arguments (including those read in from ./dvitps-opt) are stored here. */
int Argc;
char * Argv[MAX_NUM_ARGS];

int ResetOption = FALSE;
int ResetPageNum; /* Reset every ResetPageNum page. */

/* For PostScript fonts only: if TRUE assume only character codes < 128.
   Otherwise character codes < 256 are legal (default). */
int SmallPsVectors;

int DraftOption; /* Print "Draft", filename, date and time on each page? */

int ListFontSearch; /* List which fonts have been opened. */

char *FontsSearchPathCommandLine; /* -A option: search paths! */

int ConformingDocument; /* Do you want to a "conforming document", default is NO. */

int FromPageNumber; /* -f: from which page we start to print. */
int ToPageNumber; /* -t: up to which page do we print. */

/*
 * DecodeArgs
 * **********
 * Decode arguments, main_argc and main_argv are from main. Return
 * with the list of DVI files which are then read in.
 *
 * main_argc: argument counter 
 * main_argv: argument values
 * RET: counter where you have to resume reading arguments from
 *      the command line to read in the file name of dvi files.
 */
int
DecodeArgs (main_argc, main_argv)
     int main_argc;
     char *main_argv[];
{
  char buffer[256];
  char tmp[256];
  int i;
  EX_FILES ex_f;
  int c; /* Option returned from getopt() */
  int stdin_input_requested; /* Input from stdin requested. */
  int ps_output_file_specified; /* PostScript output to a specified file requested. */

  MagCommandLine = -1; /* Means not set */
  FudgeX = 1.0;
  FudgeY = 1.0;
  PageHOffset = 0.0;
  PageVOffset = 0.0;
  NumberOfCopies = 1;
  ManualFeed = FALSE;
  Verbose = V_SOME;
  DraftOption = FALSE;
  SmallPsVectors = FALSE;
  ListFontSearch = FALSE;
  PaperType = DefaultPaperSpec;
  IgnoreSpecials = FALSE;
  FontsSearchPathCommandLine = NULL;
  ConformingDocument = FALSE;
  FromPageNumber = -1000000;
  ToPageNumber =    1000000;
  ps_output_file_specified = FALSE;
  stdin_input_requested = FALSE;

  /*
   * If there is a file ./.dvitps-opt then read in arguments from
   * this file and prepend it to the arguments from the command line when
   * the driver is invoked. The file name ./.dvitps-opt changes to
   * xx-opt, if you call the driver xx.
   */
  Argv[0] = StrcpyAlloc(ProgName);
  Argc = 1;
  sprintf (buffer, "%s-opt", ProgNameS);
  if (FExOpen(&ex_f, EFT_READ, EFQ_NO_STDIN | EFQ_NO_FILE_NO_ERROR, buffer, NULL)) {
    while (fscanf(EX_FP(ex_f), "%s", tmp) == 1) {
      Argv[Argc++] = StrcpyAlloc(tmp);
    }
    FExClose(&ex_f);
  }

  /*
   * Now tag the command line arguments on to the options read in
   * from .dvitps-opt (if that file was present). Skip Argv[0] when
   * you tag this on.
   */
  for (i=1; i<main_argc; i++)
    Argv[Argc++] = main_argv[i];
  Argv[Argc] = "";

#ifdef DEBUG
  for (i=0; i<Argc; i++)
    fprintf (stderr, "%% Argv[%d]: \"%s\"\n", i, Argv[i]);
  fflush(stderr);
#endif

  while ((c=getopt(Argc, Argv, "c:d:f:m:o:p:qt:vwz:A:B:CDFH:LMO:P:Q:R:SV:X:Y:Z")) != EOF) {
#ifdef DEBUG
    fprintf (stderr, "%% Option: \"%c\"\n", c);
    fflush (stderr);
#endif
    switch (c) {
      case '?':
        Fatal ("DecodeArgs(): illegal option.");

      /* '-': Read dvi file from stdin. */
      case '\0':
	if (stdin_input_requested)
	  Fatal ("DecodeArgs(): input from stdin requested before.");
	stdin_input_requested = TRUE;
	break;

      /* '-c': print multiple copies. */
      case 'c':
	READ_ARG("%d", &NumberOfCopies);
	break;

      /* -d: direction ('forward' or 'reverse') */
      case 'd':
	if (Strcmp(optarg, "forward") == 0) {
	  OrderOfOutput = ORDER_OUTPUT_FORWARD;
	  break;
	}
	if (Strcmp(optarg, "reverse") == 0) {
	  OrderOfOutput = ORDER_OUTPUT_REVERSE;
	  break;
	}
	Fatal2 ("DecodeArgs(): illegal direction specification \"%s\" in -d option",
		optarg);
	break;

      /* -f: from page number */
      case 'f':
	if (sscanf (optarg, "%d", &FromPageNumber) != 1)
	  Fatal2 ("DecodeArgs(): illegal page specification \"%s\", -f option",
		  optarg);
	if (FromPageNumber > ToPageNumber)
	  Fatal ("DecodeArgs(): illegal page range specification (-f/-t options)");
	break;

      /* -m magnification. Set the magnification factor which replaces the 
	 magnification factor in the dvi file. */
      case 'm':
	if (strcmp(optarg, "h") == 0)
	  MagCommandLine = 1095;
	else {
	  READ_ARG("%d", &MagCommandLine);
	  if (MagCommandLine < 0)
	    Fatal ("DecodeArgs(): -m option: negative magnification.");
	  switch(MagCommandLine) {
	    case 0: MagCommandLine = 1000; break;
	    case 1: MagCommandLine = 1200; break;
	    case 2: MagCommandLine = 1440; break;
	    case 3: MagCommandLine = 1728; break;
	    case 4: MagCommandLine = 2074; break;
	    case 5: MagCommandLine = 2488; break;
	    default:
	      ;
	    }
	}
	if (Verbose > V_QUIET)
	  fprintf (stderr, "Magnification (-m option): %d\n", MagCommandLine);
	break;

      /* -q: quiet. */
      case 'q':
	Verbose = V_QUIET;
	break;

      /* -t page-number: outpu tup to the specified page number. */
      case 't':
	if (sscanf (optarg, "%d", &ToPageNumber) != 1)
	  Fatal2 ("Decodeargs(): illegal page specification \"%s\", -t option",
		  optarg);
	if (FromPageNumber > ToPageNumber)
	  Fatal ("DecodeArgs(): illegal page range specification (-f/-t options)");
	break;

      /* '-o' paper-format: set the given paper format. */
      case 'o':
	PaperType = StrcpyAlloc(optarg);
	if (strcmp(PaperType, "letter") != 0 &&
	    strcmp(PaperType, "legal") != 0 &&
	    strcmp(PaperType, "note") != 0)
	  Fatal ("DecodeArgs(): -o option: illegal paper type\
 (must be letter, legal or note)");
	break;

      /* -v: verbose. */
      case 'v':
	Verbose = V_A_LOT;
	break;

      /* -w: No width vectors for PS fonts. */
      case 'w':
	WidthVectors = FALSE;
	break;

      /* '-z' page-number: reset VM in printer every page-number pages. */
      case 'z':
	ResetOption = TRUE;
	READ_ARG("%d", &ResetPageNum);
	if (ResetPageNum <= 0)
	  Fatal ("DecodeArgs(): Argument to -z must be > 0");
	break;

      /* -A path: specify font file search path. */
      case 'A':
	FontsSearchPathCommandLine = StrcpyAlloc(optarg);
	break;

      /* -B directory: directory to locate capability file. */
      case 'B':
	dvitps_cap_dir = StrcpyAlloc(optarg);
	break;

      /* -C: generate conforming document. */
      case 'C':
	ConformingDocument = TRUE;
	break;

      /* -D: print DRAFT, date and file-name on each page of the output. */
      case 'D':
	DraftOption = TRUE;
	break;
      
      /* -F: show font file openings. */
      case 'F':
	Verbose = V_A_LOT;
	ListFontSearch = TRUE;
	break;
	
      /* -H horizontal offset: shift horizontally by the specified amount! */
      case 'H':
	PageHOffset = ReadDimensionReturnSp(optarg) / 65536.0 / 72.27;
	break;

      /* -L: Landscape mode, applies to the whole document.
	 There is no opposite direction for portrait mode because
	 portrait mode is the default. */
      case 'L':
	UsualOrientation = LANDSCAPE_MODE;
	break;

      /* -M: Manual feed. */
      case 'M':
	ManualFeed = TRUE;
	break;

      /* -O file-name: specify name for output file. */
      case 'O':
	if (ps_output_file_specified != 0)
	  Fatal ("DecodeArgs(): -O option used twice.");
	InitPsOutputFiles(0, optarg);
	ps_output_file_specified = TRUE;
	break;

      /* -P: specifies path to locate prologue files. */
      case 'P':
	PrologueFilesPath = StrcpyAlloc(optarg);
	break;

      /* -Q #: testing option. */
      case 'Q':
	READ_ARG("%d", &QTestingCode);
	switch (QTestingCode) {
	  case 1:
	    /* This call here determines the file name for the output of
	       the subdivided PS output files. See the documentation for
	       further details. */
	    InitPsOutputFiles (1, "STATISTICS");
	    ps_output_file_specified = TRUE;
	    fprintf (stderr, "-Q1: statistics oriented output generated.\n");
	    break;
	  case 2:
	    AllMovementsAbsolute = TRUE;
	    fprintf (stderr, "-Q2: all movements are now absolute.\n");
	    break;
	  case 3:
	    FileBusinessVerbose = TRUE;
	    fprintf (stderr, "-Q3: file business verbose.\n");
	    break;
	  case 4:
	    FileCachingVerbose = TRUE;
	    fprintf (stderr, "-Q4: file caching verbose.\n");
	    break;
	  case 5:
	    MaxPixelsWarning = TRUE;
	    fprintf (stderr, "-Q5: WarningTfmVersusPxlWidth(): print messages\n");
	    break;
	  default:
	    Fatal ("DecodeArgs(): -Q / illegal #.");
	}
	break;

      /* '-R': Resolution, < 50: possibly error (I always confused this option it with '-r'
	 and so I thought one should catch that here!). */
      case 'R':
	READ_ARG("%d", &Resolution);
	if (Resolution < 50)
	  Fatal ("DecodeArgs(): -R resolution, where resolution < 50");
	break;

      /* -S: assume the character codes of characters of PostScript fonts you use
	 are always below 128. */
      case 'S':
	SmallPsVectors = TRUE;
	break;

      /* -X factor: fudge factor in the horizontal direction. */
      case 'X':
	READ_ARG("%lf", &FudgeX);
	break;

      /* -Y factor: fudge factor in the vertical direction. */
      case 'Y':
	READ_ARG("%lf", &FudgeY);
	break;

      /* -V offset: vertical offset by which the page is shifted. */
      case 'V':
	PageVOffset = ReadDimensionReturnSp(optarg) / 65536.0 / 72.27;
	break;

      /* -Z: ignore \special{...} temporarily. */
      case 'Z':
	IgnoreSpecials = TRUE;
	break;

      default: Fatal ("DecodeArgs(): default.");
    }	/* switch */
  }

  /* Index where you resume reading in Argv[] of main. */
#ifdef DEBUG
  fprintf (stderr, "%% optind: %d / Argc: %d\n", optind, Argc);
  fflush (stderr);
#endif

  /* Check for inconsistent options. */
  if (ConformingDocument && ResetOption)
    Fatal ("DecodeArgs(): can not have -C and -z options simultaneously.");

  /* Now you have to insert "-" as file name in case input from stdin was requested.
     And it must be first. */
  if (stdin_input_requested) {
    Argv[Argc+1] = NULL;
    for (i=optind+1; i<=Argc; i++)
      Argv[i] = Argv[i-1];
    Argv[optind] = StrcpyAlloc("-");
    Argc++;
  }

  /* Was the -O option used? If not, open stdout now. */
  if (! ps_output_file_specified)
    InitPsOutputFiles (0, "-");

  return (optind);
}

/* Usage text of dvitps. */
char *UsageText[] = {
  "\t[-c #-of-copies]",
  "\t[-O PostScript output file name] (output appears in specified file)",
  "\t[-H | -V Offset dimension] (horizontal and vertical offset)",
  "\t[-q] (quiet)      [-v] (verbose)",
  "\t[-d forward/reverse] (order: forward/reverse)",
  "\t[-f page] [-t page] (print from / to this page)",
  "\t[-m mag] (magnification: h(half), 0..5 (magsteps), other (TeX mag))",
  "\t[-o paper-format] (must be one of letter, legal or note)",
  "\t[-z reset-count] (reset after each page-count pages)",
  "\t[-w] (no width vector for PostScript fonts)",
  "\t[-A path] (search path for all font files)",
  "\t[-B directory] (directory for capability file)",
  "\t[-F] (show font file searches and openings)",
  "\t[-D] (print draft on top of each page)",
  "\t[-L] (Landscape)	[-M] (ManualFeed)",
  "\t[-P path] (search path for prologue files)",
  "\t[-R resolution] (resolution in dots/inch)",
  "\t[-S] (small: max character code is '177 (PostScript fonts only))",
  "\t[-C] (generate \"conforming document\" PostScript code)",
  "\t[-X | -Y fudge-factor] (fudge factor in X and Y direction)",
  "\t[-Q number] (testing option, idenfied by number, see documentation)",
  "\t[-Z (ignore \\specials, also disables tpic)",
  "\tfile[.dvi]... (or '-' for stdin)",
  NULL};

/*
 * Usage
 * *****
 * Print usage message.
 */
void
Usage()
{
  char **ptr;

  fprintf (stderr, "%s:  Version %s of %s\n", ProgName, VERSION, DATE);
  fprintf (stderr, "usage: %s\n", ProgName);
  ptr = &UsageText[0];
  while (*ptr != NULL)
    fprintf (stderr, "%s\n", *ptr++);
	   ProgName, VERSION, DATE, ProgName,
  fprintf (stderr, "\tOrder of output (default): %s\n",
	   OrderOfOutput==ORDER_OUTPUT_FORWARD ? "forward":"reverse");
  fprintf (stderr, "\tTpic is enabled: %s\n", TpicIsEnabled ? "Yes":"No");
  if (ValidCapabilities) {
    fprintf (stderr, "Driver looks for the following file types in the given order:\n");
    InitFontHandling(); /* This procedure contains the only "getenv()" calls */
    ReportFontFileTypes();
  } else {
    fprintf (stderr, "No capability file available.\n");
  }
  exit(0);
}

/*
 * StartMessage
 * ************
 * Standard message printed at the beginning of every job unless
 * '-q' option was given.
 */
void
StartMessage()
{
  if (Verbose > V_QUIET)
    fprintf (stderr, "%s, Version %s of %s\n", ProgName, VERSION, DATE);
}
