/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Conversion factor business */

#include <stdio.h>
#include "defs.h"
#include "units.h"
#include "dvitps.h"

extern DVIU MaxDriftHorizontal;
extern DVIU MaxDriftHorizontalNeg;
extern DVIU MaxDriftVertical;
extern DVIU MaxDriftVerticalNeg;

extern int Verbose;
extern int MagCommandLine;

int Resolution = DEFAULT_RES; /* Resolution in dots/inch. */

int DriverMag = 0; /* Magnification as valid for this driver
		      run. If 0, it's invalid. */

int MagCommandLine; /* The magnification given on the command line, -m option.
		       Set to -1 if none is specified. */
int DviMag; /* Magnification as specified in the preamble of the current
	       dvi file. */
int DviDen; /* Denominator specified in the preamble of the current
	       dvi file. */
int DviNum; /* Numerator as specified in the preamble of the current
	       dvi file. */

int HConv, VConv; /* Divide some horizontal or vertical position in DVIU
		     to come to pixels. These factors include the global
		     Magnification. */
int HConvUnMag; /* Same as above, but unmagnified. */
int VConvUnMag;

/*
 * ComputeConversionFactors
 * ************************
 * Compute all the necessary conversion factors.
 */
void
ComputeConversionFactors()
{
  double conv; /* Conversion factor, with magnification. */
  double conv_u; /* Without magnification. */

  /* Compute the magnification factor to use. If it is not specified
     using the -m command line option, use the value from the dvi file preamble. */
  if (MagCommandLine == -1)
    DriverMag = DviMag;
  else
    DriverMag = MagCommandLine;
  if (Verbose > V_SOME) {
    fprintf (stderr, "Magnification dvi file: ", DriverMag);
    PrintMag (DriverMag, stderr);
    fprintf (stderr, "\n");
  }

  /* Compute HConv and VConv: factors are factors by which you DIVIDE
     a DVIU based position in order to convert to pixels. */
  /* DVIU * conv = 10**-7 meters */
  conv  = ((double)DviNum / (double)DviDen);
  conv_u= conv;

  conv *= ((double)DriverMag / 1000.0);

#ifdef DEBUG
  fprintf (stderr, "%% DviNum: %d, DviDen: %d\n", DviNum, DviDen);
#endif
  /* DVIU * conv = 10**-2 meters (cm) */
  conv  /= 100000.0;
  conv_u/= 100000.0;

  /* DVIU * conv = inches */
  conv  /= 2.54;
  conv_u/= 2.54;

  /* DVIU * conv = pixels */
  conv *= (double) Resolution;
  conv_u*= (double) Resolution;

  /* The definition of these factors says that we divide a DVIU value by
   * those factors to get pixels! In case of unmagnified TeX input, this factor is
   * therefore the number of scaled points one pixel has. */
  HConv =     ((int) (1.0 / conv + 0.5));
  HConvUnMag =((int) (1.0 / conv_u+ 0.5));
  VConv = HConv;
  VConvUnMag = HConvUnMag;
  
  /* Compute, in DVIUs, by how much the positions may be off, before a MAX_DRIFT
     alarm is sounded. */
  MaxDriftHorizontal    =  MAX_DRIFT * HConv;
  MaxDriftHorizontalNeg = -MAX_DRIFT * HConv;
  MaxDriftVertical      =  MAX_DRIFT * VConv;
  MaxDriftVerticalNeg   = -MAX_DRIFT * VConv;

#ifdef DEBUG
  fprintf (stderr, "%% HConv: %d (scaled points per pixel)\n", HConv);
  fprintf (stderr, "%% MaxDrift: %d (in DVIUs)\n", MaxDriftHorizontal);
#endif
}

/*
 * ConvertPosPoints
 * ****************
 * Convert a dimension from DVIUs into points (pt). The factor for the horizontal
 * axis is used. This procedure takes the global magnification (incorporated into
 * HConv) into account.
 *
 * d: some dimension in DVIUs.
 * RET: the dimension in points as a floatain point number.
 */
double
ConvertPosPoints(d)
     DVIU d;
{
  return ((double)d / HConv / Resolution * 72.27);
}

/*
 * ConvertPosPointsNoDriverMag
 * ***************************
 * Identical procedure to previous one, NOT taking the global magnification
 * into account though.
 *
 * d: some dimension in DVIUs.
 * RET: the dimension in points as a floatain point number.
 */
double
ConvertPosPointsNoDriverMag(d)
     DVIU d;
{
  return ((double)d / HConv / Resolution * 72.27 / (DriverMag/1000.0));
}
