/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Read capability file. */

#include <stdio.h>
#include "defs.h"
#include "dvitps.h"
#include "units.h"
#include "extfil.h"
#include "fontd.h"

extern char *dvitps_cap_dir;
extern char *ProgNameS;

/* Set to TRUE if a valid capability file was found and read in. */
int ValidCapabilities;

/* Keep track of all font file types and their priorities here. There are
   4 different font types and therefore this array has four entries.
   The index into this array is one of CF_GF, CF_PK, CF_PDR, CF_PXL. */
FFP FontFilePriorities[4];

/*
 * ReadDvitpsCap
 * *************
 * Read the driver capability file. This file as of now contains
 * only information about what font types are available on the
 * local installation and in which order the driver is supposed
 * to look for these font file types.
 */
void
ReadDvitpsCap()
{
  EX_FILES ex_f;
  char buffer[256];
  char buffer2[256];
  char id[16];
  int priority;
  int index;
  int i;
  int dwf; /* Done with files. */
  int count;

  ValidCapabilities = FALSE;

  /* Part 1: handle file which determines the search patterns for fonts
     i.e. which font file types does the driver know about at a specific
     installation and in which order are files searched for. */
  for (i=0; i<=3; i++) {
    FontFilePriorities[i].ffp_p = -1;
    FontFilePriorities[i].ffp_n = NULL;
    FontFilePriorities[i].ffp_s = NULL;
  }

  /* Set up all the name strings. */
  FontFilePriorities[CF_GF].ffp_n = "gf";
  FontFilePriorities[CF_PK].ffp_n = "pk";
  FontFilePriorities[CF_PXL].ffp_n ="pxl";
  FontFilePriorities[CF_PDR].ffp_n = "pdr";

  /* Open the font capability file. If none, abort. */
  sprintf (buffer, "%s/fonts.%s", dvitps_cap_dir, ProgNameS);
  if (! FExOpen(&ex_f, EFT_READ, EFQ_NO_STDIN | EFQ_NO_FILE_NO_ERROR, buffer, ""))
    return;

  /* We know that there is a valid capability file now. We will read it in now. */
  ValidCapabilities = TRUE;
  while (ReadLineIntoBuffer(EX_FP(ex_f), buffer2, 256) != EOF) {
    /* Ignore empty lines and comment lines (starting with %) in this file. */
    if (IsEmptyLine(buffer2))
      continue;
    if (buffer2[0] == '%')
      continue;
#ifdef DEBUG
    fprintf (stderr, "ReadDvitpsCap(): READ: %s\n", buffer2);
#endif
    /* Read a pair of font identifier and priority now. */
    if (sscanf (buffer2, "%s %d", id, &priority) != 2)
      Fatal2 ("ReadDvitpsCap(): Format error in \"%s\"", buffer);
    if (Strlen(id) == 0)
      Fatal ("ReadDvitpsCap(): id empty");
      
    /* Now find this entry in the font file priority table and set the priority. */
    index = -1; /* invalid */
    if (strcmp(id, "GF:") == 0)  index = CF_GF;
    if (strcmp(id, "PK:") == 0)  index = CF_PK;
    if (strcmp(id, "PXL:") == 0) index = CF_PXL;
    if (strcmp(id, "PDR:") == 0) index = CF_PDR;
    if (index == -1)
      Fatal2 ("ReadDvitpsCap(): Format error 2 [%s]", id);
    FontFilePriorities[index].ffp_p = priority;
  }
  FExClose(&ex_f);

  /*
   * Now check on whether the user's initialisation makes any sense:
   * no "double priorities", no "holes" etc. This is actually redundant,
   * because the installation program already did these checks. But we
   * had in here before, so why not leave it in here.
   */
  dwf = FALSE;
  for (priority=0; priority<=3; priority++) {
    count = 0; /* Count number of occurences  of one priority in the array. */
    for (i=0; i<=3; i++) /* CF_* loop */
      if (FontFilePriorities[i].ffp_p == priority)
	count++;
    if (count > 1) /* Error if more than one with a particular priority. */
      Fatal ("InitFilePriorities(): count > 1.");
    if (dwf && count != 0) /* If no more font file types are used currently
			      but the current priority count is not zero, then
			      this is an error. */
      Fatal ("InitFilePriorities(): FONT_TYPES wrong.");
    if (count == 0) /* None on this particular priority level: so we are done! */
      dwf = TRUE;
  } /* for priority. */
}

/*
 * ReportFontFileTypes
 * *******************
 * Report all font file types the driver looks for and report their
 * search paths.
 */
void
ReportFontFileTypes()
{
  int priority, i;
  char * p;
  int tab_now;

  for (priority=0; priority<=3; priority++) {
    /* Loops through CF_* (excluding CF_NONE) */
    for (i=0; i<=3; i++) {
      if (FontFilePriorities[i].ffp_p == priority) {
	fprintf (stderr, "\t%d. Search path for \"%s\" files:\n",
		 priority+1, FontFilePriorities[i].ffp_n);
	if ((p = FontFilePriorities[i].ffp_s) == NULL)
	  Fatal ("ReportFontFileTypes(): NULL search path.");
	tab_now = TRUE;
	while (*p) {
	  if (tab_now)
	    fprintf (stderr, "\t\t");
	  tab_now = FALSE;
	  if (*p == ':') {
	    p++;
	    fprintf (stderr, "\n");
	    tab_now = TRUE;
	    continue;
	  }
	  putc (*p++, stderr);
	} /* while */
	fprintf (stderr, "\n");
      } /* if: there is such a font file with that priority. */
    } /* for i */
  } /* for priorities. */
}
