
IMPLEMENTATION MODULE SG480VDU;

(*
   Author:         Christopher P. Andrasic
   Implementation: University of Hamburg Modula-2 under VAX/VMS version 4
   Date Started:   February, 1987

   Description:
   Implements the InitSG480 procedure that initializes the terminal dependent
   routines and parameters used in DVItoVDU.
   The SG480 VDU has a Transparent mode in which it behaves like a VT100, but
   it can also emulate a Tektronix 4010/14 terminal in which the screen is
   1024 pixels wide by 780 pixels high.
   (The actual resolution of the SG480 screen is 700 by 496 pixels;
   the appropriate scaling is done automatically.)

   The bottom left pixel is the point (x=0,y=0). Note that x coordinates
   increase to the right and y coordinates increase up the screen.
   DVItoVDU uses a coordinate scheme in which horizontal (=h) coordinates
   also increase to the right but vertical (=v) coordinates increase DOWN the
   screen, i.e. the top left pixel on the screen is the point (h=0,v=0).
   This means that the Tektronix 4010 routines will have to do a
   simple translation of the vertical coordinates passed by DVItoVDU.
*)

  FROM VDUInterface IMPORT 
   (* PROCEDURE variables: *)
                           MoveToTextLine,
                           ClearTextLine,
                           ClearScreen,
                           StartText,
                           StartGraphics,
                           LoadFont,
                           ShowChar,
                           ShowRectangle,
                           ResetVDU,
   (* INTEGER variables: *)
                           DVIstatusl,
                           windowstatusl,
                           messagel,
                           commandl,
                           bottoml,
                           windowv,
                           windowh,
                           windowht,
                           windowwd;

  FROM TEK4010VDU IMPORT 
                         TEK4010ClearScreen,
                         TEK4010StartGraphics,
                         TEK4010LoadFont,
                         TEK4010ShowChar,
                         TEK4010ShowRectangle;

  FROM ScreenIO IMPORT 
                       Write, WriteCard,
                       WriteString, WriteLn;

  FROM SYSTEM IMPORT 
                     ADR;

  FROM VMS IMPORT 
                  SYS$SETIMR, SYS$WAITFR;

  CONST
    NUL =   0C;
    CR  =  15C;
    CAN =  30C;
    ESC =  33C;
    GS  =  35C;
    DEL = 177C;

    VAR
      result, efn : CARDINAL;
      delaytim : RECORD
                   c1, c2 : INTEGER;
                 END;


(**********************************************************************)

  PROCEDURE MoveAbs (row, col : CARDINAL);

(* 
   Assuming SG480 is in VT100 mode, we move the cursor to the
   given position.  VT100 numbers text lines the same way as DVItoVDU.
*)

  BEGIN
    Write(ESC);
    Write('[');
    IF row > 1 THEN
      WriteCard(row)
    END;
    Write(';');
    IF col > 1 THEN
      WriteCard(col)
    END;
    Write('H');
  END MoveAbs;

(**********************************************************************)

  PROCEDURE SG480MoveToTextLine (line : CARDINAL);

  BEGIN
    Write(ESC);
    Write('2');   (* switch to VT100 mode *)
    MoveAbs(line,1);
  END SG480MoveToTextLine;

(**********************************************************************)

  PROCEDURE SG480ClearTextLine (line : CARDINAL);

  BEGIN
    Write(ESC);
    Write('2');   (* switch to VT100 mode *)
    MoveAbs(line,1);
    Write(ESC);
    WriteString('[K');   (* erase to end of line *)
  END SG480ClearTextLine;

(**********************************************************************)

  PROCEDURE SG480StartText;

  BEGIN
    Write(ESC);
    WriteString('2');       (* switch to VT100 mode *)
  END SG480StartText;

(**********************************************************************)

  PROCEDURE SG480StartGraphics;

  BEGIN
    Write(ESC);
    WriteString('1');
    Write(GS);        (* Enter Graphics mode from VT100 *)
  END SG480StartGraphics;

(**********************************************************************)

  PROCEDURE SG480ClearScreen;

  BEGIN
    Write(ESC);
    WriteString('2');
    Write(ESC);
    WriteString('[2J');
    Write(ESC);
    WriteString('1');
    TEK4010ClearScreen;
  END SG480ClearScreen;

(**********************************************************************)

  PROCEDURE SG480ResetVDU;

  BEGIN
    efn := 3;
    delaytim.c1 := -30000000;
    delaytim.c2 := -1;
    Write(ESC);
    Write(DEL);  (* Will have to wait 3 seconds to do anything. *)
    result := SYS$SETIMR(efn,ADR(delaytim),0,0);
(* DEBUG: IF NOT ODD(result) THEN LIB$SIGNAL(result) END; *)
    result := SYS$WAITFR(efn);
(* DEBUG: IF NOT ODD(result) THEN LIB$SIGNAL(result) END; *)
    WriteString(' ');
    Write(CR);
  END SG480ResetVDU;

(**********************************************************************)

  PROCEDURE InitSG480;

(* 
   The dialog region will be the top 4 text lines in VT100 mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 24 in VT100 mode.
*)

  BEGIN
    DVIstatusl := 1;      (* DVItoVDU assumes top text line = 1 *)
    windowstatusl := 2;
    messagel := 3;
    commandl := 4;
    bottoml := 24;        (* also number of text lines on screen *)
(* 
   The above values assume the SG480 is in VT100 mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*)
    windowv := 130;       (* approx. height in TEK4010 pixels of 4 text lines
                         i.e. 4 * 780/24 *)
    windowh := 0;
    windowht := 780-windowv;
    windowwd := 1024;

    MoveToTextLine := SG480MoveToTextLine;
    ClearTextLine  := SG480ClearTextLine;
    ClearScreen    := SG480ClearScreen;
    StartText      := SG480StartText;
    StartGraphics  := SG480StartGraphics;
    LoadFont       := TEK4010LoadFont;
    ShowChar       := TEK4010ShowChar;
    ShowRectangle  := TEK4010ShowRectangle;
    ResetVDU       := SG480ResetVDU;

(*
    Reset the Selanar graphics plane.
*)
    Write(ESC);	Write('<');		(* switch to ANSI mode	    *)
    Write(ESC); Write('1');		(* into Tektronix mode	    *)
    Write(ESC); Write(DEL);		(* clear Tektronix plane    *)

    Write(NUL); Write(NUL);
    Write(NUL); Write(NUL);
    Write(NUL); Write(NUL);
    Write(NUL); Write(NUL);

    Write(ESC); Write('2');		(* into VT100 mode	    *)
    Write(ESC); WriteString('[1;1H');
    Write(ESC); WriteString('[2J');
    Write(ESC); WriteString('[?38h');
    Write(ESC); Write('1');		(* into Tektronix mode	    *)
    WriteLn;

  END InitSG480;

(**********************************************************************)

BEGIN
END SG480VDU.
