// Copyright (c) 1991 by Parag Patel.  All Rights Reserved.
static const char rcsid[] = "$Header: util.C,v 1.14 91/02/22 15:56:43 hmgr Exp $";

// various utility functions
//
// by Parag Patel

#include "defs.h"
#include <stdarg.h>


// get an unsigned value that is "num" bytes long from the file
// - the "switch" speeds up the 1|2|3|4 byte reads
//
long getuval(int num, FILE *fp)
{
    register long val = 0;
    register unsigned long b;

    b = getc(fp);
    switch (num)
    {
    case 4:
	val = b << 24;
	b = getc(fp);
    case 3:
	val |= b << 16;
	b = getc(fp);
    case 2:
	val |= b << 8;
	b = getc(fp);
    case 1:
	val |= b;
	break;

    default:
	quit("dvi bug: getuval called with count of %d\n", num);
    }
    return val;
}


// get a signed twos-complement value of "num" bytes from file
//
long getsval(int num, FILE *fp)
{
    register long val;
    register unsigned long b;
    register char c;

    // this sets "val" to the right sign
    c = getc(fp);

    // this shifts the 8-bit value to the right part of the word so
    // that we can do the fall-through "case" statement

    val = (long)c << ((num - 1) << 3);
    switch (num)
    {
    case 4:
	b = getc(fp);
	val |= b << 16;
    case 3:
	b = getc(fp);
	val |= b << 8;
    case 2:
	b = getc(fp);
	val |= b;
    case 1:
	break;

    default:
	quit("dvi bug: getsval called with count of %d\n", num);
    }
    return val;
}


// ignore next n bytes in file
//
void skipbytes(long num, FILE *fp)
{
    while (num-- > 0)
	(void)getc(fp);
}



// for debugging output only
// 
void debug(int level, char *fmt, ...)
{
    if (level > debuglevel)
	return;

    va_list ap;

    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    fprintf(stderr, "\n");
    va_end(ap);
}


// print informative messages only if verbose mode is on
// - break lines to fit on screen
// 
void mesg(char *fmt, ...)
{
    if (!verbose)
	return;

    static col = 0;

    // just print a newline on a null format string
    // - this is to end a sequence of messages with a nice newline
    if (fmt == NULL || *fmt == '\0')
    {
	if (col != 0)
	    fprintf(stderr, "\n");
	col = 0;
	return;
    }

    va_list ap;
    char buf[256];
    char *s = buf;

    va_start(ap, fmt);
    vsprintf(s, fmt, ap);
    va_end(ap);

    int len = strlen(s);

    if (col + len > 72)
    {
	fprintf(stderr, "\n");
	col = 0;
    }
    if (col == 0)
	while (*s == ' ')
	    s++, len--;
    col += len;

    fputs(s, stderr);
    fflush(stderr);
}

// print a warning only if we are in verbose mode
// - otherwise be silent
//
void warn(char *fmt, ...)
{
    if (!verbose)
	return;

    va_list ap;

    mesg(NULL);
    va_start(ap, fmt);
    fprintf(stderr, "WARNING: ");
    vfprintf(stderr, fmt, ap);
    fprintf(stderr, ".\n");
    va_end(ap);
}

// print an error message
//
void error(char *fmt, ...)
{
    va_list ap;

    mesg(NULL);
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    fprintf(stderr, ".\n");
    va_end(ap);
}

// print error message, then exit
// 
void quit(char *fmt, ...)
{
    va_list ap;

    mesg(NULL);
    va_start(ap, fmt);
    vfprintf(stderr, fmt, ap);
    fprintf(stderr, "!\n");
    va_end(ap);
    exit(1);
}


FILE *
fopenp(char *path, char *file, char *type)
{
    // if we have a null path, just do a normal fopen()
    if (path == NULL || *path == '\0')
	return fopen(file, type);

    // just try the file straight if it is an absolute path
    if (*file == '/')
    {
	FILE *ret = fopen(file, type);
	if (ret != NULL)
	    return ret;
    }

    // we look through the path for a directory, append the filename to
    // that directory, and then try to open the file - we are finished
    // when we are out of directories to try (return NULL) or we found a
    // file (return that file)

    char *p = path;
    while (p != NULL && *p != '\0') // do until end of "path"
    {
	// copy the directory name pointed to by "p" into "buf"
	char buf[MAXPATHLEN];
	char *s = buf;
	while (*p != PATHSEP && *p != '\0')
	    *s++ = *p++;

	// bump "p" to the start of the next directory
	if (*p != '\0')
	    p++;

	// append a '/' to "buf" if necessary & then null-terminate it
	if (s != buf && *(s - 1) != '/')
	    *s++ = '/';
	*s = '\0';

	// append the filename to the directory name
	strcat(s == buf ? s : s - 1, file);

	// try to open the file - return it if the fopen succeeded
	FILE *ret = fopen(buf, type);
	if (ret != NULL)
	    return ret;
    }

    return NULL;		// we struck out
}
