#
# protopkg()
# Main function for the protopkg operation.  Reads in a ./prototype file,
# verifies it, and uses it to make a valid package.
#

protopkg() {
   # look for a prototype file
   if [ ! -f ./prototype ]
   then
      echo "ERROR:  No package prototype file found."
      exit
   else
      . ./prototype
   fi

   # common variables
   TMP=/tmp/build-${PKGNAME}
   CWD=`pwd`
   COMPILED=n
   
   if [ ! -d $TMP ]
   then
      mkdir -p $TMP
      chmod 700 $TMP
   fi
   
   # set the ignore path
   IPATH="-path /proc -prune -o"
   
   if [ "$IGNOREPATH" != "" ]
   then
      IGNOREPATH="`echo $IGNOREPATH | sed -e s/\:/\ \-prune\ \-o\ \-path\ /g`"
      IPATH="$IPATH -path $IGNOREPATH -prune -o"
   fi
   
   echo "protopkg - version $VER"
   echo

   ### VERIFY PACKAGE PROTOTYPE FILE
   echo -n "Verifying package prototype file..."
   verifyproto
   echo "good, building package [$PKGNAME]"
   echo

   # are we only building one subpackage?
   if [ ! "$BUILD_SUBPACK" = "" ]
   then
      repack $BUILD_SUBPACK

      echo -n "Cleaning up temporary files..."
      if [ "$CLEANUP" = "y" ]
      then
         cd /
         rm -rf $TMP 2>/dev/null 1>/dev/null
         echo "done."
      else
         echo "skipping."
      fi

      exit
   fi
   
   ### COMPILE PROGRAM
   echo -n "Compiling package [$PKGNAME]..."
   cd $TMP
   if [ "$VERBOSE" = "y" ]
   then
      echo
      compile
   else
      compile 2>/dev/null 1>/dev/null
      echo "done."
   fi
   COMPILED=y
   
   ### BUILD LIST OF INSTALLED FILES
   echo
   echo -n "Generating list of files for package [$PKGNAME]..."
   beforeinstall
   if [ "$VERBOSE" = "y" ]
   then
      echo
      install
   else
      install 2>/dev/null 1>/dev/null
   fi
   afterinstall
   if [ ! "$VERBOSE" = "y" ]
   then
      echo "done."
   fi

   ### HAVE USER PLAY WITH PACKAGE LIST UNLESS BATCH MODE IS SET
   if [ ! "$BATCH" = "y" ]
   then
cat << EOF

protopkg has gathered a list of what it thinks should be in the package.
You will now be given the opportunity to edit that list if you see that
protopkg missed anything or got something wrong.  Just type in the editor
you want to use and hit enter to edit the package list.  After you exit
the editor, protopkg will finish making the package.

EOF
      echo -n "Editor to use [default: $EDITOR]: "
      read usered

      if [ "$usered" = "" ]
      then
         $EDITOR $TMP/package.lst
      else
         if [ ! -x $usered -a ! -f $usered ]
         then
            echo
            echo "Error with editor specified, defaulting to $EDITOR"
            $EDITOR $TMP/package.lst
         else
            $usered $TMP/package.lst
         fi
      fi
   fi
   echo

   ### Set the package and package control variables
   PKG=$TMP/pkg-$PKGNAME
   CTL=$PKG/install

   if [ "$VERBOSE" = "y" ]
   then
      echo "Creating package tree..."
      build_package $PKGNAME $PKG
      stripeverything $PKG
      restoredoctimes $PKGNAME $PKG
      permissionize $PKG
      compressdocs $PKG
   else
      echo -n "Creating package tree..."
      build_package $PKGNAME $PKG 2>/dev/null 1>/dev/null
      stripeverything $PKG 2>/dev/null 1>/dev/null
      restoredoctimes $PKGNAME $PKG 2>/dev/null 1>/dev/null
      permissionize $PKG 2>/dev/null 1>/dev/null
      compressdocs $PKG 2>/dev/null 1>/dev/null
   fi

   if [ ! "$VERBOSE" = "y" ]
   then
      echo "done."
   fi
   
   ### CHECK FOR ZERO-LENGTH FILES
   if [ "$VERBOSE" = "y" ]
   then
      echo "Checking for zero-length files..."
   fi
   zerocheck $PKG
   echo

   ### MAKE THE SYMLINKS FOR THE DOINST
   if [ "$VERBOSE" = "y" ]
   then
      symlinks $PKG $CTL
   else
      symlinks $PKG $CTL 2>/dev/null 1>/dev/null
   fi

   ### RUN THE SPECIAL FUNCTION
   cd $PKG
   if [ "$VERBOSE" = "y" ]
   then
      special
   else
      special 2>/dev/null 1>/dev/null
   fi

   ### READ THE RULES FILE AND PREPARE INCOMING CONFIG FILES
   if [ "$VERBOSE" = "y" ]
   then
      echo "Preparing incoming configuration files for processing..."
      prepare_conf
   else
      prepare_conf 2>/dev/null 1>/dev/null
   fi

   ### FIXATE PACKAGE
   if [ "$VERBOSE" = "y" ]
   then
      echo "Fixating package [$PKGNAME]..."
   else
      echo -n "Fixating package [$PKGNAME]..."
   fi

   fixate_package $PKGNAME $PKG $CTL

   if [ ! "$VERBOSE" = "y" ]
   then
      echo "done."
   fi

   ### HANDLE ANY SUBPACKAGES
   if [ ! "$SKIP_SUBPACKS" = "y" ]
   then
      if [ "$VERBOSE" = "y" ]
      then
         echo
         echo "Building subpackages..."
         subpacks
      else
         echo -n "Building subpackages..."
         subpacks 2>/dev/null 1>/dev/null
         echo "done."
      fi
   fi

   ### CLEANUP
   echo
   echo -n "Cleaning up temporary files..."
   if [ "$CLEANUP" = "y" ]
   then
      cd /
      rm -rf $TMP 2>/dev/null 1>/dev/null
      echo "done."
   else
      echo "skipping."
   fi
}
