/*
 * Copyright (C) 2002 Stefan Holst
 * Copyright (C) 2003 the xine project team
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: server.c,v 1.3 2003/03/20 21:59:58 guenter Exp $
 *
 * gxine unix domain socket remote control interface
 */

#include <stddef.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <pthread.h>
#include <signal.h>

#include "globals.h"
#include "actions.h"

#define LOG

#define SOCKET_FILENAME "%s/.gxine/socket"
#define BUF_SIZE        1024

static int       gxsocket;
static pthread_t server_thread;

/* create an unix domain socket */
static int make_socket (char *path) {

  int                sock;
  struct sockaddr_un adr;

  sock = socket (PF_UNIX, SOCK_STREAM, 0);
  if (sock < 0) {
    perror ("socket");
    exit (EXIT_FAILURE);
  }

  unlink (path);

  /* fill socket address structures */

  memset (&adr, 0, sizeof (adr));

  adr.sun_family = AF_UNIX;
  strncpy (adr.sun_path, path, sizeof (adr.sun_path)-1);
  
  /* bind the name to the descriptor */
  if (bind (sock, (struct sockaddr *) &adr, sizeof (adr)) < 0) {
    perror ("bind");
    exit (EXIT_FAILURE);
  }

#ifdef LOG
  printf ("server: socket '%s' created \n", path);
#endif

  return sock;
}


/*
 * Check for socket validity.
 */
static int sock_check_opened(int socket) {
  fd_set   readfds, writefds, exceptfds;
  int      retval;
  struct   timeval timeout;
  
  for(;;) {
    FD_ZERO(&readfds); 
    FD_ZERO(&writefds); 
    FD_ZERO(&exceptfds);
    FD_SET(socket, &exceptfds);
    
    timeout.tv_sec  = 0; 
    timeout.tv_usec = 0;
    
    retval = select(socket + 1, &readfds, &writefds, &exceptfds, &timeout);
    
    if(retval == -1 && (errno != EAGAIN && errno != EINTR))
      return 0;
    
    if (retval != -1)
      return 1;
  }

  return 0;
}

static void sock_write (int fd, char *msg) {

  int len, pos;

  len = strlen (msg);
  pos = 0;

  while (pos<len) {
  
    int n=0;

    n = write (fd, msg+pos, len-pos);

    if (n<=0) {
      
      switch (errno) {
      case EAGAIN:
	continue;
      default:
	perror ("server: write error");
	return;
      }
    }

    pos += n;
  }
}

static void error_report (void *s, char *msg) {

  int fd = (int) s;

  sock_write (fd, msg);
  sock_write (fd, "\n\n");
}

static int running;

static void *socket_listener (void *data) {

  struct sockaddr_un client;
  socklen_t          size;
  int                s;
  char               buf[BUF_SIZE+1];
  int                buf_len;
  int                listening;

  listen (gxsocket,1);

  while (running) {
    s = accept (gxsocket, (struct sockaddr *) &client, &size);
#ifdef LOG
    printf ("server: connected.\n");
#endif
    sock_write(s, "This is gxine\n\njava script interface. type 'help();' for help.\n");
    buf_len = 0;

    listening = 1;
    while (listening) {
      int n;

      n = read (s, buf+buf_len, 1);

      if (n <= 0) {
      
	switch (errno) {
	case EAGAIN:
	  continue;
	default:
	  printf ("server: read error %d\n", errno);
	  listening = 0;
	  break;
	}
      } else {

	if ( (buf[buf_len] == 10) || (buf[buf_len]==13) ) {
	  buf[buf_len]=0;
	  gdk_threads_enter();
	  action_exec (buf, error_report, s);
	  gdk_threads_leave();
	  
	  buf_len = 0;
	} else {
	  if (buf_len<BUF_SIZE)
	    buf_len += n;
	}
      }
    }
#ifdef LOG
    printf ("server: closing client connection.\n");
#endif
    close(s);
  }
#ifdef LOG
  printf ("server: listener thread exit\n");
#endif
  pthread_exit(NULL);
}

void server_setup (void) {

  char filename [1024];
  snprintf (filename, 1024, SOCKET_FILENAME, getenv ("HOME")); 

  gxsocket = make_socket (filename);
}

void server_start (void) {

  running = 1;

  pthread_create(&server_thread, NULL, socket_listener, NULL);
}

void server_stop (void) {

  void *p;

  running = 0;

  sleep (1);
  pthread_cancel (server_thread);

  pthread_join (server_thread, &p);

  /* shutdown (gxsocket, SHUT_RDWR);*/
  close(gxsocket);
}


/*
 * functions to connect to already running gxine
 */

static int   client_fd;
static char *tstr;

static void sigpipe_handler (int sn) {

  printf ("server: sigpipe received.\n");

  signal (SIGPIPE, sigpipe_handler);
}

int server_client_connect (void) {

  struct sockaddr_un cli_adr, serv_adr;
  int                length = sizeof(struct sockaddr_un);
  char               filename [1024];

  signal (SIGPIPE, sigpipe_handler);

  /* server filename */
  snprintf (filename, 1024, SOCKET_FILENAME, getenv ("HOME")); 

#ifdef LOG
  printf ("server: trying to connect to already running instance of gxine (%s)...\n", 
	  filename);
#endif

  client_fd = socket (AF_UNIX, SOCK_STREAM, 0);

  /* initialize the client address structure */
  memset (&cli_adr, 0, sizeof (cli_adr));
  cli_adr.sun_family = AF_UNIX;
  tstr = tmpnam (NULL);
  strcpy (cli_adr.sun_path, tstr);
  
  unlink (tstr);

  /* bind the socket to the client's address */
  if (bind (client_fd, (struct sockaddr*) &cli_adr, length) < 0) {
    perror ("bind");
    return 0;
  }

  /* initialize the server address structure */
  memset(&serv_adr, 0, sizeof(serv_adr));
  serv_adr.sun_family = AF_UNIX;
  strcpy (serv_adr.sun_path, filename);

  if (connect (client_fd, (struct sockaddr *) &serv_adr, sizeof (serv_adr)) < 0) {
#ifdef LOG
    perror ("connect");
#endif
    return 0;
  }
  
  printf ("server: connected to existing gxine instance.\n");

  return 1;
}

void server_client_send (char *cmd) {

  write (client_fd, cmd, strlen (cmd));
}

void server_client_stop (void) {

  close(client_fd);

  unlink (tstr);
}
