/*
 *  Copyright (C) 2002  Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <libgnome/gnome-i18n.h>
#include "gul-gobject-misc.h"
#include "galeon-marshal.h"
#include "gul-tb-button.h"
#include "gul-gui.h"
#include "gul-string.h"
#include "galeon-new-button.h"
#include "galeon-shell.h"
#include "galeon-dnd.h"
#include <gtk/gtkstock.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkclipboard.h>
#include <string.h>

#define NOT_IMPLEMENTED g_warning ("not implemented: " G_STRLOC);
//#define DEBUG_MSG(x) g_print x
#define DEBUG_MSG(x)

/**
 * Private data
 */
struct _GaleonNewButtonPrivate 
{
	GulTbButton *widget;
	gboolean sensitive;
};

/**
 * Private functions, only availble from this file
 */
static void		galeon_new_button_class_init			(GaleonNewButtonClass *klass);
static void		galeon_new_button_init			(GaleonNewButton *tb);
static void		galeon_new_button_finalize_impl		(GObject *o);
static GtkWidget *	galeon_new_button_get_widget_impl		(GulTbItem *i);
static GdkPixbuf *	galeon_new_button_get_icon_impl		(GulTbItem *i);
static gchar *		galeon_new_button_get_name_human_impl	(GulTbItem *i);
static gchar *		galeon_new_button_to_string_impl		(GulTbItem *i);
static GulTbItem *	galeon_new_button_clone_impl			(GulTbItem *i);
static void		galeon_new_button_parse_properties_impl	(GulTbItem *i, const gchar *props);
static void		galeon_new_button_clicked_cb			(GtkWidget *w, GaleonNewButton *b);
static void		galeon_new_button_drag_data_received_cb (GtkWidget* widget, 
								 GdkDragContext *context,
								 gint x, gint y, 
								 GtkSelectionData *selection_data,
								 guint info, guint time, 
								 GaleonNewButton *b);


static gpointer gul_tb_item_class;

/**
 * TbiZoom object
 */

MAKE_GET_TYPE (galeon_new_button, "GaleonNewButton", GaleonNewButton,
	       galeon_new_button_class_init, 
	       galeon_new_button_init, GALEON_TYPE_TBI);

static void
galeon_new_button_class_init (GaleonNewButtonClass *klass)
{
	G_OBJECT_CLASS (klass)->finalize = galeon_new_button_finalize_impl;
	
	GUL_TB_ITEM_CLASS (klass)->get_widget = galeon_new_button_get_widget_impl;
	GUL_TB_ITEM_CLASS (klass)->get_icon = galeon_new_button_get_icon_impl;
	GUL_TB_ITEM_CLASS (klass)->get_name_human = galeon_new_button_get_name_human_impl;
	GUL_TB_ITEM_CLASS (klass)->to_string = galeon_new_button_to_string_impl;
	GUL_TB_ITEM_CLASS (klass)->clone = galeon_new_button_clone_impl;
	GUL_TB_ITEM_CLASS (klass)->parse_properties = galeon_new_button_parse_properties_impl;

	gul_tb_item_class = g_type_class_peek_parent (klass);
}

static void 
galeon_new_button_init (GaleonNewButton *tbi)
{
	GaleonNewButtonPrivate *p = g_new0 (GaleonNewButtonPrivate, 1);
	tbi->priv = p;

	p->sensitive = TRUE;
}

GaleonNewButton *
galeon_new_button_new (void)
{
	GaleonNewButton *ret = g_object_new (GALEON_TYPE_NEW_BUTTON, NULL);
	return ret;
}

static void
galeon_new_button_finalize_impl (GObject *o)
{
	GaleonNewButton *it = GALEON_NEW_BUTTON (o);
	GaleonNewButtonPrivate *p = it->priv;

	if (p->widget)
	{
		g_object_unref (p->widget);
	}

	g_free (p);
	
	DEBUG_MSG (("GaleonNewButton finalized\n"));
	
	G_OBJECT_CLASS (gul_tb_item_class)->finalize (o);
}

static void
galeon_new_button_setup_widget (GaleonNewButton *b)
{
	GaleonNewButtonPrivate *p = b->priv;

	if (!p->widget)
	{
		galeon_new_button_get_widget_impl (GUL_TB_ITEM (b));
	}
	g_assert (GUL_IS_TB_BUTTON (p->widget));

	gul_tb_button_set_label (p->widget, GTK_STOCK_NEW);
	gul_tb_button_set_tooltip_text (p->widget, _("Open a new tab or window"));
	gul_tb_button_set_priority (p->widget, FALSE);
	gul_tb_button_set_show_arrow (p->widget, FALSE);
	gul_tb_button_set_sensitivity (p->widget, p->sensitive);
}

static void
galeon_new_button_new_window_activate_cb (GtkMenuItem *mi, GaleonNewButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));
	GaleonTab *tab;

	g_return_if_fail (IS_GALEON_WINDOW (window));

	tab = galeon_window_get_active_tab (window);
	
	galeon_shell_new_tab (galeon_shell, window, tab, NULL, 
			      GALEON_NEW_TAB_HOMEPAGE
			      | GALEON_NEW_TAB_JUMP
			      | GALEON_NEW_TAB_IN_NEW_WINDOW);
}

static void
galeon_new_button_new_tab_activate_cb (GtkMenuItem *mi, GaleonNewButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));
	GaleonTab *tab;

	g_return_if_fail (IS_GALEON_WINDOW (window));

	tab = galeon_window_get_active_tab (window);
	
	galeon_shell_new_tab (galeon_shell, window, tab, NULL, 
			      GALEON_NEW_TAB_HOMEPAGE
			      | GALEON_NEW_TAB_JUMP
			      | GALEON_NEW_TAB_IN_EXISTING_WINDOW);
}

static gboolean
galeon_new_button_button_press_event_cb (GtkWidget *widget, GdkEventButton *event, 
					 GaleonNewButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));

	if (event->button == 2 && IS_GALEON_WINDOW (window))
	{
		static GtkClipboard *cb = NULL;
		gchar *url;

		if (!cb)
		{
			cb = gtk_clipboard_get (GDK_SELECTION_PRIMARY);
		}
		
		url = gtk_clipboard_wait_for_text (cb);

		if (url)
		{
			GaleonTab *tab = galeon_window_get_active_tab (window);
			galeon_shell_new_tab (galeon_shell, window, tab, url, 
					      GALEON_NEW_TAB_JUMP);
			g_free (url);
		}
		return TRUE;
	}
	
	return FALSE;
}

static GtkWidget *
galeon_new_button_get_widget_impl (GulTbItem *i)
{
	GaleonNewButton *iz = GALEON_NEW_BUTTON (i);
	GaleonNewButtonPrivate *p = iz->priv;

	if (!p->widget)
	{
		GtkMenuShell *ms;
		
		p->widget = gul_tb_button_new ();
		g_object_ref (p->widget);
		gtk_object_sink (GTK_OBJECT (p->widget));

		gul_tb_button_set_use_stock (p->widget, TRUE);
		gul_tb_button_set_enable_menu (p->widget, TRUE);
		
		galeon_new_button_setup_widget (iz);

		gtk_widget_show (GTK_WIDGET (p->widget));
		
		g_signal_connect (gul_tb_button_get_button (p->widget), "clicked",
				  G_CALLBACK (galeon_new_button_clicked_cb), i);
		g_signal_connect (gul_tb_button_get_button (p->widget), "button-press-event",
				  G_CALLBACK (galeon_new_button_button_press_event_cb), i);

		galeon_dnd_url_drag_dest_set (GTK_WIDGET(gul_tb_button_get_button (p->widget)));
		g_signal_connect (gul_tb_button_get_button (p->widget), "drag_data_received",
				  G_CALLBACK(galeon_new_button_drag_data_received_cb),
				  i);

		ms = gul_tb_button_get_menu (p->widget);

		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("New _Window"), 
					     G_CALLBACK (galeon_new_button_new_window_activate_cb), i);
		gul_gui_append_new_menuitem (GTK_WIDGET (ms), _("New _Tab"), 
					     G_CALLBACK (galeon_new_button_new_tab_activate_cb), i);
	}

	return GTK_WIDGET (p->widget);
}

static GdkPixbuf *
galeon_new_button_get_icon_impl (GulTbItem *i)
{
	static GdkPixbuf *pb_new = NULL;

	if (!pb_new)
	{
		/* what's the easier way? */
		GtkWidget *b = gtk_button_new ();
		pb_new = gtk_widget_render_icon (b,
						GTK_STOCK_NEW,
						GTK_ICON_SIZE_SMALL_TOOLBAR,
						NULL);
		gtk_widget_destroy (b);
	}

	return g_object_ref (pb_new);
}

static gchar *
galeon_new_button_get_name_human_impl (GulTbItem *i)
{
	return g_strdup (_("New"));
}

static gchar *
galeon_new_button_to_string_impl (GulTbItem *i)
{
	/* if it had any properties, the string should include them */

	return g_strdup_printf ("%s=new_button", i->id);
}

static GulTbItem *
galeon_new_button_clone_impl (GulTbItem *i)
{
	GulTbItem *ret = GUL_TB_ITEM (galeon_new_button_new ());

	gul_tb_item_set_id (ret, i->id);

	return ret;
}

static void
galeon_new_button_parse_properties_impl (GulTbItem *it, const gchar *props)
{
	/* we have no properties */
}

void
galeon_new_button_set_sensitive (GaleonNewButton *b, gboolean s)
{
	GaleonNewButtonPrivate *p = b->priv;
	p->sensitive = s;
	if (p->widget)
	{
		gul_tb_button_set_sensitivity (p->widget, s);
	}
	else
	{
		galeon_new_button_get_widget_impl (GUL_TB_ITEM (b));
	}
}

static void
galeon_new_button_clicked_cb (GtkWidget *w, GaleonNewButton *b)
{
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));
	GaleonTab *tab;

	g_return_if_fail (IS_GALEON_WINDOW (window));

	tab = galeon_window_get_active_tab (window);
	
	galeon_shell_new_tab (galeon_shell, window, tab, NULL, 
			      GALEON_NEW_TAB_HOMEPAGE
			      | GALEON_NEW_TAB_JUMP);
}


static void
each_url_receive_data_binder (const char * url, const char * title, 
			      gpointer data )
{
	char ** urlp = data;
	if (*urlp == NULL)
	{
		*urlp = g_strdup (url);
	}
}	

static void
galeon_new_button_drag_data_received_cb (GtkWidget* widget, 
					 GdkDragContext *dc,
					 gint x, gint y, 
					 GtkSelectionData *selection_data,
					 guint info, guint time, 
					 GaleonNewButton *b)
{
	char * url = NULL;
	GaleonWindow *window = galeon_tbi_get_window (GALEON_TBI (b));

	g_return_if_fail (IS_GALEON_WINDOW (window));

	galeon_dnd_drag_data_receive (widget, dc, x, y, selection_data,
			 info, time, &url, each_url_receive_data_binder);
	
	if (url)
	{
		GaleonTab *tab;

		tab = galeon_window_get_active_tab (window);
	
		galeon_shell_new_tab (galeon_shell, window, tab, url,
				      GALEON_NEW_TAB_JUMP);
		g_free (url);
	}
}
