/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <string.h>

#include <glib.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>

#include <pan/nntp.h>
#include <pan/task-group-count.h>

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

/*********************
**********************  Private Function Prototypes
*********************/

static void task_group_count_run (Task*, PanSocket*);
static char* task_group_count_describe (const StatusItem*);

/*********************
**********************  Variables
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

/*********************
**********************  BEGINNING OF SOURCE
*********************/

static void
task_group_count_destructor (PanObject * o )
{
	TaskGroupCount * t = TASK_GROUP_COUNT(o);
	g_ptr_array_free (t->groups, TRUE);

	/* destroy parent class */
	task_destructor (o);
}

/************
*************  PUBLIC ROUTINES
************/

PanObject*
task_group_count_new   (Group      ** groups,
                        guint         qty)
{
	TaskGroupCount * t = NULL;

	/* sanity checks */
	g_return_val_if_fail (groups!=NULL, NULL);
	g_return_val_if_fail (qty>=1, NULL);

	/* create the object */
       	t = g_new0 (TaskGroupCount, 1);
        debug1 (DEBUG_PAN_OBJECT, "task_group_count: %p", t);

	/* initialize the parent class */
	task_constructor (TASK(t), task_group_count_destructor, task_group_count_describe, groups[0]->server, TRUE);

	/* initialize the task-bodies */
	t->groups = g_ptr_array_new ();
	pan_g_ptr_array_assign (t->groups, (gpointer*)groups, qty);
	task_state_set_need_socket (&TASK(t)->state, groups[0]->server, task_group_count_run);

	return PAN_OBJECT(t);
}

/*****
******
*****/

static char*
task_group_count_describe (const StatusItem * status)
{
	const TaskGroupCount * t = TASK_GROUP_COUNT(status);

	return g_strdup_printf (
		_("Updating article counts for %u groups from server \"%s\""),
	       t->groups->len,
	       TASK(t)->server->name);
}

/*****
******
*****/

static void
task_group_count_run (Task * task, PanSocket * sock)
{
	TaskGroupCount * t = TASK_GROUP_COUNT(task);
	GPtrArray * groups = t->groups;
	StatusItem * status = STATUS_ITEM(task);
	guint i;

	/* for each group, get the numbers... */
	status_item_emit_init_steps (status, groups->len);
	for (i=0; i!=groups->len; ++i)
	{
		int result;
		gulong qty=0, low=0, high=0;
		Group * g = GROUP(g_ptr_array_index(groups,i));

		/* let the status item know what we're doing */
		status_item_emit_next_step (status);
		status_item_emit_status_va (status,
			_("Updated counts for %u groups out of %u"),
			i, groups->len);

		/* get the numbers for this group */
		result = nntp_get_group_info (STATUS_ITEM(task), sock, g->name, &qty, &low, &high, &TASK(task)->hint_abort);
		if (result == TASK_SUCCESS)
		{
			/* update the group object */
			/* group_set_article_range (g, low, high); */
			group_set_article_qty (g, qty);
		}
		else
		{
			status_item_emit_error_va (STATUS_ITEM(task),
			                           _("Unable to update counts for group \"%s\""),
			                           group_get_name(g));
		}
	}

	task_state_set (&task->state, TASK_SUCCESS);
}
