/*
   This file is part of KDE/aRts (Noatun) - xine integration
   Copyright (C) 2002 Ewald Snel <ewald@rambo.its.tudelft.nl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <limits.h>
#include <math.h>
#include <audiosubsys.h>
#include <convert.h>
#include <debug.h>
#include "xinePlayObject_impl.h"

#define VPO_PLAY		1000
#define VPO_QUIT		1001
#define VPO_STOP		1002

using namespace Arts;


int ao_fifo_arts_delay()
{
    return (int)(1000 * Arts::AudioSubSystem::the()->outputDelay());
}

xinePlayObject_impl::xinePlayObject_impl()
{
    XInitThreads();

    if (!(video.display = XOpenDisplay( NULL )))
    {
	arts_fatal( "could not open X11 display" );
    }

    XFlush( video.display );

    // Create a special window for uninterrupted X11 communication
    video.xcom_window = XCreateSimpleWindow( video.display,
					     DefaultRootWindow( video.display ),
					     0, 0, 1, 1, 0, 0, 0 );

    XSelectInput( video.display, video.xcom_window, ExposureMask );

    pthread_cond_init( &cond, 0 );
    pthread_mutex_init( &mutex, 0 );

    flpos		  = 0.0;
    config		  = 0;
    xine		  = 0;
    ao_driver		  = 0;
    vo_driver		  = 0;

    // Initialize audio and video details
    Arts::SoundServerV2 server = Arts::Reference( "global:Arts_SoundServerV2" );
    audio.sample_rate	  = 0;
    audio.num_channels	  = 0;
    audio.bits_per_sample = 0;

    video.xcom_atom	  = XInternAtom( video.display, "VPO_INTERNAL_EVENT", False );
    video.xcom_resize	  = XInternAtom( video.display, "VPO_RESIZE_NOTIFY", False );
    video.window	  = video.xcom_window;

    if (pthread_create( &thread, 0, pthread_start_routine, this ))
    {
	arts_fatal( "could not create thread" );
    }
}

xinePlayObject_impl::~xinePlayObject_impl()
{
    halt();

    // Send stop event to thread
    sendEvent( video.xcom_window, VPO_QUIT );

    pthread_join( thread, 0 );
    pthread_cond_destroy( &cond );
    pthread_mutex_destroy( &mutex );

    XSync( video.display, False );
    XDestroyWindow( video.display, video.xcom_window );
    XCloseDisplay( video.display );
}

bool xinePlayObject_impl::loadMedia( const string &filename )
{
    halt();

    // TODO: Should we protect this with a mutex?
    mrl = filename;

    return true; //running;
}

string xinePlayObject_impl::description()
{
    return "xine aRts plugin";
}

poTime xinePlayObject_impl::currentTime()
{
    poTime time;

    pthread_mutex_lock( &mutex );

    time.seconds = (xine == 0) ? 0 : (long)xine_get_current_time( xine );
    time.ms = 0;

    pthread_mutex_unlock( &mutex );

    return time;
}

poTime xinePlayObject_impl::overallTime()
{
    poTime time;

    pthread_mutex_lock( &mutex );

    time.seconds = (xine == 0) ? 0 : (long)xine_get_stream_length( xine );
    time.ms = 0;

    pthread_mutex_unlock( &mutex );

    if (!time.seconds && !time.ms)
    {
	time.ms = 1;
    }

    return time;
}

poCapabilities xinePlayObject_impl::capabilities()
{
    return static_cast<poCapabilities>( capPause | capSeek );
}

string xinePlayObject_impl::mediaName()
{
    return mrl;
}

poState xinePlayObject_impl::state()
{
    poState state;

    pthread_mutex_lock( &mutex );

    if ((xine == 0) || xine_get_status( xine ) != XINE_PLAY)
	state = posIdle;
    else if (xine_get_speed( xine ) == SPEED_PAUSE)
	state = posPaused;
    else
	state = posPlaying;

    pthread_mutex_unlock( &mutex );

    return state;
}

long xinePlayObject_impl::x11Snapshot()
{
    long pixmap = -1;

    pthread_mutex_lock( &mutex );

    if ((xine != 0) && xine_get_status( xine ) == XINE_PLAY)
    {
	pixmap = (long)vo_fifo_snapshot( &video, xine );
    }
    pthread_mutex_unlock( &mutex );

    return pixmap;
}

void xinePlayObject_impl::x11WindowId( long window )
{
    pthread_mutex_lock( &mutex );

    if (window == -1)
    {
	window = video.xcom_window;
    }

    if ((Window)window != video.window)
    {
	XLockDisplay( video.display );

	// Change window and set event mask of new window
	video.window = window;

	XSelectInput( video.display, window, ExposureMask );

	if (vo_driver)
	{
	    vo_fifo_resize_notify( &video );
	    vo_driver->gui_data_exchange( vo_driver,
					  GUI_DATA_EX_DRAWABLE_CHANGED,
					  (void *)video.window );
	}

	XUnlockDisplay( video.display );
    }
    pthread_mutex_unlock( &mutex );
}

long xinePlayObject_impl::x11WindowId()
{
    return (video.window == video.xcom_window) ? (long)-1 : video.window;
}

void xinePlayObject_impl::play()
{
    pthread_mutex_lock( &mutex );

    if ((xine != 0) && xine_get_status( xine ) == XINE_PLAY)
    {
	if (xine_get_speed( xine ) == SPEED_PAUSE)
	{
	    xine_set_speed( xine, SPEED_NORMAL );
	}
    }
    else
    {
	// Wait for the xine object to be destroyed
	while (xine != 0)
	{
	    pthread_cond_wait( &cond, &mutex );
	}

	// Send play event
	sendEvent( video.xcom_window, VPO_PLAY );

	// Wait for the xine object to start (or fail)
	pthread_cond_wait( &cond, &mutex );
    }
    pthread_mutex_unlock( &mutex );
}

void xinePlayObject_impl::halt()
{
    pthread_mutex_lock( &mutex );

    if (xine != 0)
    {
	if (xine_get_status( xine ) == XINE_PLAY)
	{
	    ao_fifo_wait_close( ao_driver );

	    xine_stop( xine );
	}

	sendEvent( video.xcom_window, VPO_STOP );

	// Wait for the xine object to be destroyed
	pthread_cond_wait( &cond, &mutex );
    }
    pthread_mutex_unlock( &mutex );
}

void xinePlayObject_impl::seek( const class poTime &t )
{
    pthread_mutex_lock( &mutex );

    if ((xine != 0) && (xine_get_status( xine ) == XINE_PLAY))
    {
	int paused = (xine_get_speed( xine ) == SPEED_PAUSE);

	ao_fifo_flush( ao_driver, ULONG_MAX );

	xine_play( xine, (char *)mrl.c_str(), 0, t.seconds );

	if (paused)
	{
	    xine_set_speed( xine, SPEED_PAUSE );
	}
    }
    pthread_mutex_unlock( &mutex );
}

void xinePlayObject_impl::pause()
{
    pthread_mutex_lock( &mutex );

    if ((xine != 0) && (xine_get_status( xine ) == XINE_PLAY))
    {
	xine_set_speed( xine, SPEED_PAUSE );

	ao_fifo_flush( ao_driver, ULONG_MAX );
    }
    pthread_mutex_unlock( &mutex );
}

void xinePlayObject_impl::calculateBlock( unsigned long samples )
{
    unsigned int skip, received = 0, converted = 0, xSamples = 0;
    unsigned char *buffer;
    double speed = 1.0;

    pthread_mutex_lock( &mutex );

    if ((xine != 0) && (xine_get_status( xine ) == XINE_PLAY))
    {
	if (xine_get_speed( xine ) != SPEED_PAUSE)
	{
	    // Calculate resampling parameters
	    speed = (double)audio.sample_rate / samplingRateFloat;
	    xSamples = (unsigned int)((double)samples * speed + 8.0);

	    received = ao_fifo_read( ao_driver, &buffer, xSamples );
	}
	else
	{
	    // Clear audio buffer
	    ao_fifo_flush( ao_driver, ULONG_MAX );
	}
    }

    pthread_mutex_unlock( &mutex );

    // Convert samples and fill gaps with zeroes
    if (received)
    {
	converted = uni_convert_stereo_2float( samples, buffer, received,
					       audio.num_channels,
					       audio.bits_per_sample,
					       left, right, speed, flpos );
	flpos += (double)converted * speed;
	skip   = (int)floor( flpos );
	skip   = (received < (xSamples - 8)) ? (xSamples - 8) : skip;
	flpos  = flpos - floor( flpos );
	ao_fifo_flush( ao_driver, skip );
    }
    for (unsigned long i=converted; i < samples; i++)
    {
	left[i] = 0;
	right[i] = 0;
    }
}

void xinePlayObject_impl::playEvent()
{
    pthread_mutex_lock( &mutex );

    running   = false;
    vo_driver = init_video_out_plugin( config, &video );

    if (vo_driver)
    {
	ao_driver = init_audio_out_plugin( config, &audio );

	xine = xine_init( vo_driver, ao_driver, config );

	xine_register_event_listener( xine, xine_handle_event, this );
	xine_select_audio_channel( xine, 0 );
	xine_select_spu_channel( xine, -1 );

	running = xine_play( xine, (char *)mrl.c_str(), 0, 0 );

	if (!running)
	{
	    sendEvent( video.xcom_window, VPO_STOP );
	}
    }
    if (!vo_driver || running)
    {
	pthread_cond_broadcast( &cond );
    }
    pthread_mutex_unlock( &mutex );
}

void xinePlayObject_impl::stopEvent()
{
    pthread_mutex_lock( &mutex );

    if (xine != 0)
    {
	ao_fifo_wait_close( ao_driver );

	xine_remove_event_listener( xine, xine_handle_event );
	xine_exit( xine );

	xine	    = 0;
	ao_driver   = 0;
	vo_driver   = 0;
    }
    pthread_cond_broadcast( &cond );
    pthread_mutex_unlock( &mutex );

    clearWindow();
}

void xinePlayObject_impl::xineEvent( xine_event_t *event )
{
    if (event->type == XINE_EVENT_PLAYBACK_FINISHED)
    {
	sendEvent( video.xcom_window, VPO_STOP );
    }
}

void xinePlayObject_impl::sendEvent( long window, long message )
{
    XEvent event;

    // Send X11 client message
    memset( &event, 0, sizeof(event) );

    event.type			= ClientMessage;
    event.xclient.window	= window;
    event.xclient.message_type	= video.xcom_atom;
    event.xclient.format	= 32;
    event.xclient.data.l[0]	= message;

    XSendEvent( video.display, window, True, 0, &event );

    XFlush( video.display );
}

void xinePlayObject_impl::clearWindow()
{
    Window root;
    unsigned int u, w, h;
    int x, y, screen;

    XLockDisplay( video.display );

    screen = DefaultScreen( video.display );

    XGetGeometry( video.display, video.window, &root, &x, &y, &w, &h, &u, &u );

    XSetForeground( video.display, DefaultGC( video.display, screen ),
		    BlackPixel( video.display, screen ) );
    XFillRectangle( video.display, video.window,
		    DefaultGC( video.display, screen ), x, y, w, h );

    XUnlockDisplay( video.display );
}

void xinePlayObject_impl::eventLoop()
{
    XEvent event;
    string config_file = string( getenv( "HOME" )).append( "/.xine/config" );

    config = xine_config_file_init( (char *)config_file.c_str() );

    // Disable OSD
    config->register_bool( config, (char *)"misc.osd_display", 0,
			   (char *)"disabled for KDE", 0, 0, 0 );
    config->update_num( config, (char *)"misc.osd_display", 0 );

    // Disable xine logo
    config->register_string( config, (char *)"misc.logo_mrl", (char *)"",
			     (char *)"disabled for KDE", 0, 0, 0 );
    config->update_string( config, (char *)"misc.logo_mrl", (char *)"" );

    while (config)
    {
	XNextEvent( video.display, &event );

	if (event.type == Expose && event.xexpose.count == 0 &&
	    event.xexpose.window == video.window)
	{
	    if (vo_driver)
	    {
		vo_driver->gui_data_exchange( vo_driver,
					      GUI_DATA_EX_EXPOSE_EVENT,
					      &event );
	    }
	    else
	    {
		clearWindow();
	    }
	}
	else if (event.type == video.shm_completion && vo_driver)
	{
	    vo_driver->gui_data_exchange( vo_driver,
					  GUI_DATA_EX_COMPLETION_EVENT,
					  &event );
	}

	if (event.type == ClientMessage &&
	    event.xclient.message_type == video.xcom_atom &&
	    event.xclient.window == video.xcom_window)
	{
	    switch (event.xclient.data.l[0])
	    {
	    case VPO_QUIT:
		// Destroy xine config object (signal quit)
		config->dispose( config );
		config = 0;
		break;
	    case VPO_PLAY:
		playEvent();
		break;
	    case VPO_STOP:
		stopEvent();
		break;
	    }
	}
    }
}

REGISTER_IMPLEMENTATION(xinePlayObject_impl);
