/*
   This file is part of KDE/aRts (Noatun) - xine integration
   Copyright (C) 2002 Ewald Snel <ewald@rambo.its.tudelft.nl>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <pthread.h>
#include <sys/time.h>
#include <xine/xine_internal.h>
#include <xine/audio_out.h>
#include "audio_fifo_out.h"

#define GAP_TOLERANCE	5000

typedef struct fifo_driver_s
{
    ao_driver_t	     ao_driver;

    xine_arts_audio *audio;

    int		     capabilities;
    int		     mode;
    int		     enabled;
    int		     locked;
    pthread_mutex_t  mutex;
    pthread_cond_t   cond;

    uint32_t	     bytes_per_frame;
    uint32_t	     bytes_in_buffer;
    uint32_t	     buffer_size;
    uint8_t	    *fifo;
    uint32_t	     fifo_read_ptr;
    uint32_t	     fifo_write_ptr;
} fifo_driver_t;

/*
 * open the audio device for writing to
 */
static int ao_fifo_open( ao_driver_t *this_gen, uint32_t bits, uint32_t rate, int mode )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;

    if ((mode & ao->capabilities) == 0)
    {
	fprintf( stderr, "fifo_audio_out: unsupported mode %08x\n", mode);
	return 0;
    }
    ao->mode			= mode;
    ao->audio->sample_rate	= rate;
    ao->audio->bits_per_sample	= bits;
    ao->fifo_read_ptr		= 0;
    ao->fifo_write_ptr		= 0;
    ao->bytes_in_buffer		= 0;

    switch (mode)
    {
    case AO_CAP_MODE_MONO:
	ao->audio->num_channels = 1;
	break;
    case AO_CAP_MODE_STEREO:
	ao->audio->num_channels = 2;
	break;
    }
    ao->bytes_per_frame	= (ao->audio->bits_per_sample * ao->audio->num_channels) / 8;
    ao->buffer_size	= ao->audio->sample_rate * ao->bytes_per_frame;
    ao->fifo		= malloc( (ao->buffer_size << 1) );

    pthread_mutex_lock( &ao->mutex );
    ao->enabled = 1;
    ao->locked  = 0;
    pthread_mutex_unlock( &ao->mutex );

    return ao->audio->sample_rate;
}

static int ao_fifo_num_channels( ao_driver_t *this_gen )
{
    return ((fifo_driver_t *)this_gen)->audio->num_channels;
}

static int ao_fifo_bytes_per_frame( ao_driver_t *this_gen )
{
    return ((fifo_driver_t *)this_gen)->bytes_per_frame;
}

static int ao_fifo_get_gap_tolerance( ao_driver_t *this_gen )
{
    return GAP_TOLERANCE;
}

static int ao_fifo_write( ao_driver_t *this_gen, int16_t *data, uint32_t num_frames )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;
    uint32_t w, n, m;
    uint8_t *src = (uint8_t *)data;

    w = ao->fifo_write_ptr;
    n = (num_frames * ao->bytes_per_frame);
    m = (ao->buffer_size - w);

    if (n > ao->buffer_size)
    {
	/* split audio packet into 2 fragments */
	n    = (num_frames >> 1);
	src += (n * ao->bytes_per_frame);

	return  ao_fifo_write( this_gen, data, n ) &&
		ao_fifo_write( this_gen, (int16_t *)src, n + (num_frames & 1) );
    }
    if ((ao->bytes_in_buffer + n) > ao->buffer_size)
    {
	pthread_mutex_lock( &ao->mutex );

	while ((ao->bytes_in_buffer + n) > ao->buffer_size)
	{
	    struct timespec ts;
	    struct timeval tv;
	    int delay;

	    if (!ao->enabled)
	    {
		/* audio has stopped */
		pthread_mutex_unlock( &ao->mutex );
		return 0;
	    }

	    gettimeofday( &tv, 0 );

	    ts.tv_sec	= tv.tv_sec;
	    ts.tv_nsec	= tv.tv_usec * 1000;

	    delay	= ao_fifo_arts_delay();
	    delay      += (1000 * num_frames) / ao->audio->sample_rate;
	    delay	= (delay < 20) ? 20 : ((delay >= 250) ? 250 : delay + 1);
	    ts.tv_sec  += delay / 1000;
	    ts.tv_nsec += 1000000 * (delay % 1000);

	    if (ts.tv_nsec >= 1000000000)
	    {
		ts.tv_sec++;
		ts.tv_nsec -= 1000000000;
	    }
	    if (pthread_cond_timedwait( &ao->cond, &ao->mutex, &ts ) != 0)
	    {
		fprintf( stderr, "fifo_audio_out: blocked for more than %d ms,\n", delay);
		fprintf( stderr, "fifo_audio_out: %d sample(s) discarded.\n", num_frames);
		pthread_mutex_unlock( &ao->mutex );
		return 0;
	    }
	}
	pthread_mutex_unlock( &ao->mutex );
    }
    if (n >= m)
    {
	memcpy( (ao->fifo + w), src, m );
	w = (n - m);
	memcpy( ao->fifo, src + m, w );
    }
    else
    {
	memcpy( (ao->fifo + w), src, n );
	w += n;
    }
    pthread_mutex_lock( &ao->mutex );
    ao->fifo_write_ptr = w;
    ao->bytes_in_buffer += n;
    pthread_mutex_unlock( &ao->mutex );
    return 1;
}

static int ao_fifo_delay( ao_driver_t *this_gen )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;
    return (ao_fifo_arts_delay() * ao->audio->sample_rate / 1000)
	   + (ao->bytes_in_buffer / ao->bytes_per_frame);
}

static void ao_fifo_close( ao_driver_t *this_gen )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;

    pthread_mutex_lock( &ao->mutex );

    while (ao->bytes_in_buffer && ao->enabled)
    {
	pthread_cond_wait( &ao->cond, &ao->mutex );
    }
    pthread_mutex_unlock( &ao->mutex );

    ao_fifo_wait_close( this_gen );

    if (ao->fifo)
    {
	free( ao->fifo );
	ao->fifo = NULL;
    }
}

static uint32_t ao_fifo_get_capabilities( ao_driver_t *this_gen )
{
    return ((fifo_driver_t *)this_gen)->capabilities;
}

static void ao_fifo_exit( ao_driver_t *this_gen )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;

    ao_fifo_close( this_gen );

    pthread_cond_destroy( &ao->cond );
    pthread_mutex_destroy( &ao->mutex );
    free( ao );
}

static int ao_fifo_get_property( ao_driver_t *this_gen, int property )
{
    return 0;
}

static int ao_fifo_set_property( ao_driver_t *this_gen, int property, int value )
{
    return ~value;
}

static int ao_fifo_control( ao_driver_t *this_gen, int cmd, ... )
{
    return 0;
}

ao_driver_t *init_audio_out_plugin( config_values_t *config, xine_arts_audio *audio )
{
    fifo_driver_t *ao = (fifo_driver_t *)malloc( sizeof(fifo_driver_t) );

    ao->audio		= audio;
    ao->fifo		= NULL;
    ao->fifo_read_ptr	= 0;
    ao->fifo_write_ptr	= 0;
    ao->bytes_in_buffer	= 0;
    ao->enabled		= 0;
    ao->locked		= 0;
    ao->capabilities	= (AO_CAP_MODE_MONO | AO_CAP_MODE_STEREO);

    ao->ao_driver.get_capabilities	= ao_fifo_get_capabilities;
    ao->ao_driver.get_property		= ao_fifo_get_property;
    ao->ao_driver.set_property		= ao_fifo_set_property;
    ao->ao_driver.open			= ao_fifo_open;
    ao->ao_driver.num_channels		= ao_fifo_num_channels;
    ao->ao_driver.bytes_per_frame	= ao_fifo_bytes_per_frame;
    ao->ao_driver.delay			= ao_fifo_delay;
    ao->ao_driver.write			= ao_fifo_write;
    ao->ao_driver.close			= ao_fifo_close;
    ao->ao_driver.exit			= ao_fifo_exit;
    ao->ao_driver.get_gap_tolerance	= ao_fifo_get_gap_tolerance;
    ao->ao_driver.control		= ao_fifo_control;

    pthread_cond_init( &ao->cond, NULL );
    pthread_mutex_init( &ao->mutex, NULL );

    return (ao_driver_t *)ao;
}

unsigned long ao_fifo_read( ao_driver_t *this_gen, unsigned char **buffer,
			    unsigned long samples )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;
    uint32_t r, m, n, a;

    /* lock audio, return on audio disabled */
    pthread_mutex_lock( &ao->mutex );

    if (!ao->enabled || !ao->bytes_in_buffer)
    {
	if (ao->locked)
	{
	    ao->locked = 0;
	    pthread_cond_broadcast( &ao->cond );
	}
	pthread_mutex_unlock( &ao->mutex );
	return 0;
    }
    ao->locked = 1;
    pthread_mutex_unlock( &ao->mutex );

    r = ao->fifo_read_ptr;
    m = (ao->buffer_size - r);
    n = (samples * ao->bytes_per_frame);
    a = ao->bytes_in_buffer;

    if (n > a)
    {
	fprintf( stderr, "Warning: Sound Buffer Underflow!\n" );
	n = a;
    }
    if (n > m)
    {
	/* copy samples from front to end of buffer */
	memcpy( ao->fifo + ao->buffer_size, ao->fifo, (n - m) );
    }
    *buffer = (ao->fifo + r);
    return n;
}

void ao_fifo_flush( ao_driver_t *this_gen, unsigned long samples )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;
    uint32_t r, m, n, a;

    pthread_mutex_lock( &ao->mutex );

    if (ao->enabled)
    {
	r = ao->fifo_read_ptr;
	m = (ao->buffer_size - r);
	n = (samples * ao->bytes_per_frame);
	a = ao->bytes_in_buffer;

	if (n > a)
	    n = a;

	a -= n;
	r += n;

	if (r >= ao->buffer_size)
	    r -= ao->buffer_size;

	ao->fifo_read_ptr = r;
	ao->bytes_in_buffer = a;
    }
    ao->locked = 0;
    pthread_cond_broadcast( &ao->cond );
    pthread_mutex_unlock( &ao->mutex );
}

void ao_fifo_wait_close( ao_driver_t *this_gen )
{
    fifo_driver_t *ao = (fifo_driver_t *)this_gen;

    pthread_mutex_lock( &ao->mutex );
    ao->enabled = 0;
    pthread_cond_broadcast( &ao->cond );
    pthread_mutex_unlock( &ao->mutex );

    while (ao->locked)
    {
	struct timespec ts;

	ts.tv_sec = 0;
	ts.tv_nsec = 50000000;	/* 50ms */

	nanosleep( &ts, NULL );
	pthread_mutex_lock( &ao->mutex );
	pthread_cond_broadcast( &ao->cond );
	pthread_mutex_unlock( &ao->mutex );
    }
}
