------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               T B U I L D                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.63 $                             --
--                                                                          --
--          Copyright (C) 1992-1997, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains various utility procedures to assist in
--  building specific types of tree nodes.

with Types; use Types;

package Tbuild is

   function Make_DT_Component
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      I    : Positive)
      return Node_Id;
   --  Gives a reference to the Ith component of the Dispatch Table of
   --  a given Tagged Type.
   --
   --  I = 1    --> Inheritance_Depth
   --  I = 2    --> Tags (array of ancestors)
   --  I = 3, 4 --> predefined primitive
   --            function _Size (X : Typ) return Long_Long_Integer;
   --            function _Equality (X : Typ; Y : Typ'Class) return Boolean;
   --  I >= 5   --> User-Defined Primitive Operations
   --

   function Make_DT_Access
     (Loc : Source_Ptr; Rec : Node_Id; Typ : Entity_Id) return Node_Id;
   --  Create an access to the Dispatch Table by using the Tag field
   --  of a tagged record : Acc_Dt (Rec.tag).all

   function Make_Unsuppress_Block
     (Loc   : Source_Ptr;
      Check : Name_Id;
      Stmts : List_Id)
      return  Node_Id;
   --  Build a block with a pragma Suppress on 'Check'. Stmts is the
   --  statements list that needs protection against the check

   function New_Constraint_Error (Loc : Source_Ptr) return Node_Id;
   --  This function builds a tree corresponding to the Ada statement
   --  "raise Constraint_Error" and returns the root of this tree,
   --  the N_Raise_Statement node.

   function New_External_Name
     (Related_Id   : Name_Id;
      Suffix       : Character := ' ';
      Suffix_Index : Nat       := 0;
      Prefix       : Character := ' ')
      return         Name_Id;
   --  Builds a new entry in the names table of the form
   --
   --    [Prefix  &] Related_Id [& Suffix] [& Suffix_Index'Image]
   --
   --  where Prefix is prepended only if Prefix is non-blank (in which case
   --  it must be an upper case letter other than O,Q,U,W (which are used for
   --  identifier encoding, see Namet), and T is reserved for use by implicit
   --  types. and X is reserved for use by debug type encoding (see package
   --  Exp_Dbug). Suffix_Index'Image is appended only if the value is non-zero.
   --  Suffix is also a single upper case letter other than O,Q,U,W,X and is a
   --  required parameter (T is permitted). The constructed name is stored
   --  using Find_Name so that it can be located using a subsequent Find_Name
   --  operation (i.e. it is properly hashed into the names table). The upper
   --  case letter given as the Suffix argument ensures that the name does
   --  not clash with any Ada identifier name. These generated names are
   --  permitted, but not required, to be made public by setting the flag
   --  Is_Public in the associated entity. Note: the reason that Prefix is
   --  Prefix is last is that it is almost always omitted. The notable case
   --  of Prefix being non-null is when it is 'T' for an implicit type.

   function New_External_Name
     (Suffix       : Character;
      Suffix_Index : Nat)
      return         Name_Id;
   --  Builds a new entry in the names table of the form
   --    Suffix & Suffix_Index'Image
   --  where Suffix is a single upper case letter other than O,Q,U,W,X and is
   --  a required parameter (T is permitted). The constructed name is stored
   --  using Find_Name so that it can be located using a subsequent Find_Name
   --  operation (i.e. it is properly hashed into the names table). The upper
   --  case letter given as the Suffix argument ensures that the name does
   --  not clash with any Ada identifier name. These generated names are
   --  permitted, but not required, to be made public by setting the flag
   --  Is_Public in the associated entity.

   function New_Internal_Name (Id_Char : Character) return Name_Id;
   --  Id_Char is an upper case letter other than O,Q,U,W (which are reserved
   --  for identifier encoding (see Namet package for details) and X which is
   --  used for debug encoding (see Exp_Dbug). The letter T is permitted, but
   --  is reserved by convention for the case of internally generated types.
   --  The result of the call is a new generated unique name of the form XyyyU
   --  where X is Id_Char, yyy is a unique serial number, and U is either a
   --  lower case s or b indicating if the current unit is a spec or a body.
   --
   --  The name is entered into the names table using Name_Enter rather than
   --  Name_Find, because there can never be a need to locate the entry using
   --  the Name_Find procedure later on. Names created by New_Internal_Name
   --  are guaranteed to be consistent from one compilation to another (i.e.
   --  if the identical unit is compiled with a semantically consistent set
   --  of sources, the numbers will be consistent. This means that it is fine
   --  to use these as public symbols.

   function New_Suffixed_Name
     (Related_Id : Name_Id;
      Suffix     : String)
      return       Name_Id;
   --  This function is used to create special suffixed names used by the
   --  debugger. Suffix is a string of upper case letters, used to construct
   --  the required name. For instance, the special type used to record the
   --  fixed-point small is called typ_SMALL where typ is the name of the
   --  fixed-point type (as passed in Related_Id), and Suffix is "SMALL".

   function New_Occurrence_Of
     (Def_Id : Entity_Id;
      Loc    : Source_Ptr)
      return   Node_Id;
   --  New_Occurrence_Of creates an N_Identifier node which is an
   --  occurrence of the defining identifier which is passed as its
   --  argument. The Entity and Etype of the result are set from
   --  the given defining identifier as follows: Entity is simply
   --  a copy of Def_Id. Etype is a copy of Def_Id for types, and
   --  a copy of the Etype of Def_Id for other entities.

   function New_Reference_To
     (Def_Id : Entity_Id;
      Loc    : Source_Ptr)
      return   Node_Id;
   --  This is like New_Occurrence_Of, but it does not set the Etype field.
   --  It is used from the expander, where Etype fields are generally not set,
   --  since they are set when the expanded tree is reanalyzed.

   function Checks_Off (N : Node_Id) return Node_Id;
   pragma Inline (Checks_Off);
   --  Returns an N_Unchecked_Expression node whose expression is the given
   --  argument. The results is a subexpression identical to the argument,
   --  except that it will be analyzed and resolved with checks off.

   function Convert_To (Typ : Entity_Id; Expr : Node_Id) return Node_Id;
   --  Returns an expression that represents the result of a checked convert
   --  of expression Exp to type T. If the base type of Exp is T, then no
   --  conversion is required, and Exp is returned unchanged. Otherwise an
   --  N_Type_Conversion node is constructed to convert the expression.
   --  If an N_Type_Conversion node is required, Relocate_Node is used on
   --  Exp. This means that it is safe to replace a node by a Convert_To
   --  of itself to some other type.

   function OK_Convert_To (Typ : Entity_Id; Expr : Node_Id) return Node_Id;
   --  Like Convert_To, except that a conversion node is always generated,
   --  and the Conversion_OK flag is set on this conversion node.

   function Unchecked_Convert_To
     (Typ  : Entity_Id;
      Expr : Node_Id)
      return Node_Id;
   --  Like Convert_To, but if a conversion is actually needed, constructs
   --  an N_Unchecked_Type_Conversion node to do the required conversion.

end Tbuild;
