/*
 * KerberosLogin.h
 *
 * $Header: /afs/andrew/system/cvs/src/sasl/mac/CommonKClient/mac_kclient3/Headers/KerberosLogin/KerberosLogin.h,v 1.2 2001/12/04 02:05:52 rjs3 Exp $
 *
 */
 
/* 
 * This file contains part of the login library API. See
 * <http://web.mit.edu/macdev/mit/lib/Login/doc/API.html>
 * for API documentation
 */

#ifndef __KERBEROSLOGIN__
#define __KERBEROSLOGIN__


/*
 *
 * Constants
 *
 */

/* Kerberos versions */
enum KLEKerberosVersion {
	kerberosVersion_Any		= 0,
	kerberosVersion_V4		= 1,
	kerberosVersion_V5		= 2,
	kerberosVersion_All		= 0xFFFFFFFF
};

/* dialog identifier constants */
enum KLEDialogIdentifiers {
	loginLibrary_LoginDialog,
	loginLibrary_OptionsDialog,
	loginLibrary_ChangePasswordDialog,
	loginLibrary_ProgressDialog,
	loginLibrary_PrompterDialog
};

/* Login dialog items */
enum KLELoginDialogItems {
	loginDialog_Username,
	loginDialog_Password,
	loginDialog_Realm,
	loginDialog_TicketLifetime,
	loginDialog_ForwardableTicket
};

/* Password dialog items */
enum KLEChangePasswordDialogItems {
	changePasswordDialog_OldPassword,
	changePasswordDialog_NewPassword,
	changePasswordDialog_VerifyPassword
};

/* Option identifier constants */
enum KLEDefaultLoginOptions {
	/* Dialog state options */
	loginOption_LoginName                    = 'name',
	loginOption_LoginInstance                = 'inst',
	loginOption_AdvancedLoginMode            = 'adv ',
	loginOption_ShowTicketLifetime           = 'life',
	loginOption_ShowForwardableTicket        = 'forw',
	loginOption_ShowProxiableTicket      	 = 'prox',
	
	/* Initial values and ranges */
	loginOption_RememberPrincipal            = 'prin',
	loginOption_RememberExtras               = 'extr',
	
	loginOption_MinimalTicketLifetime        = '-lif',
	loginOption_MaximalTicketLifetime        = '+lif',
	loginOption_DefaultTicketLifetime        = '0lif',
	loginOption_LongTicketLifetimeDisplay    = 'hms ',
	
	loginOption_DefaultForwardableTicket     = '0fwd',
	loginOption_DefaultProxiableTicket		 = '0prx'
};

/* Login mode identifier constants (for loginOption_AdvancedLoginMode) */
enum KLELoginMode {
	loginMode_Basic							 = 1,
	loginMode_Advanced						 = 2
};

/* Realm list constants */
enum KLERealmListIndexes {
	realmList_Start		= 0,
	realmList_End		= 0xFFFF
};

#define klFirstError	19276
#define klLastError		19876

/* Error codes */
enum KLEStatus {
	klNoErr										=	0,

	/* parameter errors */
	klParameterErr								=	19276,
	klBadPrincipalErr,
	klBadPasswordErr,
	klBadLoginOptionsErr,
	klInvalidVersionErr,
	
	/* Runtime Login errors */
	klUserCanceledErr							=	19476,
	klMemFullErr,
	klPreferencesReadErr,
	klPreferencesWriteErr,
	klV5InitializationFailedErr,
	klPrincipalDoesNotExistErr,
	klSystemDefaultDoesNotExistErr,
	klCredentialsExpiredErr,
	klNoRealmsErr,
	klRealmDoesNotExistErr,
	klNoCredentialsErr,
	klCredentialsBadAddressErr,
	klCacheDoesNotExistErr,
	
	/* Get/SetKerberosOption errors */
	klBufferTooSmallErr							=	19376,
	klBufferTooLargeErr,
	klInvalidOptionErr,
	klBadOptionValueErr,
	
	/* Password changing errors */
	klPasswordMismatchErr						=	19576,
	klInsecurePasswordErr,
	klPasswordChangeFailedErr,
	
	/* Dialog errors */
	klDialogDoesNotExistErr						=	19676,
	klDialogAlreadyExistsErr,
	klNotInForegroundErr,
	klNoAppearanceErr,
	klFatalDialogErr,
	klCarbonUnavailableErr,
        
	/* Login IPC errors */
	klCantContactServerErr						=	19776

};

#ifndef rez  /* This stuff will confuse rez */

#include <KerberosSupport/KerberosConditionalMacros.h>

#if TARGET_API_MAC_OSX && TARGET_API_MAC_CARBON
    #include <Carbon/Carbon.h>
#elif TARGET_API_MAC_OS8 || TARGET_API_MAC_CARBON 
    #include <Dialogs.h>
    #include <Events.h>
    #include <MacTypes.h>
#else
	#error "Unknown OS"
#endif

#if PRAGMA_ONCE
#pragma once
#endif

#if PRAGMA_IMPORT
#pragma import on
#endif

#ifdef __cplusplus
extern "C" {
#endif

#if PRAGMA_STRUCT_ALIGN
	#pragma options align=mac68k
#elif PRAGMA_STRUCT_PACKPUSH
	#pragma pack(push, 2)
#elif PRAGMA_STRUCT_PACK
	#pragma pack(2)
#endif


/*
 *
 * Types
 *
 */
 
typedef	OSStatus	KLStatus;					/* one of KLEStatus 																	*/
typedef	UInt32		KLKerberosVersion;			/* one of KLEKerberosVersion 															*/
typedef	UInt32		KLDefaultLoginOption;		/* one of KLEDefaultLoginOptions														*/
typedef	UInt32		KLLoginMode;				/* one of KLELoginMode																	*/
typedef	UInt32		KLDialogIdentifier;			/* one of KLEDialogIdentifiers															*/
typedef	UInt32		KLIndex;					/* index (used for the realm list)														*/
typedef	UInt32		KLLifetime;					/* Lifetime in seconds																	*/
typedef	UInt32		KLTime;						/* Unix time (seconds since 1/1/1970 00:00:00 GMT)										*/
typedef	UInt32		KLSize;						/* size of a buffer (KLG/SetDefaultLoginOptions) or realm list (CountKerberosRealms)	*/
typedef	UInt32		KLRefCon;					/* application ref con																	*/
typedef	Boolean		KLBoolean;					/* true or false!																		*/
typedef	SInt16		KLSInt16;					/* used for Darwin-compat for KLApplicationOptions										*/

/* Callback API for Kerberos Login event filter */
/* Must be the same as an Idle Library event filter */
/* Callback API for Event handler proc for idle loop */
typedef CALLBACK_API (Boolean, KLEventFilterProcPtr) (const EventRecord *theEvent, KLRefCon appData);

/* Procinfo for Login Library event filter */
enum {
	uppKLEventFilterProcInfo = kPascalStackBased |
		RESULT_SIZE (sizeof (Boolean)) |
		STACK_ROUTINE_PARAMETER (1, SIZE_CODE (sizeof (const EventRecord *))) |
		STACK_ROUTINE_PARAMETER (2, SIZE_CODE (sizeof (KLRefCon)))
};

#if !TARGET_API_MAC_CARBON
	/* UPP for Kerberos Login event filter */
	typedef STACK_UPP_TYPE (KLEventFilterProcPtr) KLEventFilterUPP;
	
	#define	NewKLEventFilterProc(userRoutine) 			\
		(KLEventFilterUPP) NewRoutineDescriptor((ProcPtr)(userRoutine), uppKLEventFilterProcInfo, GetCurrentArchitecture())

	/* How to call the event Handler UPPs */
	#define CallKLEventFilterProc(userRoutine, theEvent, appData)			\
		((Boolean)CALL_TWO_PARAMETER_UPP ((userRoutine), uppKLEventFilterProcInfo, theEvent, appData))

#else
	typedef KLEventFilterProcPtr KLEventFilterUPP;
	
	#define	NewKLEventFilterProc(userRoutine) 			\
		userRoutine

	#define CallKLEventFilterProc(userRoutine, theEvent, appData)			\
		((userRoutine) (theEvent, appData))
#endif

/* Application options */
typedef struct {
	KLEventFilterUPP 	eventFilter;
	KLRefCon 			eventFilterAppData;
	KLSInt16 			realmsPopupMenuID;
	KLSInt16 			loginModeMenuID;
} KLApplicationOptions;

/* Principal information */
struct OpaqueKLPrincipal;
typedef struct OpaqueKLPrincipal 	* KLPrincipal;

/* Login Options */
struct OpaqueKLLoginOptions;
typedef struct OpaqueKLLoginOptions	* KLLoginOptions;


/*
 *
 * Functions
 *
 */

/* Kerberos Login high-level API */
KLStatus KLAcquireTickets (				
		KLPrincipal		  inPrincipal, 
		KLPrincipal		 *outPrincipal, 
		char 			**outCredCacheName);

KLStatus KLAcquireNewTickets (
		KLPrincipal 	  inPrincipal, 
		KLPrincipal 	 *outPrincipal, 
		char 			**outCredCacheName);

KLStatus KLDestroyTickets (KLPrincipal inPrincipal);

KLStatus KLChangePassword (KLPrincipal inPrincipal);


/* Kerberos Login dialog low level functions */

KLStatus KLAcquireTicketsWithPassword (
		KLPrincipal		  inPrincipal,
		KLLoginOptions	  inLoginOptions,
		const char		 *inPassword,
		char 			**outCredCacheName);

KLStatus KLAcquireNewTicketsWithPassword (
		KLPrincipal		  inPrincipal,
		KLLoginOptions	  inLoginOptions,
		const char		 *inPassword,
		char 			**outCredCacheName);

KLStatus KLLastChangedTime (KLTime	*outLastChangedTime);

KLStatus KLCacheHasValidTickets (
		KLPrincipal		 	  inPrincipal,
		KLKerberosVersion	  inKerberosVersion,
		KLBoolean 			 *outFoundValidTickets,
		KLPrincipal			 *outPrincipal,
		char				**outCredCacheName);

KLStatus KLTicketStartTime (
		KLPrincipal					 inPrincipal,
		KLKerberosVersion			 inKerberosVersion,
		KLTime						*outStartTime);

KLStatus KLTicketExpirationTime (
		KLPrincipal			 inPrincipal,
		KLKerberosVersion	 inKerberosVersion,
		KLTime				*outExpirationTime);
     
KLStatus KLSetSystemDefaultCache (KLPrincipal inPrincipal);

KLStatus KLHandleError (
		KLStatus				inError,
		KLDialogIdentifier		inDialogIdentifier,
		Boolean					inShowAlert);

KLStatus KLGetErrorString (
		KLStatus		  inError,
		char			**outErrorString);

KLStatus KLCancelAllDialogs (void);

/* Kerberos change password dialog low level functions */

KLStatus KLChangePasswordWithPasswords (
		KLPrincipal		 inPrincipal,
		const char		*inOldPassword,
		const char		*inNewPassword);

/* Application Configuration functions */

KLStatus KLSetApplicationOptions (const KLApplicationOptions *inAppOptions);

KLStatus KLGetApplicationOptions (KLApplicationOptions *outAppOptions);


/* Library configuration functions */
KLStatus KLGetDefaultLoginOption (     
		const KLDefaultLoginOption	 inOption,
		void						*ioBuffer,
		KLSize						*ioBufferSize);

KLStatus KLSetDefaultLoginOption (
		const KLDefaultLoginOption	 inOption,
		const void					*inBuffer,
		const KLSize				 inBufferSize);

/* Realm configuration functions */

KLStatus KLFindKerberosRealmByName (
		const char		*inRealmName,
		KLIndex			*outIndex);

KLStatus KLGetKerberosRealm (
		KLIndex			  inIndex,
		char			**outRealmName);

KLStatus KLSetKerberosRealm (
		KLIndex			 inIndex,
		const char		*inRealmName);

KLStatus KLRemoveKerberosRealm (UInt32 inIndex);

KLStatus KLInsertKerberosRealm (
		KLIndex			 inInsertBeforeIndex,
		const char		*inRealmName);
		
KLStatus KLRemoveAllKerberosRealms (void);
		
KLSize KLCountKerberosRealms (void);
        
KLStatus KLGetKerberosDefaultRealm(KLIndex *outIndex);
     
KLStatus KLGetKerberosDefaultRealmByName (char **outRealmName);
        
KLStatus KLSetKerberosDefaultRealm (KLIndex inIndex);
        
KLStatus KLSetKerberosDefaultRealmByName (const char *inRealm);

/* KLPrincipal functions */

KLStatus KLCreatePrincipalFromTriplet(
		const char		*inName,
		const char		*inInstance,
		const char		*inRealm,
		KLPrincipal 	*outPrincipal);

KLStatus KLCreatePrincipalFromString(
		const char				*inFullPrincipal,
		KLKerberosVersion		 inKerberosVersion,
		KLPrincipal				*outPrincipal);
    
KLStatus KLGetTripletFromPrincipal(
		KLPrincipal		  inPrincipal,
		char			**outName,
		char			**outInstance,
		char			**outRealm);

KLStatus KLGetStringFromPrincipal(
		KLPrincipal			  inPrincipal,
		KLKerberosVersion	  inKerberosVersion,
		char				**outFullPrincipal);

KLStatus KLGetDisplayStringFromPrincipal(
		KLPrincipal			  inPrincipal,
		KLKerberosVersion	  inKerberosVersion,
		char				**outFullPrincipal);

KLStatus KLComparePrincipal(
		KLPrincipal		 inFirstPrincipal,
		KLPrincipal		 inSecondPrincipal,
		KLBoolean		*outAreEquivalent);

KLStatus KLDisposePrincipal(KLPrincipal inPrincipal);

/* KLLoginOptions functions */

KLStatus KLCreateLoginOptions (KLLoginOptions 	*outOptions);

KLStatus KLLoginOptionsSetTicketLifetime (
		KLLoginOptions	ioOptions,
		KLLifetime		inTicketLifetime);

KLStatus KLLoginOptionsSetForwardable (
		KLLoginOptions	ioOptions,
		KLBoolean		inForwardable);
		
KLStatus KLLoginOptionsSetProxiable (
		KLLoginOptions	ioOptions,
		KLBoolean		inProxiable);
		
KLStatus KLDisposeLoginOptions(KLLoginOptions ioOptions);


/* Misc function */

KLStatus KLDisposeString(char *inStringToDispose);

#if PRAGMA_STRUCT_ALIGN
	#pragma options align=reset
#elif PRAGMA_STRUCT_PACKPUSH
	#pragma pack(pop)
#elif PRAGMA_STRUCT_PACK
	#pragma pack()
#endif

#ifdef PRAGMA_IMPORT_OFF
#pragma import off
#elif PRAGMA_IMPORT
#pragma import reset
#endif

#ifdef __cplusplus
}
#endif

#endif /* Rez */	

#endif /* __KERBEROSLOGIN__ */

