#!/usr/bin/env python


import sys, string
from qt import *

#
## Constructs an analog clock widget that uses an internal QTimer.
#
def QMIN( x, y ):
  if y > x:
    return y
  return x

#
## Constructs an analog clock widget that uses an internal QTimer.
#

class AnalogClock( QWidget ):
    def __init__( self, *args ):
        apply( QWidget.__init__, (self,) + args )
        self.time = QTime.currentTime()     # get current time
        internalTimer = QTimer( self )      # create internal timer
        self.connect( internalTimer, SIGNAL("timeout()"), self.timeout )
        internalTimer.start( 5000 )         # emit signal every 5 seconds

#
## The QTimer::timeout() signal is received by this slot.
#

    def timeout( self ):
        new_time = QTime.currentTime()                # get the current time
        if new_time.minute() != self.time.minute():    # minute has changed
            self.update()

#
## The clock is painted using a 1000x1000 square coordinate system.
#
    def paintEvent( self, qe ):            # paint clock
        if not self.isVisible():                     # is is invisible
            return
        self.time = QTime.currentTime()             # save current time

        pts = QPointArray()
        paint = QPainter( self )
        paint.setBrush( self.foregroundColor() )    # fill with foreground color

        cp = QPoint( self.rect().center() )         # widget center point
        d = QMIN( self.width(), self.height() )     # we want a circular clock

        matrix = QWMatrix()                         # setup transformation matrix
        matrix.translate( cp.x(), cp.y() )          # origin at widget center
        matrix.scale( d / 1000.0, d / 1000.0 )      # scale coordinate system

        h_angle = 30 * ( self.time.hour() % 12 - 3 ) + self.time.minute() / 2
        matrix.rotate( h_angle )                    # rotate to draw hour hand
        paint.setWorldMatrix( matrix )
        pts.setPoints( [ -20,0, 0,-20, 300,0, 0,20 ] )
        paint.drawPolygon( pts )                    # draw hour hand
        matrix.rotate( -h_angle )                   # rotate back to zero

        m_angle = ( self.time.minute() - 15 ) * 6
        matrix.rotate( m_angle )                    # rotate to draw minute hand
        paint.setWorldMatrix( matrix )
        pts.setPoints( [ -10,0, 0,-10, 400,0, 0,10 ] )
        paint.drawPolygon( pts )                    # draw minute hand
        matrix.rotate( -m_angle )                   # rotate back to zero

        for i in range( 0, 12 ):                     # draw hour lines
            paint.setWorldMatrix( matrix )
            paint.drawLine( 450,0, 500,0 )
            matrix.rotate( 30 )


class DigitalClock( QLCDNumber ):
    def __init__( self, *args ): 
        apply( QLCDNumber.__init__,(self,) + args )
        self.showingColon = 0
        self.setFrameStyle(QFrame.Panel | QFrame.Raised)
        self.setLineWidth( 2 )
        self.showTime()
        self.normalTimer = self.startTimer( 500 )
        self.showDateTimer = -1

    def timerEvent( self, e ):
        if e.timerId() == self.showDateTimer:
            self.stopDate()
        else:
            if self.showDateTimer == -1:
                self.showTime()
  
    def mousePressEvent( self, e ):
        if e.button() == Qt.LeftButton:
            self.showDate()

    def showDate( self ):
        if self.showDateTimer != -1:
            return
        d = QDate.currentDate()
        self.display('%2d %2d' % (d.month(), d.day()))
        self.showDateTimer = self.startTimer(2000)

    def stopDate( self ):
        self.killTimer(self.showDateTimer)
        self.showDateTimer = -1
        self.showTime()

    def showTime( self ):
        self.showingColon = not self.showingColon
        s = list(str(QTime.currentTime().toString())[:5]) #.left(5)
        if not self.showingColon:
            s[2] = ' '
        if s[0] == '0':
            s[0] = ' '
        s = string.join(s,'')
        self.display( s )

    def QMIN( x, y ):
        if y > x:
            return y
        return x

TRUE  = 1
FALSE = 0
MOVIEFILENAME = "trolltech.gif"

#
# WidgetView contains lots of Qt widgets.
#

class WidgetView ( QWidget ):
    def __init__( self, *args ):
         apply( QWidget.__init__, (self,) + args )
         
         # Set the window caption/title
         self.setCaption( "Qt Widgets Demo Application" )

         # Install an application-global event filter
         qApp.installEventFilter( self )

         # Create a layout to position the widgets
         self.topLayout = QVBoxLayout( self, 10 )

         # Create a grid layout to hold most of the widgets
         self.grid = QGridLayout( 6, 3 )

         # This layout will get all of the stretch
         self.topLayout.addLayout( self.grid, 10 )

         # Create a menubar
         self.menubar = QMenuBar( self )
         #self.menubar.setSeparator( QMenuBar.InWindowsStyle )
         self.menubar.setSeparator( 1 )
         
         # Create an easter egg
         QToolTip.add( self.menubar, QRect( 0, 0, 2, 2 ), "easter egg" )

         self.popup = QPopupMenu()
         self.id = self.popup.insertItem( "&New" )
         self.popup.setItemEnabled( self.id, FALSE )
         self.id = self.popup.insertItem( "&Open" )
         self.popup.setItemEnabled( self.id, FALSE )
         self.popup.insertSeparator()
         self.popup.insertItem( "&Quit", qApp, SLOT("quit()"), Qt.CTRL+Qt.Key_Q )

         self.menubar.insertItem( "&File", self.popup )

         # Must tell the layout about a menubar in a widget
         self.topLayout.setMenuBar( self.menubar )

         # Create an analog and a digital clock
         self.aclock = AnalogClock( self )
         self.aclock.resize( 50, 50 )
         self.dclock = DigitalClock( self )
         self.dclock.setMaximumWidth( 200 )
         self.grid.addWidget( self.aclock, 0, 2 )
         self.grid.addWidget( self.dclock, 1, 2 )

         # Give the dclock widget a blue palette
         col = QColor()
         col.setRgb( 0xaa, 0xbe, 0xff )
         self.dclock.setPalette( QPalette( col ) )

         # make tool tips for both of them
         QToolTip.add( self.aclock, "custom widget: analog clock" )
         QToolTip.add( self.dclock, "custom widget: digital clock" )

         # Create a push button.
         self.pb = QPushButton( self, "button1" )	# create button 1
         self.pb.setText( "Push button 1" )
         self.pb.setFixedHeight( self.pb.sizeHint().height() )
         self.grid.addWidget( self.pb, 0, 0, Qt.AlignVCenter )
         self.connect( self.pb, SIGNAL("clicked()"), self.button1Clicked )
         QToolTip.add( self.pb, "push button 1" )
         self.pm = QPixmap()
         self.pix = self.pm.load( "qt.png" )		# load pixmap for button 2
         if not self.pix:
             QMessageBox.information( None, "Qt Widgets Example",
                                      "Could not load the file \"qt.png\", which\n"
                                      "contains an icon used...\n\n"
                                      "The text \"line 42\" will be substituted.",
                                      QMessageBox.Ok + QMessageBox.Default )

         # Create a label containing a QMovie
         self.movielabel = QLabel( self, "label0" )
         self.movie = QMovie( MOVIEFILENAME )
         self.movie.connectStatus( self.movieStatus )
         self.movie.connectUpdate( self.movieUpdate )
         self.movielabel.setFrameStyle( QFrame.Box | QFrame.Plain )
         self.movielabel.setMovie( self.movie )
         self.movielabel.setMargin( 0 )
         self.movielabel.setFixedSize( 128 + self.movielabel.frameWidth() * 2,
                                        64 + self.movielabel.frameWidth() * 2 )
         self.grid.addWidget( self.movielabel, 0, 1, Qt.AlignCenter )
         QToolTip.add( self.movielabel, "movie" )

         # Create a group of check boxes
         self.bg = QButtonGroup( self, "checkGroup" )
         self.bg.setTitle( "Check Boxes" )
         self.grid.addWidget( self.bg, 1, 0 )

         # Create a layout for the check boxes
         self.vbox = QVBoxLayout(self.bg, 10)

         self.vbox.addSpacing( self.bg.fontMetrics().height() )

         self.cb = range(3)
         self.cb[0] = QCheckBox( self.bg )
         self.cb[0].setText( "Read" )
         self.vbox.addWidget( self.cb[0] )
         self.cb[0].setMinimumSize( self.cb[0].sizeHint() )
         self.cb[1] = QCheckBox( self.bg )
         self.cb[1].setText( "Write" )
         self.vbox.addWidget( self.cb[1] )
         self.cb[1].setMinimumSize( self.cb[1].sizeHint() )
         self.cb[2] = QCheckBox( self.bg )
         self.cb[2].setText( "Execute" )
         self.cb[2].setMinimumSize( self.cb[2].sizeHint() )
         self.vbox.addWidget( self.cb[2] )
         self.bg.setMinimumSize( self.bg.childrenRect().size() )
         self.vbox.activate()

         self.connect( self.bg, SIGNAL("clicked(int)"), self.checkBoxClicked )
         
         QToolTip.add( self.cb[0], "check box 1" )
         QToolTip.add( self.cb[1], "check box 2" )
         QToolTip.add( self.cb[2], "check box 3" )

         # Create a group of radio buttons
         self.bg = QButtonGroup( self, "radioGroup" )
         self.bg.setTitle( "Radio buttons" )

         self.grid.addWidget( self.bg, 1, 1 )

         # Create a layout for the radio buttons
         self.vbox = QVBoxLayout( self.bg, 10 )

         self.vbox.addSpacing( self.bg.fontMetrics().height() )
         self.rb = QRadioButton( self.bg )
         self.rb.setText( "&AM" )
         self.rb.setChecked( TRUE )
         self.vbox.addWidget( self.rb )
         self.rb.setMinimumSize( self.rb.sizeHint() )
         QToolTip.add( self.rb, "radio button 1" )
         self.rb = QRadioButton( self.bg )
         self.rb.setText( "&FM" )
         self.vbox.addWidget( self.rb )
         self.rb.setMinimumSize( self.rb.sizeHint() )
         QToolTip.add( self.rb, "radio button 2" )
         self.rb = QRadioButton( self.bg )
         self.rb.setText( "&Short Wave" )
         self.vbox.addWidget( self.rb )
         self.rb.setMinimumSize( self.rb.sizeHint() )
         self.vbox.activate()

         self.connect( self.bg, SIGNAL("clicked(int)"), self.radioButtonClicked )
         QToolTip.add( self.rb, "radio button 3" )

         # Create a list box
         self.lb = QListBox( self, "listBox" )
         for i in range( 0, 100, 1 ): # fill list box
             txt = QString()
             txt = "line %d" % i
             if i == 42 and self.pix:
                 self.lb.insertItem( self.pm )
             else:
                 self.lb.insertItem( txt )
    
         self.grid.addMultiCellWidget( self.lb, 2, 4, 0, 0 )
         self.connect( self.lb, SIGNAL("selected(int)"), self.listBoxItemSelected )
         QToolTip.add( self.lb, "list box" )

         self.vbox = QVBoxLayout( 8 )
         self.grid.addLayout( self.vbox, 2, 1 )

         # Create a slider
         self.sb = QSlider( 0, 300, 1, 100, QSlider.Horizontal, self, "Slider" )
         #self.sb.setTickmarks( QSlider.Below )
         self.sb.setTickmarks( 1 )
         self.sb.setTickInterval( 10 )
         #self.sb.setFocusPolicy( QWidget.TabFocus )
         self.sb.setFocusPolicy( 1 )
         self.sb.setFixedHeight( self.sb.sizeHint().height() )
         self.vbox.addWidget( self.sb )

         self.connect( self.sb, SIGNAL("valueChanged(int)"), self.sliderValueChanged )
         QToolTip.add( self.sb, "slider" )

         # Create a combo box
         self.combo = QComboBox( FALSE, self, "comboBox" )
         self.combo.insertItem( "darkBlue" )
         self.combo.insertItem( "darkRed" )
         self.combo.insertItem( "darkGreen" )
         self.combo.insertItem( "blue" )
         self.combo.insertItem( "red" )
         self.combo.setFixedHeight( self.combo.sizeHint().height() )
         self.vbox.addWidget( self.combo )
         self.connect( self.combo, SIGNAL("activated(int)"), self.comboBoxItemActivated )
         QToolTip.add( self.combo, "read-only combo box" )

         # Create an editable combo box
         self.edCombo = QComboBox( TRUE, self, "edComboBox" )
         self.edCombo.insertItem( "Permutable" )
         self.edCombo.insertItem( "Malleable" )
         self.edCombo.insertItem( "Adaptable" )
         self.edCombo.insertItem( "Alterable" )
         self.edCombo.insertItem( "Inconstant" )
         self.edCombo.setFixedHeight( self.edCombo.sizeHint().height() )
         self.vbox.addWidget( self.edCombo )
         self.connect( self.edCombo, SIGNAL("activated(const QString &)"), self.edComboBoxItemActivated)
         QToolTip.add( self.edCombo, "editable combo box" )

         self.edCombo.setAutoCompletion( TRUE )
         
         self.vbox.addStretch( 1 )

         self.vbox = QVBoxLayout( 8 )
         self.grid.addLayout( self.vbox, 2, 2 )

         # Create a spin box
         self.spin = QSpinBox( 0, 10, 1, self, "spin" )
         self.spin.setSuffix( " mm" )
         self.spin.setSpecialValueText( "Auto" )
         self.spin.setMinimumSize( self.spin.sizeHint() )
         self.connect( self.spin, SIGNAL( "valueChanged(const QString &)" ), self.spinBoxValueChanged )
         QToolTip.add( self.spin, "spin box" )
         self.vbox.addWidget( self.spin )
    
         self.vbox.addStretch( 1 )

         # Create a multi line edit
         self.mle = QMultiLineEdit( self, "multiLineEdit" )

         self.grid.addMultiCellWidget( self.mle, 3, 3, 1, 2 )
         self.mle.setMinimumHeight( self.mle.fontMetrics().height() * 3 )
         self.mle.setText("This is a QMultiLineEdit widget,\n"
                          "useful for small multi-line\n"
                          "input fields.")
         QToolTip.add( self.mle, "multi line editor" )

         # Create a single line edit
         self.le = QLineEdit( self, "lineEdit" )
         self.grid.addMultiCellWidget( self.le, 4, 4, 1, 2 )
         self.le.setFixedHeight( self.le.sizeHint().height() )
         self.connect( self.le, SIGNAL("textChanged(const QString &)"), self.lineEditTextChanged )
         QToolTip.add( self.le, "single line editor" )

         # Create a horizontal line (sort of QFrame) above the message line
         self.separator = QFrame( self, "separatorLine" )
         self.separator.setFrameStyle( QFrame.HLine | QFrame.Sunken )
         self.separator.setFixedHeight( self.separator.sizeHint().height() )
         self.grid.addMultiCellWidget( self.separator, 5, 5, 0, 2 )
         QToolTip.add( self.separator, "tool tips on a separator! wow!" )

         self.grid.setRowStretch( 0, 0 )
         self.grid.setRowStretch( 1, 0 )
         self.grid.setRowStretch( 2, 0 )
         self.grid.setRowStretch( 3, 1 )
         self.grid.setRowStretch( 4, 1 )
         self.grid.setRowStretch( 5, 0 )

         self.grid.setColStretch( 0, 1 )
         self.grid.setColStretch( 1, 1 )
         self.grid.setColStretch( 2, 1 )

         # Create an label and a message in a plain widget
         # The message is updated when buttons are clicked etc.

         self.hbox = QHBoxLayout()
         self.topLayout.addLayout( self.hbox )
         self.msgLabel = QLabel( self, "msgLabel" )
         self.msgLabel.setText( "Message:" )
         self.msgLabel.setAlignment( Qt.AlignHCenter | Qt.AlignVCenter )
         self.msgLabel.setFixedSize( self.msgLabel.sizeHint() )
         self.hbox.addWidget( self.msgLabel )
         QToolTip.add( self.msgLabel, "label 1" )

         self.msg = QLabel( self, "message" )
         self.msg.setFrameStyle( QFrame.Panel | QFrame.Sunken )
         self.msg.setAlignment( Qt.AlignCenter )
         self.msg.setFont( QFont( "times", 12, QFont.Bold ) )
         self.msg.setText( "Message" )
         self.msg.setFixedHeight( self.msg.sizeHint().height() )
         self.msg.setText( "" )
         self.hbox.addWidget( self.msg, 5 )
         QToolTip.add( self.msg, "label 2" )

         self.topLayout.activate()

    def setStatus(self, text):
        self.msg.setText( text )

    def movieUpdate( self, r ):
        # Uncomment this to test animated icons on your window manager
        self.setIcon( self.movie.framePixmap() )
        
    def movieStatus( self, s ):
        if s == QMovie.SourceEmpty or s == QMovie.UnrecognizedFormat:
            pm = QPixmap('tt-logo.png')
            self.movielabel.setPixmap(pm)
            self.movielabel.setFixedSize(pm.size())
        else:
            if ( self.movielabel.movie() ):      # for flicker-free animation:
                self.movielabel.setBackgroundMode( QWidget.NoBackground )
   
    def button1Clicked( self ):
        self.msg.setText( "The first push button was clicked" )

    def checkBoxClicked( self, id ):
        txt = QString()
        txt = "Check box %s clicked : " % str(id)
        chk = ["-","-","-"]
        if self.cb[0].isChecked():
            chk[0] = "r"
        if self.cb[1].isChecked():
            chk[1] = "w"
        if self.cb[2].isChecked():
            chk[2] = "x"
        txt = txt + str(chk[0]+chk[1]+chk[2])
        self.msg.setText( txt )
        
    def edComboBoxItemActivated( self, text):
        txt = QString()
        txt = "Editable Combo Box set to %s" % text
        self.msg.setText( txt )

    def radioButtonClicked( self, id ):
        txt = QString()
        txt = "Radio button #%d clicked" % id
        self.msg.setText( txt )

    def listBoxItemSelected( self, index ):
        txt = QString()
        txt = "List box item %d selected" % index
        self.msg.setText( txt )

    def sliderValueChanged( self, value ):
        txt = QString()
        txt = "Movie set to %d%% of normal speed" % value
        self.msg.setText( txt )
        self.movie.setSpeed( value )

    def comboBoxItemActivated( self, index ):
        txt = QString()
        txt = "Comboxo box item %d activated" % index
        self.msg.setText( txt )
        if index == 0:
            QApplication.setWinStyleHighlightColor( Qt.darkBlue )
	elif index == 1:
            QApplication.setWinStyleHighlightColor( Qt.darkRed )
	elif index == 2:
            QApplication.setWinStyleHighlightColor( Qt.darkGreen )
	elif index == 3:
            QApplication.setWinStyleHighlightColor( Qt.blue )
	elif index == 4:
            QApplication.setWinStyleHighlightColor( Qt.red )

    def lineEditTextChanged( self, newText ):
        self.msg.setText("Line edit text: " + unicode(newText))

    def spinBoxValueChanged( self, valueText ):
        self.msg.setText("Spin box value: " + unicode(valueText))

    # All application events are passed throught this event filter.
    # We're using it to display some information about a clicked
    # widget (right mouse button + CTRL).
    #def eventFilter( self, event ):
    #    identify_now = TRUE
    #    if event.type() == Event_MouseButtonPress and identify_now:
    #        e = QMouseEvent( event )
    #        if (e.button() == Qt.RightButton) and (e.state() & Qt.ControlButton) != 0:
    #            txt = QString( "The clicked widget is a\n" )
    #            txt = txt + QObect.className()
    #            txt = txt + "\nThe widget's name is\n"
    #            if QObject.name():
    #                txt = txt + QObject.name()
    #            else:
    #                txt = txt + "<no name>"
    #            identify_now = FALSE		# don't do it in message box
    #            QMessageBox.message( "Identify Widget", txt, 0, QObject )
    #            identify_now = TRUE;		# allow it again
    #    return FALSE				# don't eat event

################################################################################################

#QApplication.setColourSpec( QApplication.CustomColor )
a = QApplication( sys.argv )

w = WidgetView()
a.setMainWidget( w )
w.show()
a.exec_loop()
