/*
 *  Project   : tin - a Usenet reader
 *  Module    : xface.c
 *  Author    : Joshua Crawford & Drazen Kacar
 *  Created   : 2003-04-27
 *  Updated   : 2003-06-27
 *  Notes     :
 *
 * Copyright (c) Joshua Crawford <mortarn@softhome.net> & Drazen Kacar <dave@willfork.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * TODO: - add configure check option for use_slrnface
 *         (see comments include/autoconf.hin)
 *       - document the used vars/files/dir in the mapage
 *       - move strings to lang.c
 */

#ifndef TIN_H
#	include "tin.h"
#endif /* !TIN_H */

#ifdef XFACE_ABLE

static int slrnface_fd = -1;


void
slrnface_start(
	void)
{
	char *fifo;
	char *ptr;
	int pathlen, status;
	pid_t pid, pidst;
	struct utsname u;

	if (!tinrc.use_slrnface)
		return;

#ifdef HAVE_IS_XTERM
	if (!is_xterm())
		/* error_message(_("Can't run slrnface: Not running in a xterm.")); */
		return;
#endif /* HAVE_IS_XTERM */

	/*
	 * $DISPLAY holds the (default) display name
	 */
	if (!getenv("DISPLAY")) {
		/* error_message(_("Can't run slrnface: Environment variable %s not found."), "DISPLAY"); */
		return;
	}

	/*
	 * $WINDOWID holds the X window id number of the xterm window
	 */
	if (!getenv("WINDOWID")) {
		/* error_message(_("Can't run slrnface: Environment variable %s not found."), "WINDOWID"); */
		return;
	}

	uname(&u);
	if (!(ptr = getenv("HOME"))) { /* TODO: use tin global 'homedir' instead? or even rcdir? */
		error_message(_("Can't run slrnface: Environment variable %s not found."), "HOME");
		return;
	}
	pathlen = strlen(ptr) + strlen("/.slrnfaces/") + strlen(u.nodename) + 30;
	fifo = my_malloc(pathlen);
	sprintf(fifo, "%s/.slrnfaces", ptr);
	if (mkdir(fifo, 0700)) {	/* TODO: use my_mkdir() */
		if (errno != EEXIST) {
			perror_message(_("Can't run slrnface: failed to create %s"), fifo);
			free(fifo);
			return;
		}
	} else {
		FILE *fp;

		/* We'll abuse fifo filename memory here. It's long enough. */
		sprintf(fifo, "%s/.slrnfaces/README", ptr);
		if ((fp = fopen(fifo, "w")) != NULL) {
	      fputs(_("This directory is used to create named pipes for communication between\n"
"slrnface and its parent process. It should normally be empty because\n"
"the pipe is deleted right after it has been opened by both processes.\n\n"
"File names generated by slrnface have the form \"hostname.pid\". It is\n"
"probably an error if they linger here longer than a fraction of a second.\n\n"
"However, if the directory is mounted from an NFS server, you might see\n"
"special files created by your NFS server while slrnface is running.\n"
"Do not try to remove them.\n"), fp);
			fclose(fp);
		}
	}

	snprintf(fifo, pathlen, "%s/.slrnfaces/%s.%ld", ptr, u.nodename, (long) getpid());
	if (!(status = strlen(fifo))) {
		error_message(_("Can't run slrnface: couldn't construct fifo name."));
		unlink(fifo);
		free(fifo);
		return;
	}

	unlink(fifo);
	if (mkfifo(fifo, 0600) < 0) {
		perror_message(_("Can't run slrnface: failed to create %s"), fifo);
		unlink(fifo);
		free(fifo);
		return;
	}

	switch ((pid = fork())) {
		case -1:
			break;

		case 0:
			/*
			 * TODO: allow positioning, coloring, ...
			 *       execlp("slrnface", "slrnface",
			 *              "-xOffsetChar", tinrc.xfacex,
			 *              "-yOffsetChar", tinrc.xfacey,
			 *              "-ink", tinrc.xfacefg,
			 *              "-paper", tinrc.xfacebg,
			 *              fifo, (char *) 0);
			 */
			execlp("slrnface", "slrnface", fifo, (char *) 0);
			/* This is child, exit on error. */
			giveup();
			/* NOTREACHED */
			break;

		default:
			do {
				pidst = waitpid(pid, &status, 0);
			} while (pidst == -1 && errno == EINTR);
			if (!WIFEXITED(status))
				error_message(_("Slrnface abnormally exited, code %d."), status);
			else {
				const char *message;

				switch (WEXITSTATUS(status)) {
					case 0:	/* All fine, open the pipe */
						slrnface_fd = open(fifo, O_WRONLY, 0600);
						write(slrnface_fd, "start\n", strlen("start\n"));
						message = NULL;
						break;

					/* TODO: warp into _()? */
					case 1:
						message = "couldn't connect to display";
						break;

					case 2:
						message = "WINDOWID not found in environment";
						break;

					case 3:
						message = "couldn't find controlling terminal";
						break;

					case 4:
						message = "terminal doesn't export width and height";
						break;

					case 5:
						message = "can't open FIFO";
						break;

					case 6:
						message = "fork() failed";
						break;

					case 10:
						message = "executable not found";
						break;

					default:
						message = "unknown error";
				}
				if (message)
					error_message(_("Slrnface failed: %s."), message);
			}
	}
	unlink(fifo);
	free(fifo);
}


void
slrnface_stop(
	void)
{
	if (slrnface_fd >= 0)
		close(slrnface_fd);

	slrnface_fd = -1;
	/* FIFO has been unlinked in the startup function. */
}


void
slrnface_display_xface(
	char *face)
{
	if (slrnface_fd < 0)
		return;

	if (!face || !*face)
		write(slrnface_fd, "clear\n", strlen("clear\n"));
	else {
		char buf[2000];	/* slrnface will ignore X-Faces larger than approx. 2000 chars. */

		snprintf(buf, sizeof(buf), "xface %s\n", face);
		write(slrnface_fd, buf, strlen(buf));
	}
}


void
slrnface_clear_xface(
	void)
{
	if (slrnface_fd < 0)
		return;

	write(slrnface_fd, "clear\n", strlen("clear\n"));
}


void
slrnface_suppress_xface(
	void)
{
	if (slrnface_fd < 0)
		return;

	write(slrnface_fd, "suppress\n", strlen("suppress\n"));
}


void
slrnface_show_xface(
	void)
{
	if (slrnface_fd < 0)
		return;

	write(slrnface_fd, "show\n", strlen("show\n"));
}

#else
static void no_xface(void);        /* proto-type */
static void
no_xface(	/* ANSI C requires non-empty source file */
	void)
{
}
#endif /* XFACE_ABLE */
