{***************************************************************}
{ XMLInspector - a component for automatic generation of dialog }
{ forms based on XML                                            }
{                                                               }
{    Copyright (c) 2001-2004 Alexander Sviridenkov              }
{    Contact: asv@devrace.com                                   }
{                                                               }
{    XML parser. DOM/XPath/XSD suport                           }
{ ------------------------------------------------------------- }
{    XMLInspector home page : http://www.xmlinspector.com/      }
{    Support page :  http://www.devrace.com/en/support/         }
{ ------------------------------------------------------------- }
{                                                               }
{  Please see the file License.txt for full license information }
{                                                               }
{***************************************************************}
unit xmls; //version 1.20  11.06.2003


interface

uses Classes, SysUtils, strs;

{$I xmlinspector.inc}

type
  TXMLNodeType=(ntNode, ntAttr, ntText, ntCData, ntComment);
  TXMLNode=class;

  TXMLNodeList=class(TList)
  private
   function GetNode(index: integer): TXMLNode;
  public
   Parent: TXMLNode;
   SkipCase: boolean;
   constructor Create(AParent:TXMLNode);
   destructor  Destroy; override;
   procedure Sort(Compare: TListSortCompare; Asc: Boolean=true);
   procedure ClearNodes;
   function  NodebyName(const NodeName: string): TXMLNode;
   function  NodebyAttrs(const AttrNames, AttrValues: array of string): TXMLNode;
   function  NodebyNameAttr(const NodeName, AttrName, AttrValue: string): TXMLNode;
   function  AddNode(const NodeName: string; const NodeValue: string=''; ANodeType: TXMLNodeType=ntNode): TXMLNode;
   function  AddSetValue(const NodeName, NodeValue: string; ANodeType: TXMLNodeType): TXMLNode;
   function  GetCreateNode(const NodeName: string): TXMLNode;
   procedure DeleteNode(Node: TXMLNode);
   procedure DeleteNodes(const NodeName: string; Other: boolean);
   property  Nodes[index: integer]: TXMLNode read GetNode; default;
  end;

  CXMLNode=class of TXMLNode;
  TXMLNode=class
   private
    fValue: string;
    function  GetValue: string;
    procedure SetValue(const s: string);
    function  GetAttr(const AttrName: string): string;
    function  GetAttrI(const AttrName: string): integer;
    procedure SetAttr(const AttrName, AttrValue: string);
    procedure SetAttrI(const AttrName: string; AttrValue: Integer);
    function  GetNode(const NodeName: string): string;
    procedure SetNode(const NodeName, NodeValue: string);
    function  GetNodebyIndex(Index: integer): TXMLNode;
    function  GetCount:integer;
    function  RecursiveFind(Node: TXMLNode; const NodeName: string; SetValue: boolean=false;
      const AValue: string=''; CreateNode: boolean=true): TXMLNode;
   public
    Parent: TXMLNode;
    NodeType: TXMLNodeType;
    Name: string;
    Attributes, Nodes, Headers: TXMLNodeList;
    Tag: integer;
    Data: pointer;
    HTMLMode, SkipCase: boolean;
    constructor Create(const ACode: string; AParent: TXMLNode=nil; ASkipCase: boolean=false);
    constructor CreateDoc(const ACode: string);
    {$IFNDEF BCB}
    constructor CreateHTML(const ACode: string; AParent: TXMLNode; ASkipCase: boolean=false);
    {$ENDIF}
    constructor CreateFromFile(const FileName: string; AParent: TXMLNode=nil); virtual;
    constructor CreateFromName(const AName, AValue: string; ANodeType: TXMLNodeType; AParent: TXMLNode);
    procedure   ParseXML(ACode: pchar; var sPos: integer; L: integer);
    procedure   Validate(Schema: TXMLNode);
    destructor  Destroy; override;
    function    Code: string;
    function    HeaderCode: string;
    procedure   FastCode(s: TFastString);
    function    FormattedCode(const Offset: string): string;
    function    UnionNodeValues(const NodeName, AttrName, Divider: string): string;
    procedure   NodesToStrings(const NodeName, AttrName: string; List: TStrings);
    procedure   SortbyAttr(AttrName: string; Asc: Boolean=true);
    procedure   SortbyIntAttr(const AttrName: string; Asc: Boolean=true);
    procedure   SortbyNode(const NodeName: string; Asc: Boolean=true);
    procedure   SortbyIntNode(const NodeName: string; Asc: Boolean=true);
    procedure   SortbyFloatNode(const NodeName: string; Asc: Boolean=true);
    function    NodebyName(NodeName: string; SetValue: boolean=false;
      const AValue: string=''; CreateNode: boolean=true): TXMLNode;
    function    NodebyAttr(const AttrName, AttrValue: string): TXMLNode;
    function    FindNode(const NodeName, AttrName, AttrValue: string): TXMLNode;
    function    FindNodebyAttr(Node: TXMLNode; const AttrName, AttrValue: string): TXMLNode;
    procedure   SetLinks(const LinkNode, LinkAttr: string; Linked: TXMLNode; LinkedAttr: string);
    function    NextSibling: TXMLNode;
    function    NextNode: TXMLNode;
    function    FullName(const NodeDivider, AttrDivider: string): string;
    function    XPath: string;
    procedure   DeleteNode(Index: integer);
    procedure   ReplaceNode(OldNode, NewNode: TXMLNode);
    function    Root: TXMLNode;
    procedure   MoveChildsTo(NewParent: TXMLNode);
    procedure   Assign(Source: TXMLNode);
    function    NameSpace: string;
    property    Value: string read GetValue write SetValue;
    property    Attr[const AttrName: string]: string read GetAttr write SetAttr;
    property    AttrI[const AttrName: string]: integer read GetAttrI write SetAttrI;
    property    Node[const NodeName: string]: string read GetNode write SetNode;
    property    ANodes[index: integer]: TXMLNode read GetNodebyIndex; default;
    property    Count: integer read GetCount;
  end;

  TXSDNode=class(TXMLNode)
  public
   function AddElement(const EName, EType: string; Description: string=''; Nillable: boolean=true): TXSDNode;
  end;

  TXMLDictionary=class
  private
   FList: TFastList;
   function GetCount: Integer;
  public
   constructor Create; virtual;
   destructor  Destroy; override;
   function    Items(Index: integer): TXMLNode; virtual;
   function    Ids(Index: integer): string; virtual;
   function    Names(Index: Integer): string; virtual;
   function    FindItem(const id: string): TXMLNode; virtual;
   function    IndexOf(const id: string): Integer; virtual;
   function    AddItem(const id: string; Item: TXMLNode): integer; virtual;
   procedure   UpdateItem(const id: string; Item: TXMLNode); virtual;
   procedure   DeleteItem(const id: string); virtual;
   property    Count: Integer read GetCount;
   property    List: TFastList read FList;
  end;

  function XMLEncode(const s:string):string;
  function XMLDecode(const s:string):string;
  function XMLEncodeAttr(const s:string):string;
  function XMLEncodeNode(const s:string):string;
  function CreateXMLTemplate(Schema: TXMLNode; Optional: boolean; Defaults: boolean=true): TXMLNode;
  function XSDDatetoDate(const s: string):TDateTime;
  function DatetoXSDDate(d: TDateTime): string;
  procedure ApplyXMLTemplate(Node, Template: TXMLNode; const Masks, Values: array of string);

implementation

type TXMLType=(xtTagEnd, xtSlash, xtCData, xtValue, xtStr, xtCondition, xtProcess);

const varchars: set of char=['A'..'Z','a'..'z',''..'',''..'','@','_','$','.',':','-','%','0'..'9','*'];

threadvar SortAttrName: string;

{--------XML Parsing--------------------}

function XSDDatetoDate(const s: string): TDateTime;
begin
 Result:=EncodeDate(strtoint(copy(s, 1, 4)), strtoint2(copy(s, 6, 2)), strtoint2(copy(s, 9, 2)));
end;

function DatetoXSDDate(d: TDateTime): string;
var yy, mm, dd: word;
begin
 DecodeDate(d, yy, mm, dd);
 Result:=inttostr(yy)+'-'+inttostr2(mm)+'-'+inttostr2(dd);
end;

function XMLEncode(const s:string): string;
begin
 Result:=Q_ReplaceStr(Q_ReplaceStr(Q_ReplaceStr(Q_ReplaceStr(s, '&', '&amp;'), '"', '&quot;'), '<', '&lt;'), '>', '&gt;');
end;

function XMLDecode(const s: string): string;
begin
 Result:=Q_ReplaceStr(Q_ReplaceStr(Q_ReplaceStr(Q_ReplaceStr(s, '&gt;', '>'), '&lt;', '<'), '&quot;', '"'), '&amp;', '&');
end;

function XMLEncodeNode(const s: string): string;
begin
 Result:=Q_ReplaceStr(Q_ReplaceStr(s, '&', '&amp;'), '<', '&lt;');
// Result:=Q_ReplaceStr(Q_ReplaceStr(Q_ReplaceStr(s,'&','&amp;'),'<','&lt;'), '>', '&gt;');
end;

function XMLDecodeNode(const s:string):string;
begin
 Result:=Q_ReplaceStr(Q_ReplaceStr(s, '&lt;', '<'), '&amp;', '&');
// Result:=Q_ReplaceStr(Q_ReplaceStr(Q_ReplaceStr(s,'&lt;','<'),'&amp;','&'), '&gt;', '>');
end;

function XMLEncodeAttr(const s: string): string;
begin
 Result:=Q_ReplaceStr(Q_ReplaceStr(s, '&', '&amp;'), '"', '&quot;');
end;

function XMLDecodeAttr(const s: string): string;
begin
 Result:=Q_ReplaceStr(Q_ReplaceStr(s, '&quot;', '"'), '&amp;', '&');
end;

function GetXMLToken(s: pchar; var TokenType: TXMLType; var sPos: integer; L: integer): string;
var n: integer; p, pl: pchar;
begin
 TokenType:=xtValue;
 if sPos>=L then begin Result:=''; exit end;
 case s[sPos] of
  '!': begin TokenType:=xtCondition; inc(sPos) end;
  '"': begin
        TokenType:=xtStr;
        inc(sPos);
        n:=varpos('"', s, spos, L);
        if n>0 then Result:=XMLDecodeAttr(varcopy(s, sPos, n-1, L)) else raise Exception.Create('Unterminated attribute');
        inc(sPos, n);
       end;
  '/':begin TokenType:=xtSlash; inc(sPos) end;
  '>':begin TokenType:=xtTagEnd; inc(sPos) end;
  '?': begin TokenType:=xtProcess; inc(sPos) end;
  else if s[sPos] in varchars then begin
   p:=@s[sPos+1]; pl:=@s[L];
   while (p<pl) and (p^ in varchars) do inc(p);
   Result:=varcopy(s,sPos,p-s-sPos,L);
   inc(sPos, p-s-sPos);
  end else raise Exception.Create(inttostr(sPos)+' :Illegal symbol in XML:'+s[spos]);
 end;
end;

function GetPCData(s: pchar; var sPos: integer; L: integer): string;
var n: integer;
begin
 n:=varpos('<', s, sPos, L);
 if n=0 then n:=L-sPos+2;
 Result:=XMLDecodeNode(varcopy(s, sPos, n-1, L));
 inc(sPos, n-1);
end;

procedure GetComment(s: pchar; var sPos: integer; var Value: string; L: integer);
var n: integer;
begin
 n:=varpos('-->', s, sPos, L);
 if n=0 then raise Exception.Create('Unclosed Comment in XML');
 Value:=varcopy(s, sPos, n-1, L);
 inc(sPos, n+2);
end;

procedure GetCData(s: pchar; var sPos: integer; var Value: string; L: integer);
var n: integer;
begin
 n:=varpos(']]>', s, sPos, L);
 if n=0 then n:=L-sPos+2;
 Value:=varcopy(s, sPos, n-1, L);
 inc(sPos, n+2);
end;

procedure pTrimLeft(s: pchar; var sPos: integer);
 asm
    mov ecx,[edx]
    cmp byte ptr [eax][ecx],32
    ja @end2
    push edx
@1: mov dl,[eax][ecx]
    cmp dl,32
    ja @end
    inc ecx
    jmp @1
@end:
    pop edx
    mov [edx],ecx
@end2:

{begin
 while (s[sPos]=' ') do inc(sPos)}
end;

 {-----NodeList-----------------------------}

constructor TXMLNodeList.Create(AParent: TXMLNode);
begin
 inherited Create;
 Parent:=AParent;
 SkipCase:=Parent.SkipCase;
end;

destructor TXMLNodeList.Destroy;
begin
 ClearNodes;
 inherited
end;

procedure TXMLNodeList.ClearNodes;
var i: integer;
begin
 for i:=0 to Count-1 do TXMLNode(Items[i]).Free;
 Clear;
end;

function TXMLNodeList.GetNode(index: integer): TXMLNode; begin Result:=TXMLNode(Items[index]) end;

function TXMLNodeList.NodebyName(const NodeName: string): TXMLNode;
var p, e: ^Pointer;
begin
 Result:=nil;
 if Count=0 then exit;
 p:=@List^[0];
 e:=@List^[Count-1];
 if not SkipCase then begin
  while Cardinal(p)<=Cardinal(e) do begin
   if Q_SameStr(TXMLNode(p^).Name, NodeName) then begin Result:=TXMLNode(p^); exit end;
   Inc(p);
  end;
 end else
  while Cardinal(p)<=Cardinal(e) do begin
   if AnsiSameText(TXMLNode(p^).Name, NodeName) then begin Result:=TXMLNode(p^); exit end;
   Inc(p);
  end;
end;

function TXMLNodeList.NodebyNameAttr(const NodeName, AttrName, AttrValue: string): TXMLNode;
var i: integer; N: TXMLNode; FL: PPointerList;
begin
 FL:=List;
 for i:=0 to Count-1 do begin
   if Q_SameStr(TXMLNode(FL^[i]).Name, NodeName) then begin
    N:=TXMLNode(FL^[i]).Attributes.NodebyName(AttrName);
    if ((N<>nil) and (Q_SameStr(N.fValue,AttrValue))) or ((AttrValue='') and (N=nil)) then begin Result:=TXMLNode(FL^[i]); exit end;
   end;
 end;
 Result:=nil
end;

function  TXMLNodeList.NodebyAttrs(const AttrNames, AttrValues: array of string): TXMLNode;
var i, k, n: integer; FL: PPointerList; yes: boolean;
begin
 FL:=List;
 n:=High(AttrNames);
 if n<>High(AttrValues) then raise Exception.Create('    ');
 for i:=0 to Count-1 do begin
  yes:=true;
  Result:=TXMLNode(FL^[i]);
  for k:=0 to n do if Result.Attr[AttrNames[k]]<>AttrValues[k] then begin yes:=false; break end;
  if yes then exit;
 end;
 Result:=nil
end;

{function TXMLNodeList.NodebyAttrs(const AttrNames, AttrValues: array of string): TXMLNode;
var i, k, n, a: integer; FL: PPointerList; yes: boolean; RA: TXMLNodeList;
begin
 FL:=List;
 n:=High(AttrNames);
 if n<>High(AttrValues) then raise Exception.Create('    ');
 for i:=0 to Count-1 do begin
  yes:=true;
  RA:=TXMLNode(FL^[i]).Attributes;
  for k:=0 to n do begin
   for a:=0 to RA.Count-1 do
    with RA.Nodes[a] do if (Value<>AttrValues[k]) and (Name=AttrNames[k]) then begin yes:=false; break end; //!!! ,    
   if not yes then break;
  end;
  if yes then begin Result:=TXMLNode(FL^[i]); exit end;
 end;
 Result:=nil
end;}

function TXMLNodeList.GetCreateNode(const NodeName: string): TXMLNode;
begin
 Result:=NodebyName(NodeName);
 if Result=nil then Result:=AddNode(NodeName);
end;

function TXMLNodeList.AddNode(const NodeName, NodeValue: string; ANodeType: TXMLNodeType): TXMLNode;
begin
 Result:=CXMLNode(Parent.ClassType).CreateFromName(NodeName, NodeValue, ANodeType, Parent);
 Add(Result);
end;

function TXMLNodeList.AddSetValue(const NodeName, NodeValue: string; ANodeType: TXMLNodeType): TXMLNode;
begin
 Result:=CXMLNode(Parent.ClassType).Create('', Parent);
 with Result do begin
  Name:=NodeName;
  NodeType:=ANodeType;
  if NodeType in [ntText,ntCData] then fValue:=NodeValue else
   if (NodeType=ntNode) and (NodeValue<>'') then Result.Nodes.AddNode('', NodeValue, ntText);
 end;
 Add(Result);
end;

procedure TXMLNodeList.DeleteNode(Node: TXMLNode);
begin
 Remove(Node);
 Node.Free;
end;

procedure TXMLNodeList.DeleteNodes(const NodeName: string; Other: boolean);
var i: integer; same: boolean;
begin
 for i:=Count-1 downto 0 do begin
  same:=Nodes[i].Name=NodeName;
  if (Other and not same) or (not Other and same) then DeleteNode(Nodes[i]);
 end;
end;

function SOrder(X: Integer; Asc: Boolean): Integer;
begin
 Result:=X;
 if not Asc then begin
  if X>0 then Result:=-1 else if X<0 then Result:=1
 end;
end;

procedure QuickSort(SortList: PPointerList; L, R: Integer;
  SCompare: TListSortCompare; Asc: Boolean=true);
var
 I, J: Integer; P, T: Pointer;
begin
 repeat
  I := L;
  J := R;
  P := SortList^[(L + R) shr 1];
  repeat
   while SOrder(SCompare(SortList^[I], P), Asc) < 0 do Inc(I);
   while SOrder(SCompare(SortList^[J], P), Asc) > 0 do Dec(J);
   if I <= J then begin
    if I<>J then begin
     T := SortList^[I];
     SortList^[I] := SortList^[J];
     SortList^[J] := T;
    end;
    Inc(I);
    Dec(J);
   end;
  until I > J;
  if L < J then QuickSort(SortList, L, J, SCompare, Asc);
  L := I;
 until I >= R;
end;

procedure TXMLNodeList.Sort(Compare: TListSortCompare; Asc: Boolean=true);
begin
 if (List<>nil) and (Count>0) then QuickSort(List, 0, Count-1, Compare, Asc)
end;


 {-----------XMLNode-------------------------------}

constructor TXMLNode.Create(const ACode: string; AParent: TXMLNode=nil; ASkipCase: boolean=false);
var n, L: integer;
begin
 Parent:=AParent;
 SkipCase:=ASkipCase;
 if Assigned(AParent) then begin
  HTMLMode:=AParent.HTMLMode;
  SkipCase:=AParent.SkipCase
 end;
 Nodes:=TXMLNodeList.Create(self);
 Attributes:=TXMLNodeList.Create(self);
 NodeType:=ntNode;
 n:=0;
 L:=Length(ACode);
 if L>0 then ParseXML(pointer(ACode),n,L);
end;

constructor TXMLNode.CreateDoc(const ACode: string); begin Create(ACode) end;

constructor TXMLNode.CreateFromFile(const FileName: string; AParent: TXMLNode=nil);
begin
 Create(FileToStr(FileName), AParent);
end;

{$IFNDEF BCB}
constructor TXMLNode.CreateHTML(const ACode: string; AParent: TXMLNode; ASkipCase: boolean=false);
begin
 HTMLMode:=true;
 Create(ACode, AParent, ASkipCase)
end;
{$ENDIF}

constructor TXMLNode.CreateFromName(const AName, AValue: string; ANodeType: TXMLNodeType; AParent: TXMLNode);
begin
 Name:=trim(AName);
 if (NodeType=ntAttr) and (Name='') then raise Exception.Create('Attribute/Node name cannot be empty');
 NodeType:=ANodeType;
 fValue:=AValue;
 Parent:=AParent;
 if (NodeType<>ntAttr) {and (NodeType<>ntText)} then begin
  if Assigned(AParent) then begin
   HTMLMode:=AParent.HTMLMode;
   SkipCase:=AParent.SkipCase;
  end;
  Nodes:=TXMLNodeList.Create(self);
  Attributes:=TXMLNodeList.Create(self);
 end;
end;

destructor TXMLNode.Destroy;
begin
 Attributes.Free;
 Nodes.Free;
 Headers.Free;
 inherited;
end;

function TXMLNode.GetCount: integer; begin if Assigned(Nodes) then Result:=Nodes.Count else Result:=0 end;

function TXMLNode.GetNodeByIndex(Index: integer): TXMLNode; begin Result:=TXMLNode(Nodes.Items[index]) end;

procedure TXMLNode.ParseXML(ACode: pchar; var sPos: integer; L: integer);
var s: string; xt: TXMLType; N: TXMLNode; Slash: boolean; k: integer;
begin
 repeat
  if ACode[sPos]<>'<' then raise Exception.Create('< expected');
  inc(sPos);
  Name:=GetXMLToken(ACode, xt, sPos, L);
  case xt of
  xtCondition: begin   //CDATA
    if varcopy(ACode, sPos, 7, L)='[CDATA[' then begin
     inc(sPos, 7);
     Name:='#cdata-section';
     GetCDATA(ACode, sPos, fValue, L);
     NodeType:=ntCData;
    end else if varcopy(ACode, sPos, 2, L)='--' then begin
     inc(sPos, 2);
     Name:='#comment';
     NodeType:=ntComment;
     GetComment(ACode, sPos, fValue, L);
     pTrimLeft(ACode, sPos);
     if Parent=nil then xt:=xtProcess;
    end
   end;
  xtProcess: begin   //?
    k:=varPos('?>', ACode, sPos, L);
    if k=0 then raise Exception.Create('Unclosed XML header');
    {XMLHeader:=varCopy(ACode,sPos,k+1,L);}
    if not Assigned(Headers) then Headers:=TXMLNodeList.Create(self);
    N:=Headers.AddNode(GetXMLToken(ACode, xt, sPos, L));
    while (sPos<=L) and (xt<>xtTagEnd) do begin
     pTrimLeft(ACode, sPos);
     s:=GetXMLToken(ACode, xt, sPos, L);
     case xt of
      xtValue: begin
         if ACode[sPos]<>'=' then raise Exception.Create('= expected');
         inc(sPos);
         N.Attributes.Add(CXMLNode(ClassType).CreateFromName(s, GetXMLToken(ACode, xt, sPos, L), ntAttr, self));
         if (xt<>xtStr) and ((not HTMLMode) and (xt=xtValue)) then raise Exception.Create('Attribute value expected');
        end;
     end;
    end;
    //inc(sPos,k+1);
    pTrimLeft(ACode, sPos);
    xt:=xtProcess;
    continue
   end;
  xtValue: begin  //Node
    Slash:=false;
    while (sPos<=L) and (xt<>xtTagEnd) do begin
     pTrimLeft(ACode, sPos);
     s:=GetXMLToken(ACode, xt, sPos, L);
     case xt of
      xtValue: begin
         pTrimLeft(ACode, sPos);
         if ACode[sPos]<>'=' then raise Exception.Create('= expected:'+copy(acode, spos-50, 100));
         inc(sPos);
         Attributes.Add(CXMLNode(ClassType).CreateFromName(s, GetXMLToken(ACode, xt, sPos, L),ntAttr, self));
         if (xt<>xtStr) and ((not HTMLMode) and (xt=xtValue)) then raise Exception.Create('Attribute value expected');
        end;
      xtSlash: Slash:=true;
     end;
    end;
    if xt<>xtTagEnd then raise Exception.Create('> expected');
    if (not Slash) then
     while sPos<=L do begin
      if acode[sPos]='<' then begin
        if acode[sPos+1]='/' then begin
          if varcopy(acode, sPos+2, length(name), L)<>name then
           raise Exception.Create('Tag order error. Tag: "'+name+'" code: '+copy(acode, spos-100, 200));
          inc(sPos, length(name)+3);
          break
        end else begin
          N:=CXMLNode(ClassType).Create('', self);
          N.ParseXML(ACode, sPos, L);
          Nodes.Add(N);
        end;
      end else begin //PCData
       s:=GetPCData(ACode, sPos, L);
       if trim(s)<>'' then Nodes.Add(CXMLNode(ClassType).CreateFromName('', s, ntText, self));
      end;
     end;
    end;
  else raise Exception.Create('Tag name expected: '+copy(ACode, sPos-20, 200));
  end;
 until xt<>xtProcess;
end;

 procedure TXMLNode.Validate(Schema:TXMLNode);
 var SN, ST: TXMLNode; var i, k, n: integer; v, dtype: string;

  function SubNodeCount(NodeName:string):integer;
  var i:integer;
  begin
   Result:=0;
   for i:=0 to Count-1 do if Nodes[i].Name=NodeName then inc(Result);
  end;

 begin
  SN:=Schema.NodeByAttr('name',Name);
  if SN<>nil then begin
   {--validate attributes--}
   for i:=0 to SN.Count-1 do if SN[i].Name='Attribute' then begin
    ST:=SN.NodebyAttr('name',SN[i].Attr['type']);
    if ST=nil then raise Exception.CreateFmt('Error in schema: unknown attribute type %s in element %s',[SN[i].Attr['type'],Name]);
    v:=Attr[ST.Attr['name']];
    if (ST.Attr['required']='yes') and (v='') then
     raise Exception.CreateFmt('Required attribute %s missed in element %s',[ST.Attr['name'],Name]);
    if v<>'' then begin
     dtype:=ST.Attr['dt:type'];
     if dtype='enumeration' then begin
      if pos(' '+v+' ',' '+ST.Attr['dt:values']+' ')=0 then
       raise Exception.CreateFmt('Wrong enumerated attribute %s value: "%s" in element %s',[ST.Attr['name'],v,Name]);
     end else if dtype='boolean' then begin
      if (v<>'0') and (v<>'1') then raise Exception.CreateFmt('Wrong boolean attribute %s value: "%s" in element %s',[ST.Attr['name'],v,Name]);
     end
    end;
   {--validate nodes--}
   end else if SN[i].Name='element' then begin
    k:=SubNodeCount(SN[i].Attr['type']);
    if (k<strtoint(SN[i].Attr['minOccurs'])) or ((SN[i].Attr['maxOccurs']<>'*') and (k>strtoint(SN[i].Attr['maxOccurs'])))
     then raise Exception.CreateFmt('Invalid SubElement %s count (%d) in element %s',[SN[i].Attr['type'],k,Name]);
   end else if SN[i].Name='group' then begin
    for n:=0 to SN[i].Count-1 do if SN[i][n].Name='element' then begin
     k:=SubNodeCount(SN[i][n].Attr['type']);
     if (k<strtoint(SN[i][n].Attr['minOccurs'])) or ((SN[i][n].Attr['maxOccurs']<>'*') and (k>strtoint(SN[i][n].Attr['maxOccurs'])))
      then raise Exception.CreateFmt('Invalid SubElement %s count (%d) in element %s',[SN[i][n].Attr['type'],k,Name]);
    end;
   end;
   {--validate subnodes--}
   for i:=0 to Count-1 do Nodes[i].Validate(Schema);
  end;
 end;

function TXMLNode.NodebyAttr(const AttrName, AttrValue: string): TXMLNode;
var i: integer; N: TXMLNode; FL: PPointerList;
begin
 if Assigned(Nodes) then begin
  FL:=Nodes.List;
  for i:=0 to Nodes.Count-1 do begin
    N:=TXMLNode(FL^[i]).Attributes.NodebyName(AttrName);
    if ((N<>nil) and (Q_SameStr(N.fValue, AttrValue))) or ((AttrValue='') and (N=nil))
     then begin Result:=TXMLNode(FL^[i]); exit end;
  end;
 end;
 Result:=nil
end;

function TXMLNode.GetAttr(const AttrName: string): string;
var N: TXMLNode;
begin
 if Assigned(Attributes) then begin
  N:=Attributes.NodebyName(AttrName);
  if Assigned(N) then Result:=N.Value else Result:='';
 end else Result:=''
end;

function TXMLNode.GetAttrI(const AttrName: string): integer;
begin
 Result:=StrtoInt(GetAttr(AttrName))
end;

procedure TXMLNode.SetAttr(const AttrName, AttrValue: string);
var N: TXMLNode;
begin
 if Assigned(Attributes) then begin
  N:=Attributes.NodebyName(AttrName);
  if length(AttrValue)=0 then begin
   if Assigned(N) then Attributes.DeleteNode(N);
  end else
  if Assigned(N) then N.fValue:=AttrValue
   else Attributes.Add(CXMLNode(ClassType).CreateFromName(AttrName, AttrValue, ntAttr, self))
 end
end;

procedure TXMLNode.SetAttrI(const AttrName: string; AttrValue: Integer);
begin
 Attr[AttrName]:=inttostr(AttrValue);
end;

function TXMLNode.GetNode(const NodeName: string): string;
var N: TXMLNode;
begin
 N:=NodebyName(NodeName);
 if Assigned(N) then Result:=N.Value else Result:='';
end;

procedure TXMLNode.SetNode(const NodeName, NodeValue: string);
begin
 NodebyName(NodeName, true, NodeValue);
end;

function TXMLNode.GetValue:string;
var i: integer;
begin
 case NodeType of
  ntNode: begin
           Result:='';
           for i:=0 to Nodes.Count-1 do
            if Nodes[i].NodeType in [ntText, ntCData] then begin Result:=Nodes[i].Value; break end;
          end;
  ntAttr, ntText, ntCData, ntComment: Result:=fValue;
 else Result:='' end;
end;

procedure TXMLNode.SetValue(const s: string);
var i: integer;
begin
 case NodeType of
  ntNode: begin
           for i:=0 to Nodes.Count-1 do if Nodes[i].NodeType in [ntText,ntCData] then begin Nodes[i].Value:=s; exit end;
           Nodes.AddNode('', s, ntText);
          end;
  ntAttr,ntText,ntCData: fValue:=s;
 end;
end;

function CompareAttr(N1, N2: pointer): integer;
var s, s1: string;
begin
 Result:=0;
 s:=SortAttrName;
 while (s<>'') and (Result=0) do begin
  s1:=Q_StrTok(s, ',;');
  Result:=Q_CompText(TXMLNode(N1).Attr[s1], TXMLNode(N2).Attr[s1])
 end;
end;

function CompareIntAttr(N1, N2: pointer): integer;
var s, s1: string;
begin
 Result:=0;
 s:=SortAttrName;
 while (s<>'') and (Result=0) do begin
  s1:=Q_StrTok(s, ',;');
  Result:=StrtoInt(TXMLNode(N1).Attr[s1])-StrtoInt(TXMLNode(N2).Attr[s1])
 end;
end;

function CompareNodes(N1, N2: pointer): integer;
var s, s1: string;
begin
 Result:=0;
 s:=SortAttrName;
 while (s<>'') and (Result=0) do begin
  s1:=Q_StrTok(s, ',;');
  Result:=Q_CompText(TXMLNode(N1).Node[s1], TXMLNode(N2).Node[s1])
 end;
end;

function CompareIntNodes(N1, N2: pointer): integer;
var s, s1: string;
begin
 Result:=0;
 s:=SortAttrName;
 while (s<>'') and (Result=0) do begin
  s1:=Q_StrTok(s, ',;');
  Result:=StrtoInt(TXMLNode(N1).Node[s1])-StrtoInt(TXMLNode(N2).Node[s1])
 end;
end;

function CompareFloatNodes(N1, N2: pointer): integer;
var s, s1: string; d1, d2: Extended;
begin
 Result:=0;
 s:=SortAttrName;
 while (s<>'') and (Result=0) do begin
  s1:=Q_StrTok(s, ',;');
  d1:=iStrtoFloat(TXMLNode(N1).Node[s1]);
  d2:=iStrtoFloat(TXMLNode(N2).Node[s1]);
  if d1>d2 then Result:=1 else if d1<d2 then Result:=-1;
 end;
end;

procedure TXMLNode.SortbyAttr(AttrName: string; Asc: Boolean=true);
begin
 SortAttrName:=AttrName;
 Nodes.Sort(CompareAttr, Asc);
end;

procedure TXMLNode.SortbyIntAttr(const AttrName: string; Asc: Boolean=true);
begin
 SortAttrName:=AttrName;
 Nodes.Sort(CompareIntAttr, Asc);
end;

procedure TXMLNode.SortbyNode(const NodeName:string; Asc: Boolean=true);
begin
 SortAttrName:=NodeName;
 Nodes.Sort(CompareNodes, Asc);
end;

procedure TXMLNode.SortbyIntNode(const NodeName: string; Asc: Boolean=true);
begin
 SortAttrName:=NodeName;
 Nodes.Sort(CompareIntNodes, Asc);
end;

procedure TXMLNode.SortbyFloatNode(const NodeName: string; Asc: Boolean=true);
begin
 SortAttrName:=NodeName;
 Nodes.Sort(CompareFloatNodes, Asc);
end;

function TXMLNode.HeaderCode: string;
var i: integer;
begin
 Result:='<?'+Name;
 for i:=0 to Attributes.Count-1 do Result:=Result+' '+Attributes[i].Code;
 Result:=Result+' ?>';
end;

function TXMLNode.Code: string;
var i: integer;
begin
 case NodeType of
  ntNode: begin Result:='<'+Name;
           for i:=0 to Attributes.Count-1 do Result:=Result+' '+Attributes[i].Code;
           if Nodes.Count=0 then Result:=Result+'/>' else begin
            Result:=Result+'>';
            for i:=0 to Nodes.Count-1 do Result:=Result+Nodes[i].Code;
            Result:=Result+'</'+Name+'>';
           end;
          end;
  ntAttr: Result:=Name+'="'+XMLEncodeAttr(fValue)+'"';
  ntText:  Result:=XMLEncodeNode(Value);
  ntCData: Result:='<![CDATA['+Value+']]>';
  ntComment: Result:='<!--'+Value+'-->';
 else Result:='' end;
 if Assigned(Headers) then
  for i:=Headers.Count-1 downto 0 do Result:=Headers[i].HeaderCode+Result;
end;

procedure TXMLNode.FastCode(s: TFastString);
var i: integer;
begin
 case NodeType of
  ntNode: begin s.Add('<'+Name);
           for i:=0 to Attributes.Count-1 do s.Add(' '+Attributes[i].Code);
           if Nodes.Count=0 then s.Add('/>') else begin
            s.Add('>');
            for i:=0 to Nodes.Count-1 do Nodes[i].FastCode(s);
            s.Add('</'+Name+'>');
           end;
          end;
  ntAttr: s.Add(Name+'="'+XMLEncodeAttr(fValue)+'"');
  ntText: s.Add(XMLEncodeNode(Value));
  ntCData: s.Add('<![CDATA['+Value+']]>');
  ntComment: s.Add('<!--'+Value+'-->');
 end;
end;

function TXMLNode.FormattedCode(const offset: string): string;
var i: integer; hasnodes: boolean;
begin
 case NodeType of
  ntNode: begin Result:='<'+Name;
           for i:=0 to Attributes.Count-1 do Result:=Result+' '+Attributes[i].Code;
           if (Nodes.Count=0) then Result:=Result+'/>'+#13#10 else begin
            hasnodes:=false;
            if Nodes[0].NodeType<>ntText then Result:=Result+'>'#13#10 else Result:=Result+'>';
            for i:=0 to Nodes.Count-1 do
             if Nodes[i].NodeType=ntNode then begin
              hasnodes:=true;
              Result:=Result+offset+Nodes[i].FormattedCode(offset+' ')
             end else Result:=Result+Nodes[i].Code;
            if hasnodes then Result:=Result+copy(offset, 1, Length(offset)-1)+'</'+Name+'>'#13#10
             else Result:=Result+'</'+Name+'>'#13#10;
           end;
          end;
 else Result:=Code end;
end;

function TXMLNode.UnionNodeValues(const NodeName, AttrName, Divider: string): string;
var i: integer; s: string;
begin
 Result:='';
 for i:=0 to Nodes.Count-1 do
  if (NodeName='') or (Nodes[i].Name=NodeName) then begin
   if AttrName<>'' then s:=Nodes[i].Attr[AttrName] else s:=Nodes[i].Value;
   if s<>'' then Result:=Result+s+Divider;
  end;
 Result:=trim(copy(Result, 1, length(Result)-length(Divider)));
end;

procedure TXMLNode.NodesToStrings(const NodeName, AttrName: string; List: TStrings);
var i: integer;
begin
 List.Clear;
 for i:=0 to Nodes.Count-1 do
  if Nodes[i].Name=NodeName then List.Add(Nodes[i].Attr[AttrName]);
end;

function TXMLNode.FindNode(const NodeName, AttrName, AttrValue: string): TXMLNode;
var i:integer;
begin
 if Assigned(Nodes) then begin
  Result:=Nodes.NodebyNameAttr(NodeName, AttrName, AttrValue);
  if Result=nil then
   for i:=0 to Count-1 do begin
    Result:=Nodes[i].FindNode(NodeName, AttrName, AttrValue);
    if Assigned(Result) then exit;
   end;
 end else Result:=nil;
end;

function TXMLNode.NextSibling: TXMLNode;
var n: integer; P, XN: TXMLNode;
begin
 if Count>0 then Result:=Nodes[0] else begin
  Result:=nil;
  XN:=self;
  repeat
   P:=XN.Parent;
   if P=nil then begin Result:=nil; exit end;
   n:=P.Nodes.Indexof(XN);
   if n=-1 then begin Result:=nil; exit end;
   if n<P.Count-1 then begin Result:=P[n+1]; break end
   else XN:=P
  until XN=nil;
 end
end;

function TXMLNode.NextNode: TXMLNode;
var n: integer;
begin
 Result:=nil;
 if Parent<>nil then begin
  n:=Parent.Nodes.Indexof(Self);
  if n<Parent.Nodes.Count-1 then Result:=Parent.Nodes[n+1];
 end;
end;

procedure TXMLNode.SetLinks(const LinkNode, LinkAttr: string; Linked: TXMLNode; LinkedAttr: string);
var i: integer; s: string;
begin
 for i:=0 to Count-1 do begin
  if Nodes[i].Name=LinkNode then begin
   s:=Nodes[i].Attr[LinkAttr];
   if s<>'' then Nodes[i].Data:=Linked.NodebyAttr(LinkedAttr, s);
  end;
  if Nodes[i].Count>0 then Nodes[i].SetLinks(LinkNode, LinkAttr, Linked, LinkedAttr);
 end;
end;

function TXMLNode.RecursiveFind(Node: TXMLNode; const NodeName: string; SetValue: boolean=false;
  const AValue: string=''; CreateNode: boolean=true): TXMLNode;
var i: integer;
begin
 Result:=Node.NodebyName(NodeName, SetValue, AValue, CreateNode);
 if Assigned(Result) and (not SetValue) then exit;
 for i:=0 to Node.Count-1 do begin
  Result:=RecursiveFind(Node[i], NodeName, SetValue, AValue, CreateNode);
  if Assigned(Result) and (not SetValue) then exit;
 end;
end;

function TXMLNode.FindNodebyAttr(Node: TXMLNode; const AttrName, AttrValue: string): TXMLNode;
var i: integer;
begin
 Result:=nil;
 for i:=0 to Node.Count-1 do begin
  if Node[i].Attr[AttrName]=AttrValue then Result:=Node[i]
   else Result:=FindNodebyAttr(Node[i], AttrName, AttrValue);
  if Assigned(Result) then exit;
 end;
end;

function TXMLNode.NodebyName(NodeName: string; SetValue: boolean=false;
  const AValue: string=''; CreateNode: boolean=true): TXMLNode;
var n, k, ind, rind: integer; s, nod, query, qnode, qvalue: string; F: TXMLNode;
begin
 Result:=Self;
 while (length(NodeName)>0) and (Result<>nil) do begin
  if StartsWith(NodeName, '//') then begin
   Result:=RecursiveFind(Result.Root, copy(NodeName, 3, MaxInt), SetValue, AValue, false);
   exit
  end else if StartsWith(NodeName, './/') then begin
   while Result.Parent<>nil do Result:=Result.Parent;
   Result:=RecursiveFind(Result, copy(NodeName, 3, MaxInt), SetValue, AValue, false);
   exit
  end else begin
   n:=pos('/', NodeName);
   if n>0 then begin s:=copy(NodeName, 1, n-1); delete(NodeName, 1, n) end
    else begin s:=NodeName; NodeName:='' end;
   if s='' then continue;
   if s[1]='@' then begin
    if not SetValue then Result:=Result.Attributes.NodebyName(copy(s, 2, MaxInt))
     else Result.Attr[copy(s, 2, MaxInt)]:=AValue;
    exit
   end;
   if s='.' then continue;
   if s='..' then begin Result:=Result.Parent; continue end;
   n:=pos('[', s);
   if n>0 then begin
    nod:=copy(s, 1, n-1);
    query:=trim(copy(s, n+1, pos(']', s)-n-1));
    if query<>'' then begin
     F:=nil;
     if query='last()' then begin
      for k:=Result.Count-1 downto 0 do
       if ((nod='') or (nod=Result[k].Name)) then begin F:=Result[k]; break end;
      Result:=F;
     end else if query[1] in ['0'..'9'] then begin
      ind:=strtoint(query);
      rind:=0;
      for k:=0 to Result.Count-1 do
       if ((nod='') or (nod=Result[k].Name)) then begin
        if ind=rind then begin F:=Result[k]; break end else inc(rind);
       end;
      Result:=F;
     end else if pos('=', query)>0 then begin
      qnode:=CopyToPos(query, '=');
      qvalue:=copy(query, pos('=', query)+1, maxint);
      if StartsWith(qvalue, '"') then qvalue:=copy(qvalue,2,length(qvalue)-2);
      for k:=0 to Result.Count-1 do
       if ((nod='') or (nod='*') or (nod=Result[k].Name)) then begin
        if Result[k].Node[qnode]=qvalue then begin F:=Result[k]; break end;
       end;
      Result:=F;
     end
    end
   end else begin
    F:=Result.Nodes.NodebyName(s);
    if SetValue and (F=nil) and CreateNode then Result:=Result.Nodes.AddNode(s) else Result:=F;
   end;
  end;
 end;
 if Assigned(Result) and SetValue then Result.Value:=AValue;
end;

function TXMLNode.FullName(const NodeDivider, AttrDivider: string): string;
var N: TXMLNode;
begin
 Result:=Name;
 N:=Parent;
 while N<>nil do begin
  Result:=N.Name+NodeDivider+Result;
  N:=N.Parent
 end;
end;

function CreateXMLTemplate(Schema: TXMLNode; Optional: boolean; Defaults: boolean=true): TXMLNode;
var Element, Res: TXMLNode; NS: string;

procedure IndexNode(Node, SchemaNode: TXMLNode; level: integer);
var XN, SI, Schema1: TXMLNode; i: integer; element, datatype: string; processed: boolean;
begin
 if level>10 then exit;
 processed:=false;
 for i:=0 to SchemaNode.Count-1 do begin
  if processed and (SchemaNode.name=NS+'choice') then break;
  SI:=SchemaNode[i];
  if SI.Name=NS+'attribute' then begin
   if (SI.Attr['default']<>'') and Defaults then Node.Attr[SI.Attr['name']]:=SI.Attr['default'];
  end else
  if SI.Name=NS+'sequence' then begin
   if (not Optional) and (StrIn(SI.Attr['minOccurs'], ['0'])) then continue;
   IndexNode(Node, SI, level+1); processed:=true
  end else
  if SI.Name=NS+'element' then begin
    processed:=true;
    if (not Optional) and (StrIn(SI.Attr['minOccurs'], ['0'])) then continue;
    if SI.Attr['ref']<>'' then begin
      XN:=Node.Nodes.AddNode(SI.Attr['ref']);
      Schema1:=Schema.Nodes.NodebyNameAttr(NS+'element', 'name', SI.Attr['ref']);
      if Assigned(Schema1) then IndexNode(XN, Schema1, level+1);
      continue
    end;
    datatype:=SI.Attr['type'];
    element:=SI.Attr['name'];
    if (datatype<>'') and StartsWith(datatype, NS) then begin
     XN:=Node.Nodes.AddNode(element);
     if (SI.Attr['default']<>'') and Defaults then XN.Value:=SI.Attr['default'];
    end else begin
     if datatype<>'' then begin
      XN:=Node.Nodes.AddNode(element);
      Schema1:=Schema.Nodes.NodebyNameAttr(NS+'complexType', 'name', datatype);
      if Assigned(Schema1) then IndexNode(XN, Schema1, level+1);
     end else begin
      XN:=Node.Nodes.AddNode(SI.Attr['name']);
      if (SI.Attr['default']<>'') and Defaults then XN.Value:=SI.Attr['default'];
      IndexNode(XN, SI, level+1);
     end;
    end
  end else
  if SI.Name=NS+'extension' then begin
  end else
  if SI.Name=NS+'complexType' then begin IndexNode(Node, SI, level+1); processed:=true end else
  if SI.Name=NS+'choice' then begin IndexNode(Node, SI, level+1); processed:=true end else
  if SI.Name=NS+'simpleContent' then begin IndexNode(Node, SI, level+1); processed:=true end
 end;
end;

begin
 Element:=Schema;
 Schema:=Schema.Root;
 NS:=Schema.NameSpace;
 Res:=TXMLNode.Create('<'+Element.Attr['name']+'/>');
 if (Element.Attr['type']<>'') and not StartsWith(Element.Attr['type'], NS)
  then Element:=Schema.Nodes.NodebyNameAttr(NS+'complexType', 'name', Element.Attr['type']);
 IndexNode(Res, Element, 1);
 Result:=Res;
end;

procedure TXMLNode.DeleteNode(index: integer);
begin
 Nodes[Index].Free;
 Nodes.Delete(index)
end;

procedure TXMLNode.ReplaceNode(OldNode, NewNode: TXMLNode);
var n:integer;
begin
 n:=Nodes.Indexof(OldNode);
 if n=-1 then raise Exception.Create('Replaced node not found');
 Nodes.Items[n]:=NewNode;
 NewNode.Parent:=Self;
 OldNode.Free
end;

function TXMLNode.Root: TXMLNode;
begin
 Result:=Self;
 while Result.Parent<>nil do Result:=Result.Parent;
end;

procedure TXMLNode.MoveChildsTo(NewParent: TXMLNode);
begin
 while Count>0 do begin
  Nodes[0].Parent:=NewParent;
  NewParent.Nodes.Add(Nodes[0]);
  Nodes.Delete(0);
 end; 
end;

function TXMLNode.XPath: string;
var N: TXMLNode;
begin
 Result:=Name;
 N:=Parent;
 while (N<>nil) and (N.Parent<>nil) do begin
  Result:=N.Name+'/'+Result;
  N:=N.Parent
 end;
end;

procedure ApplyXMLTemplate(Node, Template: TXMLNode; const Masks, Values: array of string);
var i: integer;

function XValue(s: string): string;
var k: integer;
begin
 for k:=0 to High(Masks) do s:=Q_ReplaceStr(s, Masks[k], Values[k]);
 Result:=s
end;

begin
 for i:=0 to Template.Attributes.Count-1 do
  Node.Attr[Template.Attributes[i].Name]:=XValue(Template.Attributes[i].Value);
 for i:=0 to Template.Nodes.Count-1 do if Template.Nodes[i].NodeType=ntNode then begin
  if Template.Nodes[i].Value<>'' then
   Node.Node[Template.Nodes[i].Name]:=XValue(Template.Nodes[i].Value);
  if Template.Nodes[i].Count>0 then begin
   Node.Nodes.GetCreateNode(Template.Nodes[i].Name);
   ApplyXMLTemplate(Template.Nodes[i], Node.Nodes.GetCreateNode(Template.Nodes[i].Name), Masks, Values);
  end;
 end;
end;


{ TXSDNode }

function TXSDNode.AddElement(const EName, EType: string; Description: string=''; Nillable: boolean=true): TXSDNode;
begin
 Result:=TXSDNode(Nodes.AddNode(NameSpace+'element'));
 Result.Attr['name']:=EName;
 Result.Attr['type']:=Etype;
 if not Nillable then Result.Attr['nillable']:='false';
 if Description<>'' then Result.Node[NameSpace+'annotation/'+NameSpace+'documentation']:=Description;
end;


{-------------- TXMLDictionary ---------------------------}


function TXMLDictionary.GetCount: integer;
begin
 Result:=FList.Count
end;

constructor TXMLDictionary.Create;
begin
 inherited;
 FList:=TFastList.Create;
 FList.OwnsObjects:=true;
 FList.Sorted:=true;
end;

destructor TXMLDictionary.Destroy;
begin
 FreeAndNil(FList);
 inherited;
end;

function TXMLDictionary.AddItem(const id: string; Item: TXMLNode): integer;
begin
 Result:=FList.AddObject(id, Item);
end;

procedure TXMLDictionary.DeleteItem(const id: string);
var n: Integer;
begin
 if FList.Find(id, n) then begin
  FList.Objects[n].Free;
  FList.Delete(n);
 end else raise Exception.CreateFmt('ID %s not found in dictionary', [id]);
end;

procedure TXMLDictionary.UpdateItem(const id: string; Item: TXMLNode);
var n: Integer;
begin
 if FList.Find(id, n) then begin
  if FList.Objects[n]<>Item then begin
   FList.Objects[n].Free;
   FList.Objects[n]:=Item
  end;
 end;
end;

function TXMLDictionary.FindItem(const id: string): TXMLNode;
var n: integer;
begin
 if FList.Find(id, n) then Result:=TXMLNode(FList.Objects[n]) else Result:=nil;
end;

function TXMLDictionary.Ids(Index: integer): string; begin Result:=FList[Index] end;

function TXMLDictionary.Names(Index: Integer): string;
begin
 Result:=Items(Index).Attr['name']
end;

function TXMLDictionary.Items(Index: integer): TXMLNode;
begin
 Result:=TXMLNode(FList.Objects[Index])
end;

function TXMLDictionary.IndexOf(const id: string): Integer;
begin
 Result:=FList.Indexof(id)
end; 

procedure TXMLNode.Assign(Source: TXMLNode);
var i: integer;
begin
 Nodes.ClearNodes;
 Attributes.ClearNodes;
 for i:=0 to Source.Attributes.Count-1 do Attributes.AddNode(Source.Attributes[i].Name, Source.Attributes[i].Value, ntAttr);
 for i:=0 to Source.Count-1 do Nodes.Add(TXMLNode.Create(Source.Nodes[i].Code, Self));
end;

function TXMLNode.NameSpace: string;
begin
 Result:=Root.Name;
 Result:=copy(Result, 1, pos(':', Result));
end;

end.
