/*++

Copyright (c) 1997-1998 Jetico, Inc.

Module Name: BC_ALG.c
Abstract:    This is a common code for all encryption algorithm drivers.
Author:      Sergey A. Frolov
Environment: Kernel mode only.

Notes:

Revision History:

--*/


#include <ntddk.h>
#include <ntiologc.h>
#include <ntdddisk.h>

#include <string.h>

#include "bc_ioctl.h"
#include "bc_alg.h"
#include "bcstatus.h"
#include "algorithm.h"


#if DBG
ULONG   BCDiskDebugLevel = 2;
BOOLEAN shouldBreak	= TRUE;
#else
BOOLEAN shouldBreak	= FALSE;
#endif

/***************************************************************************/

#define TABLE_LENGTH 32

KEY_ELEMENT KeyTable[ TABLE_LENGTH ];

/***************************************************************************/

NTSTATUS
DriverEntry(
    IN OUT PDRIVER_OBJECT   DriverObject,
    IN PUNICODE_STRING      RegistryPath
    )
/*++

Routine Description:
    This routine is called by the Operating System to initialize the driver.
    It fills in the dispatch entry points in the driver object.  Then
    BCalgInitializeDisk is called to create the device object and complete
    the initialization.

Arguments:
    DriverObject - a pointer to the object that represents this device
    driver.
    RegistryPath - a pointer to our Services key in the registry.

Return Value:
    STATUS_SUCCESS if this disk is initialized; an error otherwise.
--*/

{
    NTSTATUS         ntStatus;
    USHORT i;

    // if (shouldBreak) DbgBreakPoint();

    //
    // Initialize the driver object with this driver's entry points.
    DriverObject->MajorFunction[IRP_MJ_CREATE] = BCalgCreateClose;
    DriverObject->MajorFunction[IRP_MJ_CLOSE]  = BCalgCreateClose;
    DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL] = BCalgDeviceControl;

    // We'll enable unload when the Filesystem drivers support it
    //DriverObject->DriverUnload = BCalgUnloadDriver;

    // Initialize KeyTable
    for(i=0; i<TABLE_LENGTH; i++) KeyTable[i].keyExtended = NULL;
    
	ntStatus = BCalgInitializeDriver(DriverObject);

    return ntStatus;
}



/***************************************************************************/

NTSTATUS
BCalgInitializeDriver( IN PDRIVER_OBJECT   DriverObject )

/*++
Routine Description:
    This routine is called at initialization time by DriverEntry().

    It creates and initializes a device object.

Arguments:
    DriverObject - a pointer to the object that represents this device driver.
    ParamPath - a pointer to the Parameters key under our key in the Services
        section of the registry.

Return Value:
    STATUS_SUCCESS if this disk is initialized; an error otherwise.
--*/

{
    PDEVICE_OBJECT      deviceObject = NULL;    // ptr to device object
    NTSTATUS            ntStatus;
	STRING			    serviceNameString;
    UNICODE_STRING      deviceNameUString, prefix, linkPrefix, suffix;
    PBCALG_EXTENSION   gostExtension;


    // Initialize Unicode string for Service Name
    RtlInitString( &serviceNameString, SERVICE_NAME );
    ntStatus = RtlAnsiStringToUnicodeString( &suffix, &serviceNameString, TRUE );
    if (!NT_SUCCESS( ntStatus )) goto BCalgInitializeDiskExit;
    
    // Initialize deviceNameUString
	RtlInitUnicodeString (&prefix, L"\\Device\\") ; 

	deviceNameUString.Length = 0;
	deviceNameUString.MaximumLength = prefix.Length + suffix.Length + sizeof(WCHAR);
	deviceNameUString.Buffer = ExAllocatePool( PagedPool, deviceNameUString.MaximumLength);

	if (!deviceNameUString.Buffer) 
		{ ntStatus = STATUS_INSUFFICIENT_RESOURCES;
          goto BCalgInitializeDiskExit;
	    }

    RtlAppendUnicodeStringToString( &deviceNameUString, &prefix );
    RtlAppendUnicodeStringToString( &deviceNameUString, &suffix );
             
	// Create DeviceObject
    ntStatus = IoCreateDevice( DriverObject,             // Our Driver Object
                               sizeof(BCALG_EXTENSION), // Size of device extension 
                               &deviceNameUString,       // Device name "\Device\BC_gost"
                               FILE_DEVICE_UNKNOWN,      // Device type
                               0,                        // Device characteristics
                               FALSE,                    // Exclusive device
                               &deviceObject );          // Returned ptr to Device Object

    if ( !NT_SUCCESS( ntStatus ) ) goto BCalgInitializeDiskExit;
    
    // Initialize device object
    deviceObject->Flags |= DO_DIRECT_IO;
    deviceObject->AlignmentRequirement = FILE_WORD_ALIGNMENT;
    
    gostExtension = (PBCALG_EXTENSION)deviceObject->DeviceExtension;
    gostExtension->DeviceObject = deviceObject;
  
    // Initialize gostExtension->deviceLinkUString
    RtlInitUnicodeString (&linkPrefix, L"\\DosDevices\\") ; 

	gostExtension->deviceLinkUString.Length = 0;
	gostExtension->deviceLinkUString.MaximumLength = linkPrefix.Length + suffix.Length + sizeof(WCHAR);
	gostExtension->deviceLinkUString.Buffer = ExAllocatePool( PagedPool, gostExtension->deviceLinkUString.MaximumLength);

	if (!gostExtension->deviceLinkUString.Buffer) 
		{ ntStatus = STATUS_INSUFFICIENT_RESOURCES;
          goto BCalgInitializeDiskExit;
	    }

    RtlAppendUnicodeStringToString( &gostExtension->deviceLinkUString, &linkPrefix );
    RtlAppendUnicodeStringToString( &gostExtension->deviceLinkUString, &suffix );
             
	
	// Create a symbolic link between our device name "\Device\BC_gost" and
    // the Win32 name (ie "\DosDevices\BC_gost").
    ntStatus = IoCreateSymbolicLink( &gostExtension->deviceLinkUString, &deviceNameUString );


BCalgInitializeDiskExit:

    if (deviceNameUString.Buffer != NULL) RtlFreeUnicodeString( &deviceNameUString );
    
    if ( !NT_SUCCESS( ntStatus ) )
      {
        // Delete everything that this routine has allocated.
        if ( deviceObject != NULL ) 
		  { if (gostExtension)
               if (gostExtension->deviceLinkUString.Buffer != NULL) 
			      RtlFreeUnicodeString( &gostExtension->deviceLinkUString );
            IoDeleteDevice( deviceObject );
		  }
	  }
    return ntStatus;
}



/***************************************************************************/

NTSTATUS
BCalgCreateClose(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
    )
/*++
Routine Description:
    This routine is called by the I/O system when the BCDisk is opened or closed.
    No action is performed other than completing the request successfully.

Arguments:
    DeviceObject - a pointer to the object that represents the device
    that I/O is to be done on.
    Irp - a pointer to the I/O Request Packet for this request.

Return Value:
    STATUS_INVALID_PARAMETER if parameters are invalid,
    STATUS_SUCCESS otherwise.
--*/

{
    Irp->IoStatus.Status = STATUS_SUCCESS;
    Irp->IoStatus.Information = 0;
    IoCompleteRequest( Irp, IO_NO_INCREMENT );
    return STATUS_SUCCESS;
}



/***************************************************************************/

NTSTATUS
BCalgDeviceControl(
    IN PDEVICE_OBJECT DeviceObject,
    IN PIRP Irp
    )

/*++
Routine Description:
    This routine is called by the I/O system to perform a device I/O
    control function.

Arguments:
    DeviceObject - a pointer to the object that represents the device
    that I/O is to be done on.
    Irp - a pointer to the I/O Request Packet for this request.

Return Value:
    STATUS_SUCCESS if recognized I/O control code,
    STATUS_INVALID_DEVICE_REQUEST otherwise.
--*/

{   PIO_STACK_LOCATION  irpSp;
    NTSTATUS            ntStatus;

    // Set up necessary object and extension pointers.
    irpSp = IoGetCurrentIrpStackLocation( Irp );

    // Assume failure.
    Irp->IoStatus.Status = STATUS_INVALID_DEVICE_REQUEST;

    // Determine which I/O control code was specified.
    switch ( irpSp->Parameters.DeviceIoControl.IoControlCode )
    {
      case IOCTL_PRIVATE_GET_VERSION_INFO:
		if ( irpSp->Parameters.DeviceIoControl.OutputBufferLength < sizeof(VERSION_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PVERSION_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PVERSION_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				// Get output buffer if its passed as an MDL
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(outputBuffer->Signature),"LS06CXX ", 8 );
				outputBuffer->Status		= BCSTATUS_SUCCESS;
				outputBuffer->Major_version = DRIVER_MAJOR_VERSION;
				outputBuffer->Minor_version = DRIVER_MINOR_VERSION;

				Irp->IoStatus.Status = STATUS_SUCCESS;
				Irp->IoStatus.Information = sizeof(VERSION_BUFFER);
			 }
		break;

      case IOCTL_PRIVATE_GET_INFO:
		if ( irpSp->Parameters.DeviceIoControl.OutputBufferLength < sizeof(INFO_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PINFO_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PINFO_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;
				char dName[] = DISPLAY_NAME;
				int i;

				// Get output buffer if its passed as an MDL
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(outputBuffer->Signature),"LS06CXX ", 8 );
				outputBuffer->Status		= BCSTATUS_SUCCESS;

				outputBuffer->AlgorithmID = ALGORITHM_ID;
				outputBuffer->KeyLength   = KEY_LENGTH;
				for(i=0; (i<DISPLAY_NAME_LENGTH) && (dName[i] != '\0'); i++)
				  { outputBuffer->DisplayName[i] = dName[i]; }
				dName[ DISPLAY_NAME_LENGTH - 1 ] = '\0';

				Irp->IoStatus.Status = STATUS_SUCCESS;
				Irp->IoStatus.Information = sizeof(INFO_BUFFER);
			 }
		break;

      case IOCTL_PRIVATE_CREATE_KEY_HANDLE:
		if ( irpSp->Parameters.DeviceIoControl.OutputBufferLength < sizeof(CREATEKEY_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PCREATEKEY_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PCREATEKEY_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				// Get output buffer if its passed as an MDL
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(outputBuffer->Signature),"LS06CXX ", 8 );

				ntStatus = BCalgCreateKey(outputBuffer);

				//outputBuffer->Status		= BCSTATUS_SUCCESS;
				Irp->IoStatus.Status = ntStatus;
				Irp->IoStatus.Information = sizeof( CREATEKEY_BUFFER );
			 }
		break;

      case IOCTL_PRIVATE_TEST_KEY_HANDLE:
		if ( irpSp->Parameters.DeviceIoControl.OutputBufferLength < sizeof(TESTKEY_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PTESTKEY_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PTESTKEY_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(outputBuffer->Signature),"LS06CXX ", 8 );

				ntStatus = BC_TestKey( inputBuffer->KeyHandle );
              
				if (!NT_SUCCESS( ntStatus ))
				  { outputBuffer->Status = BCSTATUS_ERR_INVALID_KEY;
				    outputBuffer->IsValid = FALSE;
				  }
				else 
				  {	outputBuffer->Status = BCSTATUS_SUCCESS;
				    outputBuffer->IsValid = TRUE;
				  }

				Irp->IoStatus.Status = ntStatus;
				Irp->IoStatus.Information = sizeof( TESTKEY_BUFFER );
			 }
		break;

      case IOCTL_PRIVATE_FREE_KEY_HANDLE:
		if ( irpSp->Parameters.DeviceIoControl.OutputBufferLength < sizeof(TESTKEY_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PTESTKEY_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PTESTKEY_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(outputBuffer->Signature),"LS06CXX ", 8 );

				ntStatus = BC_FreeKey( inputBuffer->KeyHandle );
              
				if (!NT_SUCCESS( ntStatus ))
				  { outputBuffer->Status = BCSTATUS_ERR_INVALID_KEY;
				    outputBuffer->IsValid = FALSE;
				  }
				else 
				  {	outputBuffer->Status = BCSTATUS_SUCCESS;
				    outputBuffer->IsValid = TRUE;
				  }

				Irp->IoStatus.Status = ntStatus;
				Irp->IoStatus.Information = sizeof( TESTKEY_BUFFER );
			 }
		break;


      case IOCTL_PRIVATE_ENCRYPT:
		if ( irpSp->Parameters.DeviceIoControl.InputBufferLength < sizeof(ENCRYPT_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PENCRYPT_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PUCHAR          outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				// Get output buffer if its passed as an MDL (METHOD_OUT_DIRECT used)
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				//RtlCopyMemory(&(inputBuffer->Signature),"LS06CXX ", 8 );
				ntStatus = BC_Encrypt(inputBuffer->KeyHandle,
                                      inputBuffer->IVector,
                                      outputBuffer, outputBuffer,
                                      inputBuffer->Length );

				//outputBuffer->Status		= BCSTATUS_SUCCESS;
				Irp->IoStatus.Status = ntStatus;
				Irp->IoStatus.Information = sizeof( ENCRYPT_BUFFER );
			 }
		break;

      case IOCTL_PRIVATE_DECRYPT:
		if ( irpSp->Parameters.DeviceIoControl.InputBufferLength < sizeof(ENCRYPT_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PENCRYPT_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
				PUCHAR          outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				// Get output buffer if its passed as an MDL (METHOD_OUT_DIRECT used)
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				//RtlCopyMemory(&(inputBuffer->Signature),"LS06CXX ", 8 );
				ntStatus = BC_Decrypt(inputBuffer->KeyHandle,
                                      inputBuffer->IVector,
                                      outputBuffer, outputBuffer,
                                      inputBuffer->Length );

				//outputBuffer->Status		= BCSTATUS_SUCCESS;
				Irp->IoStatus.Status = ntStatus;
				Irp->IoStatus.Information = sizeof( ENCRYPT_BUFFER );
			 }
		break;

      case IOCTL_PRIVATE_GETFUNCTIONS:
		if ( irpSp->Parameters.DeviceIoControl.InputBufferLength < sizeof(GETFUNC_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PGETFUNC_BUFFER inputBuffer  = Irp->AssociatedIrp.SystemBuffer;
				PGETFUNC_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				// Get output buffer if its passed as an MDL
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(inputBuffer->Signature),"LS06CXX ", 8 );
                
                outputBuffer->FuncTestKey = (PVOID)(&BC_TestKey);
                outputBuffer->FuncFreeKey = (PVOID)(&BC_FreeKey);
                outputBuffer->FuncEncrypt = (PVOID)(&BC_Encrypt);
                outputBuffer->FuncDecrypt = (PVOID)(&BC_Decrypt);

				//outputBuffer->Status		= BCSTATUS_SUCCESS;
				Irp->IoStatus.Status = STATUS_SUCCESS;
				Irp->IoStatus.Information = sizeof(GETFUNC_BUFFER);
			 }
		break;

      case IOCTL_PRIVATE_SUPPORTS_SYNCRO_64BITS:
		if ( irpSp->Parameters.DeviceIoControl.OutputBufferLength < sizeof(SYNCRO64BITS_BUFFER) )
			 {	Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; }
		else {	PSYNCRO64BITS_BUFFER inputBuffer = Irp->AssociatedIrp.SystemBuffer;
		        PSYNCRO64BITS_BUFFER outputBuffer = Irp->AssociatedIrp.SystemBuffer;

				// Get output buffer if its passed as an MDL
				if( Irp->MdlAddress ) 
					outputBuffer = MmGetSystemAddressForMdl( Irp->MdlAddress );

				if ( !RtlCompareMemory( &(inputBuffer->Signature), "LOCOS97 ", 8))
					{ Irp->IoStatus.Status = STATUS_INVALID_PARAMETER; 
					  break;
					}

				RtlCopyMemory(&(outputBuffer->Signature),"LS06CXX ", 8 );
				outputBuffer->Status        = BCSTATUS_SUCCESS;
				outputBuffer->SupportStatus = TRUE;

				Irp->IoStatus.Status = STATUS_SUCCESS;
				Irp->IoStatus.Information = sizeof(VERSION_BUFFER);
			 }
		break;

	default:

        // The specified I/O control code is unrecognized by this driver.
        // The I/O status field in the IRP has already been set so just
        // terminate the switch.
        break;
    }

    // Finish the I/O operation by simply completing the packet and returning
    // the same status as in the packet itself.
    ntStatus = Irp->IoStatus.Status;
    IoCompleteRequest( Irp, IO_NO_INCREMENT );
    return ntStatus;
}

/***************************************************************************/

VOID
BCalgUnloadDriver(
    IN PDRIVER_OBJECT DriverObject
    )
/*++
Routine Description:
    This routine is called by the I/O system to unload the driver.
    Any resources previously allocated must be freed.

Arguments:
    DriverObject - a pointer to the object that represents our driver.

Return Value:
    None
--*/

{
    PDEVICE_OBJECT      deviceObject = DriverObject->DeviceObject;

    if ( deviceObject != NULL )
    {
        PBCALG_EXTENSION gostExtension = deviceObject->DeviceExtension;

        if (gostExtension != NULL)
        { if (gostExtension->deviceLinkUString.Buffer != NULL)
           { IoDeleteSymbolicLink( &gostExtension->deviceLinkUString );
             ExFreePool( gostExtension->deviceLinkUString.Buffer );
           }
        }
        IoDeleteDevice( deviceObject );
    }
}


/***************************************************************************/

NTSTATUS
BCalgCreateKey(IN PCREATEKEY_BUFFER CkBuffer)
/*++
Routine Description:
 
Arguments:
 
Return Value:
--*/
{ KEY_HANDLE   keyHandle;
  UCHAR *keyExtended = NULL;
  UCHAR *pool        = NULL;    
	
  if ((CkBuffer->KeyLength) != KEY_LENGTH)
    { CkBuffer->Status = BCSTATUS_ERR_INVALID_KEY_LENGTH;
	  goto errorExit;
    }

  // Define an empty element in Key Table
  for(keyHandle = 0; keyHandle < TABLE_LENGTH ; keyHandle++)
    if (KeyTable[keyHandle].keyExtended == NULL) break;

  if (keyHandle >= TABLE_LENGTH) 
    { CkBuffer->Status = BCSTATUS_ERR_NOSYSTEM_RESOURCES;
      goto errorExit;
    }
  
  
  // Allocate memory space for Key structure
  keyExtended = (UCHAR *)ExAllocatePool( 
	                       NonPagedPool, KEY_LENGTH_EXTENDED / 8 );
  pool        = (UCHAR *)ExAllocatePool( 
	                       NonPagedPool, POOL_SIZE_BYTES );
  if ((keyExtended == NULL) ||
	  (pool        == NULL)
	 )
    { CkBuffer->Status = BCSTATUS_ERR_NOSYSTEM_RESOURCES;
      goto errorExit;
    }

  // Initialize pool of random data
  RtlCopyBytes( pool, CkBuffer->Pool, POOL_SIZE_BYTES );
  
  // Initialize Key structure
  BC_ExtendKey( CkBuffer->Key, keyExtended );

  KeyTable[keyHandle].pool        = pool;
  KeyTable[keyHandle].keyExtended = keyExtended;
  CkBuffer->KeyHandle = keyHandle + 1; // 0 is invalid Key HAndle
  CkBuffer->Status = BCSTATUS_SUCCESS;
  return STATUS_SUCCESS;

errorExit:
  if (keyExtended != NULL)
  { ExFreePool(keyExtended);
  }
  if (pool != NULL)
  { ExFreePool(pool);
  }
  CkBuffer->KeyHandle = 0;
  return STATUS_SUCCESS;
}


/***************************************************************************/

BOOL 
BC_ExtendKey( PUCHAR KeyFrom, PUCHAR KeyTo )
/*++
Routine Description:
 
Arguments:
 
Return Value:
--*/
{
  if ((!KeyFrom) || (!KeyTo)) return FALSE;
  
  return KeyExtend( KeyFrom, (PDWORD) KeyTo );
}


/**************************************************************************/

VOID
Transform_IV( DWORD *IVector, DWORD *Pool)
{
	DWORD i;
	i = (IVector[0] & 0x3f) << 1;
	IVector[0] ^= Pool[i];
	IVector[1] ^= Pool[i+1];
}


VOID Increment_IV( PUCHAR Syncro )
{
  PWORD          pdwSyncro;
  UINT           i;
  PLARGE_INTEGER pliSyncro;

  pdwSyncro = (PWORD)Syncro;

  if ( (pdwSyncro[0] == pdwSyncro[1]) && // for compatibility with older versions of BestCrypt
       (pdwSyncro[1] == pdwSyncro[2]) && 
       (pdwSyncro[2] == pdwSyncro[3]) )
  {
     
     for( i=0; i<4; i++) (pdwSyncro[i])++;
  }
  else
  {
     pliSyncro = (PLARGE_INTEGER)Syncro;
     (pliSyncro->QuadPart)++;
  }
}



VOID
Transform_Buf(	DWORD *Pool, 
                DWORD *InBuffer, 
                DWORD *OutBuffer, 
                DWORD Length )    // Length in DWORDs
{
	DWORD i;
	for ( i=0; i<Length; i++ ) OutBuffer[i] = InBuffer[i] ^ Pool[i];
}


/***************************************************************************/

NTSTATUS 
BC_Encrypt( KEY_HANDLE KeyHandle,
            PUCHAR     IVector,
            PUCHAR     InBuffer,
            PUCHAR     OutBuffer,
            DWORD      Length )       // in bytes
/*++
Routine Description:
 
Arguments:
 
Return Value:
--*/
{ KEY_HANDLE hKey  = KeyHandle - 1;
  DWORD length = Length;
  PWORD s;
  WORD  ivector[4], ivectorS[4];
  UINT  i, offset;
  
  if (!Length) return STATUS_SUCCESS;

  if (!NT_SUCCESS( BC_TestKey( KeyHandle ) ))
     return STATUS_INSUFFICIENT_RESOURCES;

  s = (PWORD)IVector;
  for( i=0; i<4; i++) ivector[i] = ivectorS[i] = s[i];
  
  for( offset = 0; ; offset += 512)
  { 
	Transform_IV(	(PDWORD)ivector, (PDWORD)KeyTable[hKey].pool );
	if (length > 512) 
	  { length = length - 512;
		Transform_Buf(	(PDWORD)KeyTable[hKey].pool,
						(PDWORD)(&(InBuffer[offset] )),
						(PDWORD)(&(OutBuffer[offset])), 128  );
        Encrypt( (PDWORD)ivector,  (PDWORD)KeyTable[hKey].keyExtended,
                 (PDWORD)(&(OutBuffer[offset] )),
				 (PDWORD)(&(OutBuffer[offset])), 512 );
	  }
	else
	  {	Transform_Buf(	(PDWORD)KeyTable[hKey].pool,
						(PDWORD)(&(InBuffer[offset] )),
						(PDWORD)(&(OutBuffer[offset])), length >> 2 );
		Encrypt( (PDWORD)ivector,  (PDWORD)KeyTable[hKey].keyExtended,
                 (PDWORD)(&(OutBuffer[offset] )),
				 (PDWORD)(&(OutBuffer[offset])), length );
	    break;
	  }
    // We need to update IVector every 512 bytes
    Increment_IV( (PUCHAR)ivectorS );
    for( i=0; i<4; i++) ivector[i] = ivectorS[i]; 
  }
  return STATUS_SUCCESS;
}


/***************************************************************************/

NTSTATUS 
BC_Decrypt( KEY_HANDLE KeyHandle,
            PUCHAR     IVector,
            PUCHAR     InBuffer,
            PUCHAR     OutBuffer,
            DWORD      Length )       // in bytes
/*++
Routine Description:
 
Arguments:
 
Return Value:
--*/
{ KEY_HANDLE hKey  = KeyHandle - 1;
  DWORD length = Length;
  PWORD s;
  WORD  ivector[4], ivectorS[4];
  UINT  i, offset;

  if (!Length) return STATUS_SUCCESS;

  if (!NT_SUCCESS( BC_TestKey( KeyHandle ) ))
     return STATUS_INSUFFICIENT_RESOURCES;

  s = (PWORD)IVector;
  for( i=0; i<4; i++) ivector[i] = ivectorS[i] = s[i];
  
  for( offset = 0; ; offset += 512)
  {
	Transform_IV(	(PDWORD)ivector, (PDWORD)KeyTable[hKey].pool );
	if (length > 512) 
	  { length = length - 512;
        Decrypt( (PDWORD)ivector,  (PDWORD)KeyTable[hKey].keyExtended,
                 (PDWORD)(&(InBuffer[offset] )),
				 (PDWORD)(&(OutBuffer[offset])), 512 );
		Transform_Buf(	(PDWORD)KeyTable[hKey].pool,
						(PDWORD)(&(OutBuffer[offset] )),
						(PDWORD)(&(OutBuffer[offset])), 128 );
	  }
	else
	  { Decrypt( (PDWORD)ivector,  (PDWORD)KeyTable[hKey].keyExtended,
                 (PDWORD)(&(InBuffer[offset] )),
				 (PDWORD)(&(OutBuffer[offset])), length );
		Transform_Buf(	(PDWORD)KeyTable[hKey].pool,
						(PDWORD)(&(OutBuffer[offset] )),
						(PDWORD)(&(OutBuffer[offset])), length >> 2 );
	    break;
	  }
    // We need to update IVector every 512 bytes
    Increment_IV( (PUCHAR)ivectorS );
    for( i=0; i<4; i++) ivector[i] = ivectorS[i]; 
  }
  return STATUS_SUCCESS;
}



/***************************************************************************/

NTSTATUS 
BC_TestKey( KEY_HANDLE KeyHandle )
/*++
Routine Description:
 
Arguments:
 
Return Value:
--*/
{ KEY_HANDLE hKey = KeyHandle - 1;
  
  if (hKey >= TABLE_LENGTH) return STATUS_INSUFFICIENT_RESOURCES;
  
  if (!(KeyTable[hKey].keyExtended)) return STATUS_INSUFFICIENT_RESOURCES;

  return STATUS_SUCCESS;
}


/***************************************************************************/

BC_ALGID_BLOCK sig = { EYE_CATCH_STRING, ALGORITHM_ID, KEY_LENGTH, DISPLAY_NAME, SERVICE_NAME, EYE_CATCH_STRING_1, sizeof(BC_ALGID_BLOCK), ALGORITHM_BLOCK_SIZE_BYTES };

NTSTATUS 
BC_FreeKey( KEY_HANDLE KeyHandle )
/*++
Routine Description:
 
Arguments:
 
Return Value:
--*/
{ KEY_HANDLE hKey = KeyHandle - 1;
  DWORD      i;

  if (!NT_SUCCESS( BC_TestKey( KeyHandle ) ))
     return STATUS_INSUFFICIENT_RESOURCES;

  /* Erase key data and pool from memory */
  RtlFillMemory( KeyTable[hKey].keyExtended, KEY_LENGTH_EXTENDED / 8, 0x55);
  RtlFillMemory( KeyTable[hKey].pool, POOL_SIZE_BYTES, 0x55);

  ExFreePool( KeyTable[hKey].keyExtended );
  ExFreePool( KeyTable[hKey].pool );

  KeyTable[hKey].keyExtended = NULL;
  KeyTable[hKey].pool = NULL;

  return STATUS_SUCCESS;
}


