/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSActionA.h
*********************************************************************/

#ifndef VSActionA_H
#define VSActionA_H
#ifndef __GNUC__ /* suppress gcc warning */
#pragma once
#endif /* __GNUC__ */

#include "VSTypes.h"
#include "VSThreadA.h"

#ifdef __cplusplus
extern "C"{
#endif



    /* Available types of the PDF actions */
    typedef enum PDFActionType {
        pdfActionGoTo                   = 0,    /* Go to a destination in the current document.                 */
        pdfActionGoToR,                         /* ("Go-to remote") Go to a destination in another document.    */ 
        pdfActionLaunch,                        /* Launch an application, usually to open a file.               */
        pdfActionThread,                        /* Begin reading an article thread.                             */
        pdfActionURI,                           /* Resolve a uniform resource identifier.                       */
        pdfActionSound,                         /* Play a sound.                                                */
        pdfActionMovie,                         /* Play a movie.                                                */
        pdfActionHide,                          /* Set an annotation's Hidden flag.                             */
        pdfActionNamed,                         /* Execute an action predefined by the viewer application.      */
        pdfActionSubmitForm,                    /* Send data to a uniform resource locater.                     */
        pdfActionResetForm,                     /* Set fields to their default values.                          */
        pdfActionImportData,                    /* Import field values from a file.                             */
        pdfActionJavaScript,                    /* Execute a JavaScript script.                                 */
        pdfActionUnknow                         /* Other unknown type.                                           */
    } PDFActionType;


    /* PDF explicit destination types */
    typedef enum PDFExplicitDestType {
        edtXYZ,                                 /*  Display the page, with the coordinates (<i>left</i>, <i>top</i>) positioned
                                                    at the top-left corner of the window and the contents of the page magnified
                                                    by the factor <i>zoom</i>. A null value for any of the parameters <i>left</i>, <i>top</i>, or
                                                    <i>zoom</i> specifies that the current value of that parameter is to be retained unchanged. */
        edtFit,                                 /*  Display the page, with its contents magnified just enough
                                                    to fit the entire page within the window both horizontally and vertically. If
                                                    the required horizontal and vertical magnification factors are different, use
                                                    the smaller of the two, centering the page within the window in the other dimension. */   
        edtFitH,                                /*  Display the page, with the vertical coordinate top positioned
                                                    at the <i>top</i> edge of the window and the contents of the page magnified
                                                    just enough to fit the entire width of the page within the window.                   */
        edtFitV,                                /*  Display the page, with the horizontal coordinate left positioned
                                                    at the <i>left</i> edge of the window and the contents of the page magnified
                                                    just enough to fit the entire height of the page within the window.                  */ 
        edtFitR,                                /*  Display the page, with its contents magnified just enough to fit the rectangle 
                                                    specified by the coordinates <i>left</i>, <i>bottom</i>, <i>right</i>, and <i>top</i> entirely 
                                                    within the window both horizontally and vertically. If the required horizontal and 
                                                    vertical magnification factors are different, use the smaller of the two, centering 
                                                    the rectangle within the window in the other dimension.                              */
        edtFitB,                                /*  Display the page, with its contents magnified just enough to fit its bounding box 
                                                    entirely within the window both horizontally and vertically. If the required horizontal 
                                                    and vertical magnification factors are different, use the smaller of the two, centering 
                                                    the bounding box within the window in the other dimension. */
        edtFitBH,                               /*  Display the page, with the vertical coordinate top positioned at the <i>top</i>
                                                    edge of the window and the contents of the page magnified just enough to fit the entire 
                                                    width of its bounding box within the window.                                         */
        edtFitBV                                /*  Display the page, with the horizontal coordinate left positioned at the <i>left</i> edge of 
                                                    the window and the contents of the page magnified just enough to fit the entire height 
                                                    of its bounding box within the window.                                               */   
    } PDFExplicitDestType;


    /* This struct for specifying a destination explicitly in a PDF file */
    typedef struct  {
        PDFExplicitDestType Type;           /*  Explicit destination type                           */
        ppInt32             Page;           /*  Index of the page in PDF document (Begin from 0 )   */
        ppReal              a;              /*  Value depended from type of the destination 
                                                left for edtXYZ, top for edtFitH and edtFitBH, left for edtFitV,edtFitBV, and edtFitR */
                                            
        ppReal              b;              /*  Value depended from type of the destination          
                                                top for edtXYZ, bottom for edtFitR                  */
        ppReal              c;              /*  Value depended from type of the destination          
                                                zoom for edtXYZ, right for edtFitR                  */
        ppReal              d;              /*  Value depended from type of the destination          
                                                top for edtFitR                                     */
    } PDFExplicitDest, *PPDFExplicitDest;


    /* Type of the PDF destination */
    typedef enum PDFDestinationType {
        pdfdtExplicit,                  /* Directly via explicit destination    */
        pdfdtNamed                      /* Indirectly via name destination      */
    } PDFDestinationType;

    /* Description
       Creates new destination name in name table and assigns to it
       explicit destination
       Parameters
       Doc :          [in] PDF Document handle
       String :       [in] Specifies the name which will be appended to
                      name table.
       Length :       [in] Length of the name
       Destination :  [in] Explicit destination which will be respected
                      to this name
       Returns
       None.                                                            */
    void PDFDestinationNameNew(PDFDocHandle Doc, char *String, ppInt32 Length,
        PDFExplicitDest Destination);



    /* Description
       Converts PDFDestinationHandle ( received from
       PDFActionGetGoToDestination and
       PDFActionGetGoToRemoteDestination functions) to
       PDFExplicitDest structure.
       Parameters
       Doc :          [in] PDF Document handle
       DestH :        [in] Destination handle which is needed to be converted
                      to structure.
       Destination :  [out] Pointer to PDFExplicitDest structure
       Returns
       If the function succeeds, the return value is nonzero. If the
       function fails, the return value is zero.                              */
    ppBool PDFDestinationGetInfo(PDFDocHandle Doc, PDFDestinationHandle DestH,
        PDFExplicitDest *Destination);



    /* Description
       This function returns the type of the specified object.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the action object.
       Returns
       If the function succeeds, the return value will identify the
       \object.                                                     */
    PDFActionType PDFActionGetType(PDFDocHandle Doc, PDFActionHandle Action);



    /* Description
       This function retrieves the number of actions which will be
       executed after this action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the action object.
       Returns
       The number of actions.                                      */
    ppInt32 PDFActionGetNextItemCount(PDFDocHandle Doc, PDFActionHandle Action);



    /* Description
       This function retrieves the actions which will be executed
       after this action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the action object.
       Index :   [in] Index of the action which needs to be
                 received.
       Returns
       The return value is a handle to the specified object.      */
    PDFActionHandle PDFActionGetNextItem(PDFDocHandle Doc, PDFActionHandle Action, ppInt32 Index);



    /* Description
       This function sets action which will be executed after
       current action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the action object.
       Next :    [in] Handle of the next action object.
       Returns
       None.                                                  */
    void PDFActionSetNext(PDFDocHandle Doc, PDFActionHandle Action, PDFActionHandle Next);

    /* GoTo Action */


/* Description
   Creates new "GoTo" action and sets destination to name
   destination.
   Parameters
   Doc :     [in] PDF Document handle
   String :  [in] Name destination.
   Length :  [in] Length of the name destination.
   Returns
   The return value is a handle to a PDF action.          */
    PDFActionHandle PDFActionNewGoToName(PDFDocHandle Doc, char *String, ppInt32 Length);



    /* Description
       Creates new "GoTo" action and sets destination to explicit
       destination.
       Parameters
       Doc :          [in] PDF Document handle
       Destination :  [in] Explicit destination.
       Returns
       The return value is a handle to a PDF action.              */
    PDFActionHandle PDFActionNewGoToDestination(PDFDocHandle Doc, PDFExplicitDest Destination);


    
    /* Description
       \Returns destination handle for this action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the PDF action.
       Returns
       Destination handle which can be processed with
       PDFDestinationGetInfo function.                */
    PDFDestinationHandle PDFActionGetGoToDestination(PDFDocHandle Doc, PDFActionHandle Action);

    /* GoToRemote Action */



    /* Description
       Creates new "GoToRemote" action and sets destination to name
       destination.
       Parameters
       Doc :             [in] PDF Document handle
       FileName :        [in] Filename of the PDF document which need open
       FileNameLength :  [in] Length of the filename.
       String :          [in] Name destination.
       Length :          [in] Length of the name destination.
       InNewWindow :     [in] Specifying whether to open the destination
                         document in a new window. If this flag is false,
                         the destination document will replace the current
                         document in the same window.
       Returns
       The return value is a handle to a PDF action.                       */

    PDFActionHandle PDFActionNewGoToRemoteName(PDFDocHandle Doc, char *FileName,
        ppInt32 FileNameLength, char *String, ppInt32 Length, ppBool InNewWindow);


    /* Description
       Creates new "GoToRemote" action and sets destination to name
       destination.
       Parameters
       Doc :             [in] PDF Document handle
       FileName :        [in] Filename of the PDF document which need open
       FileNameLength :  [in] Length of the filename.
       Destination :     [in] Explicit destination.
       InNewWindow :     [in] Specifying whether to open the destination
                         document in a new window. If this flag is false,
                         the destination document will replace the current
                         document in the same window.
       Returns
       The return value is a handle to a PDF action.                       */
    PDFActionHandle PDFActionNewGoToRemoteDestination(PDFDocHandle Doc, char *FileName,
        ppInt32 FileNameLength, PDFExplicitDest Dest, ppBool InNewWindow);



    /* Description
       \Returns destination handle for this action.
       Parameters
       Doc :       [in] PDF Document handle
       Action :    [in] Handle of the PDF action.
       FileName :  [out] Name of the PDF document which will be opened
                   after execution of this action.
       Returns
       Destination handle which can be processed with
       PDFDestinationGetInfo function.                                 */
    PDFDestinationHandle PDFActionGetGoToRemoteDestination(PDFDocHandle Doc,
        PDFActionHandle Action, PDFString *FileName);



    /* Description
       \Returns stored JavaScript in CosStream or in string.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the PDF action.
       Returns
       If the new document is opened in new window, the return value
       will be nonzero. If the new document is opened in current
       window, the return value will be zero.                        */
    ppBool PDFActionGetGoToRemoteInNewWindow(PDFDocHandle Doc, PDFActionHandle Action);

    /* Launch Action */


    /* This structure used by PDFActionNewLaunch to create new
       Launch action                                           */
    typedef struct PDFLaunch {
        PDFString   FileName;           /* The application to be launched or the document to be opened or 
                                        printed. If this entry is absent and the viewer application does not 
                                        understand any of the alternative entries, it should do nothing.     */
        PDFString   DefaultDir;         /* A string specifying the default directory in standard DOS syntax. */
        PDFString   Operation;          /* A string specifying the operation to perform:
                                        <table>
                                        open        Open a document.
                                        print       Print a document.
                                        </table>                                                             */
        PDFString   Params;             /* A parameter string to be passed to the application.               */
    }PDFLaunch, *PDFLaunchP;


    /********************************************************************
    Description:
        Creates new "Launch" action. 
    Parameters:
        Doc         - [in]  PDF Document handle
        Launch      - [in]  Point to a <b>PDFLaunch</b> structure that defines 
                            the characteristics of the launch action. 
        InNewWindow - [in]  Specifying whether to open the destination document
                            in a new window. If this flag is false, the destination document will
                            replace the current document in the same window.
    Returns: 
        The return value is a handle to a PDF action.
    *********************************************************************/
    PDFActionHandle PDFActionNewLaunch(PDFDocHandle Doc, PDFLaunch Launch, ppBool InNewWindow);



    /* Description
       \Returns information that defines the characteristics of the
       launch action.
       Parameters
       Doc :          [in] PDF Document handle
       Action :       [in] Handle of the PDF action.
       Launch :       [out] Point to a <b>PDFLaunch</b> structure where the
                      characteristics of the launch action will be stored.
       IsNewWindow :  [out] If the launched application is opened in new
                      window, the value will be true.
       IsWinFormat :  [out] If information is stored for windows platform
                      the value will be true.
       Note
       Adobe Acrobat (r) supports only windows platform information.
       Returns
       None.                                                                */
    void PDFActionGetLaunch(PDFDocHandle Doc, PDFActionHandle Action, PDFLaunchP Launch,
        ppBool *InNewWindow, ppBool *IsWinFormat);

    /* Thread Action */

    /* This enumeration defines type of the store thread info in
       thread action                                             */
    typedef enum PDFThreadActionType{
        taThreadHandle,                 /* Destination is stored in PDFThreadHandle */
        taThreadIndex,                  /* Destination is stored in index of the threads in PDF
                                           document. The first thread in the document has index 0. */
        taThreadTitle                   /* Destination is stored in the title of the thread. */
    } PDFThreadActionType;

    typedef enum PDFBeadActionType{
        taBeadHandle,                   /* Destination stored in PDFBeadHandle */
        taBeadIndex                     /* Destination stored in index of the bead within its thread. 
                                            The first bead in a thread has index 0. */
    } PDFBeadActionType;

    /* This structure used by PDFActionNewThread to create new
       Thread action                                           */
    typedef struct PDFThreadActionParam {
        PDFThreadActionType DestThreadType; /* Depends type of the thread */
        union {
            PDFThreadHandle ThreadHandle;
            ppInt32         ThreadIndex;
            PDFString       ThreadTitle;
        } DestThread;                       /* The desired destination thread */
        PDFBeadActionType   DestBeadType;   /* Depends type of the bead */
        union {
            PDFBeadHandle   BeadHandle;
            ppInt32         BeadIndex;
        } DestBead;                         /* The desired bead in the destination thread */
    }PDFThreadActionParam, *PDFThreadActionParamP;


    /* Description
       Creates new "Thread" action.
       Parameters
       Doc :             [in] PDF Document handle
       Launch :          [in] Point to a <b>PDFThreadActionParam</b>
                         structure that defines the characteristics of the
                         thread action.
       FileName :        [in] Filename of the PDF documents where destination
                         thread may be the desired. 
       FileNameLength :  [in] Length of the filename.
       Note
       Destination thread is in current PDF document if filename is
       NULL. In other case PDFBeadHandle or PDFThreadHandle
       impossible to use in <b>PDFThreadActionParam</b> structure.
       Returns
       The return value is a handle to a PDF action.                          */
    PDFActionHandle PDFActionNewThread(PDFDocHandle Doc, char *FileName, ppInt32 FileNameLength,
                                       PDFThreadActionParam Thread);
    
    
    /* Description
       \Returns information that defines the characteristics of the
       thread action.
       Parameters
       Doc :       [in] PDF Document handle
       Action :    [in] Handle of the PDF action.
       FileName :  [out] Name of the PDF document where destination
                   thread must be desired.
       Thread :    [out] Point to a <b>PDFThreadActionParam</b> structure
                   where the characteristics of the thread action will be
                   stored.
       Returns
       None.                                                              */
    void PDFActionGetThread(PDFDocHandle Doc, PDFActionHandle Action, PDFString *FileName,
                            PDFThreadActionParamP Thread);




    /* URI Action */

    /********************************************************************
    Description:
        Creates new "URI" action. 
    Parameters:
        Doc         - [in]  PDF Document handle.
        URI         - [in]  The uniform resource identifier to resolve, encoded in 7-bit ASCII.
        Length      - [in]  Length of the URI.
        IsMap       - [in]  A flag specifying whether to track the mouse 
                            position when the URI is resolved.
    Returns: 
        The return value is a handle to a PDF action.
    *********************************************************************/
    PDFActionHandle PDFActionNewURI(PDFDocHandle Doc, char *URI, ppInt32 Length, ppBool IsMap);




    /* Description
       \Returns information that defines the characteristics of the
       "URI" action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the PDF action.
       URI :     [out] The uniform resource identifier to resolve.
       IsMap :   [out] If tracking the mouse position, the value will
                 be true.
       Returns
       None.                                                          */
    void PDFActionGetURI(PDFDocHandle Doc, PDFActionHandle Action, PDFString *URI, ppBool *IsMap);



    /* Sound Action */

    /* Description
       Creates new "Sound" action.
       Parameters
       Doc :        [in] PDF Document handle.
       Sound :      [in] PDF Annotation with sound information
       Volume :     [in] The volume at which to play the sound, in the
                    range from \-1.0 to 1.0. Higher values denote greater
                    volume; negative values mute the sound.
       Synch :      [in] A flag specifying whether to play the sound
                    synchronously or asynchronously.
       Repeating :  [in] A flag specifying whether to repeat the sound
                    indefinitely.
       Mix :        [in] A flag specifying whether to mix this sound with
                    any other sound already playing.
       Returns
       The return value is a handle to a PDF action.                      */

    PDFActionHandle PDFActionNewSound(PDFDocHandle Doc, PDFSoundHandle Sound, ppReal Volume,
        ppBool Synch, ppBool Repeating, ppBool Mix);


    /* Description
       \Returns information that defines the characteristics of the
       sound action.
       Parameters
       Doc :        [in] PDF Document handle
       Action :     [in] Handle of the PDF action.
       Volume :     [out] The volume at which to play the sound, in the
                    range from \-1.0 to 1.0. Higher values denote greater
                    volume; negative values mute the sound.
       Synch :      [out] A flag specifying whether to play the sound
                    synchronously or asynchronously.
       Repeating :  [out] A flag specifying whether to repeat the sound
                    indefinitely.
       Mix :        [our] A flag specifying whether to mix this sound with
                    any other sound already playing.
       Returns
       The return value is sound annotation handle.                        */
    PDFSoundHandle PDFActionGetSound(PDFDocHandle Doc, PDFActionHandle Action, ppReal *Volume,
        ppBool *Synch, ppBool *Repeating, ppBool *Mix);




    /* Movie Action */



    /* This enumeration defined type of the store annotation info in action */
    typedef enum {
        acAnnotationHandle,     /* Store annotation handle in action.*/
        acAnnotationName        /* Store annotation name in action. */
    } PDFAnnotationIdentifyType;




    /* Structure used to store information about annotation if PDF
       action                                                      */
    typedef struct PDFAnnatationIdentify {
        PDFAnnotationIdentifyType   Type;           /* Type of the information */
        union {
            PDFString           AnnotationName;     
            PDFAnnotationHandle AnnotationHandle;   
        } Annotation;                       /* Annotation identifier */
    } PDFAnnatationIdentify, *PDFAnnatationIdentifyP;




    /* This enumeration defined types of the action for movie */
    typedef enum PDFMovieActionOperation {
        moPlay,                     /* Start playing the movie.     */
        moStop,                     /* Stop playing the movie.      */
        moPause,                    /* Pause a playing movie.       */
        moResume                    /* Resume a paused movie.       */
    } PDFMovieActionOperation;





    /********************************************************************
    Description:
        Creates new "Movie" action. 
    Parameters:
        Doc         - [in]  PDF Document handle.
        Movie       - [in]  PDF Annotation with movie information
        Operation   - [in]  The operation to be performed on the movie.
    Returns: 
        The return value is a handle to a PDF action.
    *********************************************************************/
    PDFActionHandle PDFActionNewMovie(PDFDocHandle Doc, PDFAnnotationHandle Movie,
                                      PDFMovieActionOperation Operation);




    /********************************************************************
    Description:
        Creates new "Movie" action. 
    Parameters:
        Doc         - [in]  PDF Document handle.
        String      - [in]  PDF Annotation name with movie information
        Length      - [in]  Annotation name length.
        Operation   - [in]  The operation to be performed on the movie.
    Returns: 
        The return value is a handle to a PDF action.
    *********************************************************************/
    PDFActionHandle PDFActionNewMovieName(PDFDocHandle Doc, char *String, ppInt32 Length,
                                          PDFMovieActionOperation Operation);



    /* Description
       \Returns information that defines the characteristics of the
       movie action.
       Parameters
       Doc :        [in] PDF Document handle
       Action :     [in] Handle of the PDF action.
       Param :      [out] Pointer to description of the movie
                    annotation
       Operation :  [out] Returns type of the performed operation on
                    the movie.
       Returns
       None.                                                         */
    void PDFActionGetMovie(PDFDocHandle Doc, PDFActionHandle Action, PDFAnnatationIdentifyP Param,
                           PDFMovieActionOperation *Operation);

    /* Hide Action */

    /* Description
       Creates new "Hide" action.
       Parameters
       Doc :     [in] PDF Document handle.
       IsHide :  [in] Type of the action. Execution of this action will
                 hide selected annotations if value sets in "true". In
                 other case selected annotations will be shown.
       Returns
       The return value is a handle to a PDF action.                    */
    PDFActionHandle PDFActionNewHide(PDFDocHandle Doc, ppBool IsHide);




    /* Description
       Appends annotation to list in the hide action.
       Parameters
       Doc :         [in] PDF Document handle.
       Action :      [in] Handle of the PDF action.
       Annotation :  [in] Handle of the annotation which is needed to
                     be appended to list.
       Returns
       None.
       Note
       Operation will be performed for all annotations in the PDF
       document if any annotation for this action is not selected.    */
    void PDFActionHideAddAnnotation(PDFDocHandle Doc, PDFActionHandle Action,
                                    PDFAnnotationHandle Annotation);




    /* Description
       Appends annotation to list in the hide action.
       Parameters
       Doc :             [in] PDF Document handle.
       Action :          [in] Handle of the PDF action.
       AnnotationName :  [in] Name of the annotation which is needed to be
                         appended to the list.
       Length :          [in] Length of the name.
       Returns
       None.
       Note
       Operation will be performed for all annotations in the PDF
       document if any annotation for this action is not selected.         */
    void PDFActionHideAddAnnotationName(PDFDocHandle Doc, PDFActionHandle Action,
                                        char *AnnotationName, ppInt32 Length);



    /* Description
       \Returns count of the annotation which will be used by this
       action.
       Parameters
       Doc :     [in] PDF Document handle.
       Action :  [in] Handle of the PDF action.
       Returns
       Count of the used annotations.
       Note
       If result is zero all annotations will be used.             */
    ppInt32 PDFActionGetHideCount(PDFDocHandle Doc, PDFActionHandle Action);




    /* Description
       \Returns information about annotation which will be used by
       this action.
       Parameters
       Doc :         [in] PDF Document handle.
       Action :      [in] Handle of the PDF action.
       Index :       [in] Index of the action in the list, referenced
                     by a 0\-based index.
       Annotation :  [in] Information about annotation
       Returns
       None.                                                          */
    void PDFActionGetHideItem(PDFDocHandle Doc, PDFActionHandle Action, ppInt32 Index,
                              PDFAnnatationIdentifyP Annotation);




    /* Description
       \Returns operation which will be executed by this action.
       Parameters
       Doc :     [in] PDF Document handle.
       Action :  [in] Handle of the PDF action.
       Returns
       \Result indicating whether to hide the annotation (true) or
       show it (false).                                            */
    ppBool PDFActionGetHideIsHide(PDFDocHandle Doc, PDFActionHandle Action);



    /* Named Action */

    /* This enumeration defines type of the operation for named
       action                                                   */
    typedef enum {
        naNextPage,     /* Go to the next page of the document.     */
        naPrevPage,     /* Go to the previous page of the document. */  
        naFirstPage,    /* Go to the first page of the document.    */
        naLastPage      /* Go to the last page of the document.     */
    } PDFNamedActionType;


    /********************************************************************
    Description:
        Creates new "Named" action. 
    Parameters:
        Doc         - [in]  PDF Document handle.
        NamedType   - [in]  Operation for the named action.
    Returns: 
        The return value is a handle to a PDF action.
    *********************************************************************/
    PDFActionHandle PDFActionNewNamed(PDFDocHandle Doc, PDFNamedActionType NamedType);




    /* Description
       \Returns operation which will be executed by this action.
       Parameters
       Doc :        [in] PDF Document handle.
       Action :     [in] Handle of the PDF action.
       NamedType :  [out] Operation for the named action.
       Returns
       None.                                                     */
    void PDFActionGetNamed(PDFDocHandle Doc, PDFActionHandle Action, PDFNamedActionType *NamedType);




    /* Submit-Form Action */
#define PDF_SUBMIT_FORM_FLAG_EXCLUDE                        1
#define PDF_SUBMIT_FORM_FLAG_INCLUDE_NO_VALUE_FIELDS        2
#define PDF_SUBMIT_FORM_FLAG_EXPORT_FORMAT                  4
#define PDF_SUBMIT_FORM_FLAG_GET_METHOD                     8
#define PDF_SUBMIT_FORM_FLAG_SUBMIT_COORDINATES             16
#define PDF_SUBMIT_FORM_FLAG_XML                            32
#define PDF_SUBMIT_FORM_FLAG_INCLUDE_ANNOTATIONS            128
#define PDF_SUBMIT_FORM_FLAG_SUBMIT_PDF                     256



    /********************************************************************
    Description:
        Creates new "SubmitForm" action. 
    Parameters:
        Doc         - [in]  PDF Document handle.
        URI         - [in]  A URL file specification giving the uniform resource locater of the script
                            at the Web server that will process the submission.
        Length      - [in]  Length of the URI string
        Flag        - [in]  A set of flags specifying various characteristics
                            of the action.
                            
    Returns: 
        The return value is a handle to a PDF action.
    Note: 
        Below you can find flag meaning table:
        <table>
        Flag                                            Meaning
        ----                                            -------
        PDF_SUBMIT_FORM_FLAG_EXCLUDE                    The list of the acroform objects exclude from the submission.
        PDF_SUBMIT_FORM_FLAG_INCLUDE_NO_VALUE_FIELDS    All acroform object will included in submission ( With empty values too )
        PDF_SUBMIT_FORM_FLAG_EXPORT_FORMAT              If this flag set, export will execute in HTML form format, else in FDF format
        PDF_SUBMIT_FORM_FLAG_GET_METHOD                 If this flag set, field names and values are submitted using an HTTP GET request; if clear, they are submitted using a POST request.
        PDF_SUBMIT_FORM_FLAG_SUBMIT_COORDINATES         If set, the coordinates of the mouse click that caused the submitform action are transmitted as part of the form data. The coordinate values are relative to the upper-left corner of the acroform object rectangle.
        PDF_SUBMIT_FORM_FLAG_XML                        If set, field names and values are submitted in XML format; if clear, they are submitted in HTML Form format or Forms Data Format (FDF), according to the value of the PDF_SUBMIT_FORM_FLAG_EXPORT_FORMAT flag.
        PDF_SUBMIT_FORM_FLAG_SUBMIT_PDF                 If set, the document is submitted in PDF format, using the MIME content type application/pdf (described in Internet RFC 2045, Multipurpose Internet Mail Extensions (MIME), Part One: Format of Internet Message Bodies; see the Bibliography). If this flag is set, all other flags are ignored except PDF_SUBMIT_FORM_FLAG_GET_METHOD.
        </table>

    *********************************************************************/
    PDFActionHandle PDFActionNewSubmitForm(PDFDocHandle Doc, char *URI, ppInt32 Length,
        ppInt32 Flags);


    /* Description
       Appends annotation to list in the submitform action.
       Parameters
       Doc :         [in] PDF Document handle.
       Action :      [in] Handle of the PDF action.
       Annotation :  [in] Handle of the annotation which need append
                     to list.
       Returns
       None.
       Note
       Operation will be performed for all acroform object in the
       PDF document (flag PDF_SUBMIT_FORM_FLAG_EXCLUDE not used ) if
       its not selected any annotation for this action.              */
    void PDFActionSubmitFormAddAnnotation(PDFDocHandle Doc, PDFActionHandle Action,
        PDFAnnotationHandle Annotation);



    /* Description
       Appends annotation to list in the submitform action.
       Parameters
       Doc :             [in] PDF Document handle.
       Action :          [in] Handle of the PDF action.
       AnnotationName :  [in] Name of the annotation which is needed to be
                         appended to the list.
       Length :          [in] Length of the name.
       Returns
       None.
       Note
       Operation will be performed for all acroform object in the
       PDF document (flag PDF_SUBMIT_FORM_FLAG_EXCLUDE not used ) if
       its not selected any annotation for this action.                    */
    void PDFActionSubmitFormAddAnnotationName(PDFDocHandle Doc, PDFActionHandle Action,
        char *AnnotationName, ppInt32 Length);



    /* Description
       \Returns count of the annotation which will be used by this
       action.
       Parameters
       Doc :     [in] PDF Document handle.
       Action :  [in] Handle of the PDF action.
       Returns
       Count of the used annotations.
       Note
       If result is zero, operation will be performed for all
       acroform object in the PDF document, flag
       PDF_SUBMIT_FORM_FLAG_EXCLUDE is not used.                   */
    ppInt32 PDFActionGetSubmitFormCount(PDFDocHandle Doc, PDFActionHandle Action);



    /* Description
       \Returns information about annotation which will be used by
       this action.
       Parameters
       Doc :         [in] PDF Document handle.
       Action :      [in] Handle of the PDF action.
       Index :       [in] Index of the action in the list, referenced
                     by a 0\-based index.
       Annotation :  [in] Information about annotation
       Returns
       None.                                                          */
    void PDFActionGetSubmitFormItem(PDFDocHandle Doc, PDFActionHandle Action, ppInt32 Index,
        PDFAnnatationIdentifyP Annotation);


    /********************************************************************
    Description:
        Returns information about submitform action.  
    Parameters:
        Doc         - [in]  PDF Document handle.
        Action      - [in]  Handle of the PDF action.
        URI         - [out] A URL file specification giving the uniform resource locater of the script
                            at the Web server that will process the submission
        Flag        - [out] A set of flags specifying various characteristics
                            of the action.
    Returns: 
        None.
    *********************************************************************/
    void PDFActionGetSubmitForm(PDFDocHandle Doc, PDFActionHandle Action, PDFString *URI,
        ppInt32 *Flags);

    /* Reset-Form Action */

    /* Description
       Creates new "Reset" action.
       Parameters
       Doc :      [in] PDF Document handle.
       Exclude :  [in] If false, the list specifies which fields to reset.
                  If true, the list of the acroform objects informs which
                  fields to be excluded from resetting; all fields in the
                  documents interactive form are reset excepting those
                  listed.
       Returns
       The return value is a handle to a PDF action.                       */
    PDFActionHandle PDFActionNewResetForm(PDFDocHandle Doc, ppBool Exclude);



    /* Description
       Appends annotation to list in the resetform action.
       Parameters
       Doc :         [in] PDF Document handle.
       Action :      [in] Handle of the PDF action.
       Annotation :  [in] Handle of the annotation which is needed to
                     be appended to list.
       Returns
       None.
       Note
       Reset action will be performed for all annotations in the PDF
       document if its not selected any annotation for this action.   */
    void PDFActionResetFormAddAnnotation(PDFDocHandle Doc, PDFActionHandle Action,
        PDFAnnotationHandle Annotation);



    /* Description
       Appends annotation to list in the resetform action.
       Parameters
       Doc :             [in] PDF Document handle.
       Action :          [in] Handle of the PDF action.
       AnnotationName :  [in] Name of the annotation which is needed to be
                         appended to the list.
       Length :          [in] Length of the name.
       Returns
       None.
       Note
       Reset action will be performed for all annotations in the PDF
       document if its not selected any annotation for this action.        */
    void PDFActionResetFormAddAnnotationName(PDFDocHandle Doc, PDFActionHandle Action,
        char *AnnotationName, ppInt32 Length);


    /* Description
       \Returns count of the annotation which will be used by this
       action.
       Parameters
       Doc :     [in] PDF Document handle.
       Action :  [in] Handle of the PDF action.
       Returns
       Count of the used annotations.
       Note
       If result is zero all acroform objects will be reset.       */
    ppInt32 PDFActionGetResetFormCount(PDFDocHandle Doc, PDFActionHandle Action);




    /* Description
       \Returns information about annotation which will be used by
       this action.
       Parameters
       Doc :         [in] PDF Document handle.
       Action :      [in] Handle of the PDF action.
       Index :       [in] Index of the action in the list, referenced
                     by a 0\-based index.
       Annotation :  [in] Information about annotation
       Returns
       None.                                                          */
    void PDFActionGetResetFormItem(PDFDocHandle Doc, PDFAnnotationHandle Action, ppInt32 Index,
        PDFAnnatationIdentifyP Annotation);



    /* Description
       \Returns information which operation will be performed with
       presenting acroform object in the list.
       Parameters
       Doc :      [in] PDF Document handle.
       Action :   [in] Handle of the PDF action.
       Exclude :  [out] If false, the list specifies which fields to be
                  reset. If true, the list of the acroform objects informs
                  which fields to be excluded from resetting; all fields
                  in the documents interactive form are reset excepting
                  those listed.
       Returns
       None.
       Note
       If list is empty, all acroform objects will be reset.               */
    void PDFActionGetResetForm(PDFDocHandle Doc, PDFActionHandle Action, ppBool *Exclude);
        

    /* Import Data Action */

    /********************************************************************
    Description:
        Creates new "Import Data" action. 
    Parameters:
        Doc         - [in]  PDF Document handle.
        FileName    - [in]  The FDF filename from which to import the data.
        Length      - [in]  The length of the filename.
    Returns: 
        The return value is a handle to a PDF action.
    *********************************************************************/
    PDFActionHandle PDFActionNewImportData(PDFDocHandle Doc, char *FileName, ppInt32 Length);




    /* Description
       \Returns information about filename from which will be
       imported the data.
       Parameters
       Doc :       [in] PDF Document handle.
       Action :    [in] Handle of the PDF action.
       FileName :  [out] The FDF filename from which data will be imported.
       Returns
       None.
       Note
       \ \                                                                  */
    void PDFActionGetImportData(PDFDocHandle Doc, PDFActionHandle Action, PDFString *FileName);


    /* JavaScript Action */

    /* Description
       Creates new "JavaScript" action from string.
       Parameters
       Doc :         [in] PDF Document handle
       JavaScript :  [in] JavaScript string which will be executed
       Length :      [in] Length of javascript string.
       Returns
       The return value is a handle to a PDF action.               */
    PDFActionHandle PDFActionNewJavaScript(PDFDocHandle Doc, char *JavaScript, ppInt32 Length);

    /* Description
       Creates new "JavaScript" action from CosStream where this
       javascript is stored.
       Parameters
       Doc :         [in] PDF Document handle
       JavaScript :  [in] Cos Stream where this JavaScript is stored
       Returns
       The return value is a handle to a PDF action.                 */
    PDFActionHandle PDFActionNewJavaScriptStream(PDFDocHandle Doc, PDFCosHandle JavaScript);



    /* Description
       \Returns JavaScript storage either in CosStream or in string.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the PDF action.
       Returns
       If the javascript is stored in CosStream, the return value
       will be nonzero. If the javascript is stored in string, the
       \return value will be zero.                                   */
    ppBool PDFActionGetJavaScriptIsHandle(PDFDocHandle Doc, PDFActionHandle Action);



    /* Description
       \Returns CosStream where javascript is stored for this
       action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the PDF action.
       Returns
       CosStream handle with javascript                       */
    PDFCosHandle PDFActionGetJavaScriptHandle(PDFDocHandle Doc, PDFActionHandle Action);



    /* Description
       \Returns string where stored javascript for this action.
       Parameters
       Doc :     [in] PDF Document handle
       Action :  [in] Handle of the PDF action.
       Length :  [out] Pointer to integer where will be stored size
                 of the javascript string.
       Returns
       Pointer to string with javascript.                           */
    char *PDFActionGetJavaScriptString(PDFDocHandle Doc, PDFActionHandle Action, ppInt32 *Length);


#ifdef __cplusplus
}
#endif

#endif /* #ifndef VSActionA_H */

