/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSAcroObjects.h
*********************************************************************/

#ifndef _H_VSACROOBJECTS
#define _H_VSACROOBJECTS /* Acro Form Interactive Objects Module */

#include "time.h"

#include "VSTypes.h"
#include "VSFontA.h"
#include "VSCanvasA.h"
#include "VSActionA.h"

/* Acro Form Object Type. Interactive Control Type. */

typedef enum {
    PDFAcroTypeUnknown = 0,     /*  Unknown Type, in failure case                           */
    PDFAcroTypePushButton,      /*  Button for select single action                         */
    PDFAcroTypeCheckBox,        /*  Button for check single item                            */
    PDFAcroTypeRadioButton,     /*  Button from group for select only one item from ensemble*/
    PDFAcroTypeEditBox,         /*  Variable text edit field for change text item           */
    PDFAcroTypeComboBox,        /*  Field for select one text item from list                */
    PDFAcroTypeListBox,         /*  Box for select item(s) from list                        */
    PDFAcroTypeSignature        /*  Field for sign in document, maybe invisible             */
} TPDFAcroType; 

/* Action Event Type for Acro Form Objects. Set of events. */
typedef enum {
    PDFAcroEventTypeActivate = 0,   /* event on activate, primary action      */
    PDFAcroEventTypeEnter,          /* event on enter in the active area      */
    PDFAcroEventTypeExit,           /* event on exit from the active area     */
    PDFAcroEventTypePress,          /* event on press mouse button inside it  */
    PDFAcroEventTypeRelease,        /* event on release mouse button inside   */
    PDFAcroEventTypeFocusOn,        /* event on receive the input focus       */
    PDFAcroEventTypeFocusOff,       /* event on lose the input focus          */
    PDFAcroEventTypeKeystroke,      /* event on change text value in field    */
    PDFAcroEventTypeFormat,         /* event on format value in the field     */
    PDFAcroEventTypeValidate,       /* event on change fields value in field */
    PDFAcroEventTypeCalculate       /* event on recalculate value             */
} TPDFAcroEventType;

/* Acro Form Object Field's Flags. Property of interactive object. */
#define PDFAcroFlags_Default                  0 /* Default flag without any features */
#define PDFAcroFlags_ReadOnly                 1 /* If set, the user may not change the value of the field.
                                                Any associated widget annotations will not interact with
                                                the user; that is, they will not respond to mouse clicks
                                                or change their appearance in response to mouse motions.
                                                This flag is useful for fields whose values are computed
                                                or imported from a database.*/
#define PDFAcroFlags_Required                 2 /* If set, the field must have a value at the time it is
                                                exported by a submit-form action */
#define PDFAcroFlags_NoExport                 4 /* If set, the field must not be exported by a submit-form
                                                action */
#define PDFAcroFlags_Multiline             4096 /* If set, the field may contain multiple lines of text;
                                                if clear, the fields text is restricted to a single line.*/
#define PDFAcroFlags_Password              8192 /* If set, the field is intended for entering a secure
                                                password that should not be echoed visibly to the screen.
                                                Characters typed from the keyboard should instead be echoed
                                                in some unreadable form, such as asterisks or bullet
                                                characters. To protect password confidentiality, viewer
                                                applications should never store the value of the text field
                                                in the PDF file if this flag is set. */
#define PDFAcroFlags_NoToggleToOff        16384 /* For Radio buttons only. If set, exactly one radio button
                                                must be selected at all times; clicking the currently
                                                selected button has no effect. If clear, clicking the
                                                selected button deselects it, leaving no button selected.*/
#define PDFAcroFlags_Radio                32768 /* If set, the field is a set of radio buttons; if clear,
                                                the field is a check box. This flag is meaningful only if
                                                the Pushbutton flag is clear. */
#define PDFAcroFlags_Pushbutton           65536 /* If set, the field is a pushbutton that does not retain
                                                a permanent value. */
#define PDFAcroFlags_Combo               131072 /* If set, the field is a combo box; if clear, the field is
                                                a list box. */
#define PDFAcroFlags_Edit                262144 /* If set, the combo box includes an editable text box as
                                                well as a drop list; if clear, it includes only a drop list.
                                                This flag is meaningful only if the Combo flag is set. */
#define PDFAcroFlags_Sort                524288 /* If set, the fields option items should be sorted
                                                alphabetically. This flag is intended for use by form
                                                authoring tools, not by PDF viewer applications; viewers
                                                should simply display the options in the order in which
                                                they occur in the items array */
#define PDFAcroFlags_FileSelect         1048576 /* If set, the text entered in the field represents
                                                the pathname of a file whose contents are to be submitted
                                                as the value of the field. */
#define PDFAcroFlags_MultiSelect        2097152 /* If set, more than one of the fields option items may
                                                be selected simultaneously; if clear, no more than one
                                                item at a time may be selected. */
#define PDFAcroFlags_DoNotSpellCheck    4194304 /* If set, the text entered in the field will not be
                                                spell-checked. */
#define PDFAcroFlags_DoNotScroll        8388608 /* If set, the field will not scroll (horizontally for 
                                                single-line fields, vertically for multiple-line fields)
                                                to accommodate more text than will fit within its annotation
                                                rectangle. Once the field is full, no further text will be
                                                accepted. */
#define PDFAcroFlags_Comb              16777216 /* Meaningful only if the MaxLen entry is present in the
                                                text field dictionary and if the Multiline, Password, and
                                                FileSelect flags are clear. If set, the field is
                                                automatically divided up into as many equally spaced
                                                positions, or combs, as the value of MaxLen, and the text
                                                is laid out into those combs. */
#define PDFAcroFlags_RadiosInUnison    33554432 /* If set, a group of radio buttons within a radio button
                                                field that use the same value for the on state will turn on
                                                and off in unison; that is if one is checked, they are all
                                                checked. If clear, the buttons are mutually exclusive (the
                                                same behavior as HTML radio buttons). */
#define PDFAcroFlags_CommitOnSelChange 67108864 /* If set, the new value is committed as soon as
                                                a selection is made with the pointing device. This allows
                                                applications to perform an action once a selection is made,
                                                without requiring the user to exit the field. If clear,
                                                the new value is not committed until the user exits the
                                                field. */
#define PDFAcroFlags_RichText         134217728 /* If set, the value of this field should be represented
                                                as a rich text string. If the field has a value, the RV
                                                entry of the field dictionary specifies the rich text
                                                string. */
typedef ppUns32 TPDFAcroFlags;  /* Acro Object's set of Field's Flags */
typedef TPDFAcroFlags *PPDFAcroFlags; /* Pointer to Field's Flags */

/* Acro Form Object Annotation's Flags. Property of controls. */
#define PDFAnnotFlags_Default         0 /* Default flag without any features */
#define PDFAnnotFlags_Invisible       1 /* If set, do not display the annotation if it does not belong
                                        to one of the standard annotation types and no annotation
                                        handler is available. If clear, display such an unknown
                                        annotation using an appearance stream specified by its 
                                        appearance dictionary, if any */
#define PDFAnnotFlags_Hidden          2 /* If set, do not display or print the annotation or allow it
                                        to interact with the user, regardless of its annotation type or
                                        whether an annotation handler is available. In cases where
                                        screen space is limited, the ability to hide and show annotations
                                        selectively can be used in combination with appearance streams
                                        to display auxiliary pop-up information similar in function to
                                        online help systems. */
#define PDFAnnotFlags_Print           4 /* If set, print the annotation when the page is printed. If clear,
                                        never print the annotation, regardless of whether it is displayed
                                        on the screen. This can be useful, for example, for annotations
                                        representing interactive pushbuttons, which would serve no
                                        meaningful purpose on the printed page. */
#define PDFAnnotFlags_NoZoom          8 /* If set, do not scale the annotations appearance to match the
                                        magnification of the page. The location of the annotation on the
                                        page (defined by the upper-left corner of its annotation rectangle)
                                        remains fixed, regardless of the page magnification. See below for
                                        further discussion. */
#define PDFAnnotFlags_NoRotate       16 /* If set, do not rotate the annotations appearance to match the
                                        rotation of the page. The upper-left corner of the annotation
                                        rectangle remains in a fixed location on the page, regardless of
                                        the page rotation. See below for further discussion. */
#define PDFAnnotFlags_NoView         32 /* If set, do not display the annotation on the screen or allow it
                                        to interact with the user. The annotation may be printed (depending
                                        on the setting of the Print flag), but should be considered hidden
                                        for purposes of on-screen display and user interaction. */
#define PDFAnnotFlags_ReadOnly       64 /* If set, do not allow the annotation to interact with the user.
                                        The annotation may be displayed or printed (depending on the
                                        settings of the NoView and Print flags), but should not respond to
                                        mouse clicks or change its appearance in response to mouse motions.*/
#define PDFAnnotFlags_Locked        128 /* If set, do not allow the annotation to be deleted or its
                                        properties (including position and size) to be modified by the user.
                                        However, this does not restrict changes to the annotations contents,
                                        such as a form field value. */
#define PDFAnnotFlags_ToggleNoView  256 /* If set, invert the interpretation of the NoView flag for certain
                                        events. A typical use is to have an annotation that appears only
                                        when a mouse cursor is held over it. */
typedef ppUns16 TPDFAnnotFlags;  /* Acro Object's set of Annotation's Flags */
typedef TPDFAnnotFlags *PPDFAnnotFlags;  /* Pointer to Annotation's Flags */

/* Acro Form Object Quadding Type. Text justification style. */
typedef enum {
    PDFAcroQuaddingLeftTop = 0, /* attach text to left top corner of field */
    PDFAcroQuaddingTop,         /* attach text to top central site of field */
    PDFAcroQuaddingRightTop,    /* attach text to right top corner of field */
    PDFAcroQuaddingLeft,        /* attach text to left central site of field */
    PDFAcroQuaddingCenter,      /* attach text to center of field */
    PDFAcroQuaddingRight,       /* attach text to right central site of field */
    PDFAcroQuaddingLeftBottom,  /* attach text to left bottom corner of field */
    PDFAcroQuaddingBottom,      /* attach text to bottom central site of field */
    PDFAcroQuaddingRightBottom  /* attach text to right bottom corner of field */
} TPDFAcroQuadding;

/* Acro Form Object Appearance Type. For point to appearance type. */
typedef enum {
    PDFAcroNormalAppearance = 0,    /* The normal appearance is used when the annotation
                                    is not interacting with the user. This is also the 
                                    appearance that is used for printing the annotation. */
    PDFAcroRolloverAppearance,      /* The rollover appearance is used when the user moves
                                    the cursor into the annotations active area without
                                    pressing the mouse button. */
    PDFAcroDownAppearance           /* The down appearance is used when the mouse button is
                                    pressed or held down within the annotations active area. */
} TPDFAcroAppearance;

/* Acro Form Signature's Flags. Document Digital Signature's properties. */
#define PDFAcroSigFlags_Default         0   /* Default flag - document without any signatures */
#define PDFAcroSigFlags_SignaturesExist 1   /* If set, the document contains at least one signature
                                            field. This flag allows a viewer application to enable
                                            user interface items (such as menu items or pushbuttons)
                                            related to signature processing without having to scan
                                            the entire document for the presence of signature fields.*/
#define PDFAcroSigFlags_AppendOnly      2   /* If set, the document contains signatures that may be
                                            invalidated if the file is saved (written) in a way that
                                            alters its previous contents, such as with the optimize
                                            option. Merely updating the file by appending new
                                            information to the end of the previous version is safe.
                                            Viewer applications can use this flag to present a user
                                            requesting an optimized save with an additional alert box
                                            warning that signatures will be invalidated and requiring
                                            explicit confirmation before continuing with the operation.*/
typedef ppUns8 TPDFAcroSigFlags; /* Digital Signature's set of Flags */
typedef TPDFAcroSigFlags *PPDFAcroSigFlags; /* Pointer to Siggnature Flags */

/* Macro Definitions of NOCOLOR for PDF Color Type - Transparent, Invisible Color. */
#define NoPDFColor      0xDAB                           /* Nonexistent Color Code   */
#define SetNoPDFColor   ( ULGrayToColor( NoPDFColor ))  /* Transparent PDFColor     */
#define IsNoPDFColor(c) ( c.Color.Gray == NoPDFColor )  /* Check on invisibility    */

typedef void ( * PDFPaintContent ) ( PBXHandle PaintBox, void * Data ); /* Functional Type for Repaint Acroform
                                                                        Objects Routines - reload drawing */

/* Font Index Structure - to support early PDF versions ( 1.2
   and below )                                                */
typedef struct {
    ppBool IsStdFont;                   /* Boolean flag an accessories to 14 standard fonts */
    union {                             /* Font source */
        ppInt32           Index;        /* Index of loaded font in Document */
        TPDFStdandardFont StandardFont; /* If standard font then it must be named, see TPDFStdandardFont */
    } From;                             /* Font source */
} TPDFFontID, *PPDFFontID;

/* Font Type. Specifies text font properties of the annotation on the page.
( Acroform Object Characteristic ) */
typedef struct {
    TPDFFontID  ID;     /* Identifier of font type, see TPDFFontID      */
    ppReal      Size;   /* Size of font in points                       */
    TPDFColor   Color;  /* Color of font for displaying, see TPDFColor  */
} TPDFFont, *PPDFFont;

/* Border Type. Specifies position of the annotation on the page. ( Acroform Object Characteristic )*/
typedef struct {
    TPDFRect    Rect;           /* Rectangle region for active area of Acro Form Object.
                                Four coordinates - left, top, right and bottom of border.
                                See TPDFRect */
    TPDFColor   BorderColor;    /* Color of border for displaying, see TPDFColor */
    TPDFColor   FillColor;      /* Color for filling inside area, background color */
    ppReal      Width;          /* Width of border line in points */
} TPDFBorder, *PPDFBorder;

/* Determination of Text object for setting on Page or Acroform Content */
typedef struct {
    char           *Caption;        /* Text string of label */
    TPDFFont        Font;           /* Font for displaying text, see TPDFFont */
    TPDFBorder      Border;         /* Border of text label, see TPDFBorder */
    TPDFAcroQuadding Align;         /* Alignment text label option, see TPDFAcroQuadding */
    ppReal          Orientation;    /* Incline level, angle in degrees */
} TPDFTextBox, *PPDFTextBox;

/* Determination of Variable Text object for setting on Acroform Content. For text entering from document.*/
typedef struct {
    char           *Name;       /* Name(text string) of Acroform object, Name (text string) of
                                   Acroform field is used to export when the PDF document is
                                   submitted.                                                  */
    ppUns32         Flag;       /* Specify the behavior of the annotation when is printed,
                                   rotated etc. AcroField Type ( see TPDFAcroFlags )       */
    char           *Caption;    /* Default text string for appearance in EditBox. Text is
                                   displayed in EditBox when control is created.          */
    TPDFFont        Font;       /* Text font for appearance text label. Attributes of text
                                written on or in the control. */
    TPDFBorder      Border;     /* EditBox rectangle specifies position of the annotation
                                on the page. Border width and colors. */
    ppUns32         MaxLen;     /* The maximum length of the field's text, in characters. */
    TPDFAcroQuadding Align;     /* Text alignment in edit box, justification of input text. */
    PDFPaintContent PaintContent; /* Pointer to overload function to repaint edit box, instead of
                                     default ( optional )                                         */
} TPDFEditBox, *PPDFEditBox;

/* Determination of Pushbutton object for setting on Acroform Content. Action selection. Submit action. */
typedef struct {
    char           *Name;               /* Name(text string) of Acroform object, Name (text string) of
                                           Acroform field is used to export when the PDF document is
                                           submitted.                                                  */
    ppUns32         Flag;               /* Specify the behavior of the annotation when is printed,
                                           rotated etc. AcroField Type ( see TPDFAcroFlags )       */
    char           *Caption;            /* Specifies a text string that identifies the control to
                                        the user. Text label for appearance on button. */
    TPDFFont        Font;               /* Text font for appearance text label. Attributes of text
                                        written on or in the control. */
    TPDFBorder      Border;             /* Pushbutton rectangle specifies position of the
                                        annotation on the page. Border width and colors. */
    ppReal          Miter;              /* Miter of pushbutton, bevel size. */
    PDFPaintContent PaintContentUp;     /* Pointer to overload function to repaint pushbutton in normal
                                           state ( optional ), instead of default appearance.           */
    PDFPaintContent PaintContentDown;   /* Pointer to overload function to repaint pushbutton in pressed
                                           state ( optional ), instead of default appearance             */
} TPDFPushButton, *PPDFPushButton;

/* Type of CheckBox Style. */
typedef enum {
    cbfRectangle = 0,           /* Rectangle style */
    cbfCircle                   /* Circle style */
} TPDFCheckBoxStyle;

/* Type of CheckBox Mark - code of Zapf Dingbats sign ( 32 - 255 ). Here is definition in octal system */
#define cbsFinger       0053 /* 'Finger' mark */
#define cbsPen          0062 /* 'Pen' mark */
#define cbsVmark        0063 /* V-style mark */
#define cbsNike         0064 /* V-style mark */
#define cbsCross        0065 /* Cross-style mark */
#define cbsX            0066 /* Cross-style mark */
#define cbsCheck        0070 /* Cross-style mark */
#define cbsPlus         0072 /* Cross-style mark */
#define cbsDiamond      0106 /* Rhombus-style mark */
#define cbsStar         0110 /* 'Star' mark */
#define cbsFlower       0137 /* 'Flower' mark */
#define cbsSnow         0144 /* 'Snowflake' mark */
#define cbsCircle       0154 /* 'Circle' mark */
#define cbsRectangle    0156 /* 'Rectangle' mark */
#define cbsRhombus      0165 /* Rhombus-style mark */
typedef ppUns8 TPDFCheckBoxSign; /* Type of CheckBox Mark - code of Zapf Dingbats sign ( 32 - 255 ) */

/* Determination of CheckBox object and RadioButton object for setting on Acroform Content. Item selection. */
typedef struct {
    char           *Name;           /* Name(text string) of Acroform object, Name (text string) of
                                       Acroform field is used to export when the PDF document
                                       submitted.                                                  */
    ppUns32         Flag;           /* Specify the behavior of the annotation when is printed,
                                       rotated etc. AcroField Type ( see TPDFAcroFlags )       */
    TPDFFont        Font;           /* Text font for appearance text label. Attributes of
                                    text written on or in the control. Use only Font's color
                                    for displaying Mark character. */
    TPDFBorder      Border;         /* CheckBox or RadioButton rectangle specifies position
                                    of the annotation on the page. Border width and colors */
    TPDFCheckBoxStyle Style;        /* Style of CheckBox or RadioButton - rectangle or
                                    circle ( see TPDFCheckBoxStyle ). */
    TPDFCheckBoxSign Sign;          /* Code of Mark character in CheckBox or RadioButton
                                    ( see TPDFCheckBoxSign ). */
    ppBool          Value;          /* Value of CheckBox or RadioButton. Variable interactive
                                    value on Acroform ( see VSAcroForm.h ). */
    PDFPaintContent PaintContentOn; /* Pointer to overload function to repaint CheckBox in checked
                                       state ( optional ), instead of default appearance.          */
    PDFPaintContent PaintContentOff;/* Pointer to overload function to repaint CheckBox in unchecked
                                       state ( optional ), instead of default appearance.            */
} TPDFCheckBox, *PPDFCheckBox, TPDFRadioButton, *PPDFRadioButton;

/* Determination of ComboBox object and ListBox object for setting on Acroform Content.
Item(s) selection from items list. */
typedef struct {
    char           *Name;           /* Name(text string) of Acroform object, Name (text string)
                                    of Acroform field used for export when the PDF document
                                    submitted. */
    ppUns32         Flag;           /* Specify the behavior of the annotation when is printed,
                                       rotated etc. AcroField Type ( see TPDFAcroFlags )       */
    char           *Caption;        /* Default text string for appearance in ComboBox. Wasted in
                                       ListBox. Text is displayed in ComboBox when control is
                                       created.                                                  */
    TPDFFont        Font;           /* Text font for appearance text label. Attributes of
                                    text written on or in the control. */
    TPDFBorder      Border;         /* ComboBox or ListBox rectangle specifies position of
                                    the annotation on the page. Border width and colors. */
    PDFPaintContent PaintContent;   /* Pointer to overload function to repaint ComboBox ( optional
                                       ), instead of default.                                      */
} TPDFComboBox, *PPDFComboBox, TPDFListBox, *PPDFListBox, TPDFItemsBox;

/* Determination of Empty Signature object for setting on
   Acroform Content. Item to sign document                */
typedef struct {
    char           *Name;       /* Name(text string) of Acroform object, Name (text string) of
                                   Acroform field is used to export when the PDF document is
                                   submitted.                                                  */
    TPDFBorder      Border;     /* Signature specifies position of the annotation
                                on the page. */
    TPDFAcroSigFlags SigFlags;  /* A set of flags specifying various
                                document-level characteristics related to signature
                                fields. See TPDFAcroSigFlags */
    TPDFAnnotFlags  AnnotFlag;  /* Specify the behavior of the annotation when is printed,
                                   rotated etc. See TPDFAnnotFlags                         */
} TPDFSignature, *PPDFSignature;

/* Determination of Personal Invisible Signature object to sign
   Document                                                     */
typedef struct {
    char       *Name;       /* Name(text string) of Acroform object, Name (text string) of
                               Acroform field is used to export when the PDF document is
                               submitted.                                                  */
    char       *Owner;      /* Owner of Signature, Person Name. Text string,
                            for example "Ted Thompson" */
    char       *Reason;     /* Reason of Sign this document. Text string,
                            for example "I agree..." */
    ppBool      PKCS7;      /* Boolean flag of coding type :
                                    true    - 'Adobe.PPKMS' and 'adbe.pkcs7.sha1' crypt
                                            system sub filter
                                    false   - 'Adobe.PPKLite' and 'adbe.x509.rsa_sha1'
                                            crypt system sub filter */
    char       *FileName;   /* PFX Personal Signature FileName. Text string. */
    char       *Password;   /* Owner Password for Personal Signature. Text string. */
} TPDFDocumentSignature, *PPDFDocumentSignature;

/************************************************************************/
#ifdef __cplusplus
extern "C" {
#endif
/* Description
   Draw TextBox on PaintBox Content
   Parameters
   PaintBox :  [in] Created PaintBox for drawing
   TextBox :   [in] TextBox structure to create text object
   Returns
   None.                                                    */
void            PBXDrawTextBox( PBXHandle PaintBox, TPDFTextBox TextBox );

/************************************************************************
    Active Features API
************************************************************************/



/* Description
   Sets Digital Signature Blank Field on Page Content for
   further sign (filling)
   Parameters
   Doc :        [in] Destination Document
   PageIndex :  [in] Page index ( as from 0 ) where Digital Signature
                will be stated.
   Signature :  [in] Digital Signature Structure which describes all
                needed fields ( see TPDFSignature ).
   Returns
   Index of acroform object in document.                              */
ppInt32 PDFAcroSignatureInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                    TPDFSignature Signature );


/* Description
   Sets Invisible Filled Digital Signature on Page. This
   signature is in PFX Digital Signature File.
   Parameters
   Doc :                [in] Destination Document
   PageIndex :          [in] Page index ( as from 0 ) where Digital
                        Signature will be linked.
   DocumentSignature :  [in] Digital Signature Structure which describes
                        owner, reason and information about PFX file ( see
                        TPDFDocumentSignature ).
   Returns
   None.                                                                   */
void PDFAcroSignDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                          TPDFDocumentSignature DocumentSignature );



/* Description
   Sets EditBox on Page Content for text entering
   Parameters
   Doc :        [in] Destination Document.
   PageIndex :  [in] Page index ( as from 0 ) where EditBox will be
                stated.
   EditBox :    [in] EditBox Structure which describes all needed
                fields ( see TPDFEditBox ).
   Returns
   Index of acroform object in document.                            */
ppInt32 PDFAcroEditBoxInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                  TPDFEditBox EditBox );



/* Description
   Sets PushButton on Page Content for action control.
   Parameters
   Doc :         [in] Destination Document.
   PageIndex :   [in] Page index ( as from 0 ) where PushButton will
                 be stated.
   PushButton :  [in] PushButton Structure which describes all needed
                 fields ( see TPDFPushButton ).
   Returns
   Index of acroform object in document.                              */
ppInt32 PDFAcroPushButtonInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                     TPDFPushButton PushButton );



/* Description
   Sets CheckBox on Page Content for item selection.
   Parameters
   Doc :        [in] Destination Document.
   PageIndex :  [in] Page index ( as from 0 ) where CheckBox will be
                stated.
   CheckBox :   [in] CheckBox Structure which describes all needed
                fields ( see TPDFCheckBox ).
   Returns
   Index of acroform object in document.                             */
ppInt32 PDFAcroCheckBoxInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                   TPDFCheckBox CheckBox );



/* Description
   Sets RadioButton on Page Content for item selection.
   Parameters
   Doc :          [in] Destination Document.
   PageIndex :    [in] Page index ( as from 0 ) where RadioButton will
                  be stated.
   GroupName :    [in] Name of Radio Buttons Group to which
                  RadioButton will be linked
   RadioButton :  [in] RadioButton Structure which describes all
                  needed fields ( see TPDFRadioButton ).
   Returns
   Index of acroform object in document.                               */
ppInt32 PDFAcroRadioButtonInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                      char *GroupName, TPDFRadioButton RadioButton );



/* Description
   Sets ComboBox on Page Content for item selection.
   Parameters
   Doc :        [in] Destination Document.
   PageIndex :  [in] Page index ( as from 0 ) where ComboBox will be
                stated.
   ComboBox :   [in] ComboBox Structure which describes all needed
                fields ( see TPDFComboBox ).
   See Also
   Array of text items(Opt) in ComboBox structure will be filled
   with function PDFAcroItemsBoxAdd.
   Returns
   Index of acroform object in document.                             */
ppInt32 PDFAcroComboBoxInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                   TPDFComboBox ComboBox );



/* Description
   Sets ListBox on Page Content for item selection.
   Parameters
   Doc :        [in] Destination Document.
   PageIndex :  [in] Page index ( as from 0 ) where ListBox will be
                stated.
   ListBox :    [in] ListBox Structure which describes all needed
                fields ( see TPDFListBox ).
   See Also
   Array of text items(Opt) in ListBox structure will be filled
   with function PDFAcroItemsBoxAdd.
   Returns
   Index of acroform object in document.                            */
ppInt32 PDFAcroListBoxInDocument( PDFDocHandle Doc, ppInt32 PageIndex,
                                  TPDFListBox ListBox );



/* Description
   Adds Text Item to items array of ComboBox or ListBox.
   Parameters
   Doc :              [in] Destination document.
   AcroObjectIndex :  [in] Index of acroform object in document ComboBox
                      or ListBox.
   String :           [in] Text string for addition in items array
   ItemsBox :         [in] ComboBox or ListBox Structure which describes
                      all needed fields ( see TPDFItemsBox ).
   See Also
   PDFAcroComboBoxInDocument, PDFAcroListBoxInDocument
   Returns
   None.                                                                 */
void PDFAcroItemsBoxAdd( PDFDocHandle Doc, ppInt32 AcroObjectIndex,
                         char *String, TPDFItemsBox ItemsBox );



/* Description
   Adds Action OnEvent in Acro Form Object (Control).
   Parameters
   Doc :              [in] Destination document.
   AcroObjectIndex :  [in] Index of acroform object in document.
   Action :           [in] Handle on action for include to trigger.
   Type :             [in] Type of event on Control for Action
   See Also
   TPDFAcroEventType, Actions
   Returns
   None.                                                            */
void PDFAcroObjectAddAction( PDFDocHandle Doc, ppInt32 AcroObjectIndex,
                             PDFActionHandle Action, TPDFAcroEventType Type);



/* Description
   Gets Acro Form Object from Document with Index as Acro Object
   Handle.
   Parameters
   Doc :    [in] Current document.
   Index :  [in] Index of acroform object in document.
   Returns
   Acro Object Handle in PDF Document by Index.                  */
PDFCosHandle PDFAcroCosObjectGet( PDFDocHandle Doc, ppInt32 Index );


/************************************************************************
    Name-Value Acro Fields API
************************************************************************/


/* Description
   Gets number of Acro Objects in Document.
   Parameters
   Doc :  [in] Source Document.
   Returns
   Number of Acro Objects in Document, integer number. */
ppUns32     PDFAcroGetCount( PDFDocHandle Doc );



/* Description
   Gets Full name of Acro Form Object in Document according to
   Key(Index).
   Parameters
   Doc :  [in] Source Document.
   Key :  [in] Key as from 0. Object number in AcroForm.
   Returns
   Text string ( where partial names are separated by point ).
   See Also
   PDFAcroSetNameByKey                                         */
char       *PDFAcroGetNameByKey( PDFDocHandle Doc, ppUns32 Key );




/* Description
   Gets Value of Acro Form Object in Document according to
   Key(Index).
   Parameters
   Doc :  [in] Source Document.
   Key :  [in] Key as from 0. Object number in AcroForm.
   Returns
   Text string
   See Also
   PDFAcroSetValueByKey                                    */
char       *PDFAcroGetValueByKey( PDFDocHandle Doc, ppUns32 Key );



/* Description
   Gets Type of Acro Form Object in Document according to
   Key(Index).
   Parameters
   Doc :  [in] Source Document.
   Key :  [in] Key as from 0. Object number in AcroForm.
   Returns
   Typed value, for example PushButton, ComboBox, etc.
   See Also
   PDFAcroType                                            */
TPDFAcroType PDFAcroGetTypeByKey( PDFDocHandle Doc, ppUns32 Key );




/* Description
   Gets Key(Index) of Acro Form Object in Document according to
   Full Name.
   Parameters
   Doc :   [in] Source Document.
   Name :  [in] Full Name of Acro Object, where partial names are
           separated by point, for example 'button.2'
   Returns
   Integer, Index of object( as from 0 ), -1 when failure.
   See Also
   PDFAcroGetNameByKey                                            */
ppInt32     PDFAcroGetKeyByName( PDFDocHandle Doc, char *Name );



/* Description
   Gets Value of Acro Form Object in Document according to its
   Full Name.
   Parameters
   Doc :   [in] Source Document.
   Name :  [in] Full Name of Acro Object, where partial names are
           separated by point, for example 'button.2'
   Returns
   Text string
   See Also
   PDFAcroSetValueByName                                          */
char       *PDFAcroGetValueByName( PDFDocHandle Doc, char *Name );



/* Description
   Gets Type of Acro Form Object in Document according to its
   Full Name.
   Parameters
   Doc :   [in] Source Document.
   Name :  [in] Full Name of Acro Object, where partial names are
           separated by point, for example 'button.2'
   Returns
   Typed value, for example PushButton, ComboBox, etc.
   See Also
   PDFAcroType                                                    */
TPDFAcroType PDFAcroGetTypeByName( PDFDocHandle Doc, char *Name );



/* Description
   Gets Key(Index) of Acro Form Object in Document according to
   its Value.
   Parameters
   Doc :    [in] Source Document.
   Value :  [in] Value of Acro Object, text string.
   Returns
   Integer, Index of object( as from 0 ), -1 when failure.
   See Also
   PDFAcroGetValueByKey                                         */
ppInt32     PDFAcroGetKeyByValue( PDFDocHandle Doc, char *Value );



/* Description
   Gets Full name of Acro Form Object in Document according to
   its Value.
   Parameters
   Doc :    [in] Source Document.
   Value :  [in] Value of Acro Object, text string.
   Returns
   Text string ( where partial names are separated by point ).
   See Also
   PDFAcroSetNameByValue                                       */
char       *PDFAcroGetNameByValue( PDFDocHandle Doc, char *Value );



/* Description
   Gets Type of Acro Form Object in Document according to its
   Value.
   Parameters
   Doc :    [in] Source Document.
   Value :  [in] Value of Acro Object, text string.
   Returns
   Typed value, for example PushButton, ComboBox, etc.
   See Also
   PDFAcroType                                                */
TPDFAcroType PDFAcroGetTypeByValue( PDFDocHandle Doc, char *Value );



/* Description
   Sets name of Acro Form Object in Document according to
   Key(Index).
   Parameters
   Doc :   [in] Source Document.
   Key :   [in] Key as from 0. Object number in AcroForm.
   Name :  [in] Partial Name of Acro Object.
   Returns
   None.
   See Also
   PDFAcroGetNameByKey                                    */
void PDFAcroSetNameByKey( PDFDocHandle Doc, ppUns32 Key, char *Name );



/* Description
   Sets name of Acro Form Object in Document according to Full
   Name.
   Parameters
   Doc :      [in] Source Document.
   OldName :  [in] Full Name of Acro Object, where partial names
              are separated by point, for example 'button.2'
   NewName :  [in] Partial Name of Acro Object.
   Returns
   None.                                                         */
void PDFAcroSetNameByName( PDFDocHandle Doc, char *OldName, char *NewName );



/* Description
   Sets name of Acro Form Object in Document according to its
   Value.
   Parameters
   Doc :    [in] Source Document.
   Value :  [in] Value of Acro Object, text string.
   Name :   [in] Partial Name of Acro Object.
   Returns
   None.
   See Also
   PDFAcroGetNameByValue                                      */
void PDFAcroSetNameByValue( PDFDocHandle Doc, char *Value, char *Name );



/* Description
   Sets Value of Acro Form Object in Document according to
   Key(Index).
   Parameters
   Doc :    [in] Source Document.
   Key :    [in] Key as from 0. Object number in AcroForm.
   Value :  [in] Value of Acro Object, text string.
   Returns
   None.
   See Also
   PDFAcroGetValueByKey                                    */
void PDFAcroSetValueByKey( PDFDocHandle Doc, ppUns32 Key, char *Value );




/* Description
   Sets Value of Acro Form Object in Document according to its
   Name.
   Parameters
   Doc :    [in] Source Document.
   Key :    [in] Key as from 0. Object number in AcroForm.
   Value :  [in] Value of Acro Object, text string.
   Returns
   None.
   See Also
   PDFAcroGetValueByName                                       */
void PDFAcroSetValueByName( PDFDocHandle Doc, char *Name, char *Value );




/* Description
   Sets New Value of Acro Form Object in Document according to
   its Old Value.
   Parameters
   Doc :       [in] Source Document.
   OldValue :  [in] Old Value of Acro Object, text string.
   NewValue :  [in] New Value of Acro Object, text string.
   Returns
   None.                                                       */
void PDFAcroSetValueByValue( PDFDocHandle Doc, char *OldValue, char *NewValue );
/************************************************************************/
#ifdef __cplusplus
}
#endif

#endif /* #ifndef _H_VSACROOBJECTS */
