/*
 * PeakSeq
 * Version 1.01
 * Paper by Joel Rozowsky, et. al.
 * Coded in C by Theodore Gibson.
 * window.h
 * This module deals with everything that differs between windows.
 */


#ifndef FILTER_WINDOW_H
#define FILTER_WINDOW_H


#include "filter.h"
#include "analyze.h"
#include "io.h"


//----------------------------------------------------------------------------
// PUBLIC STRUCTURE DEFINITIONS
//----------------------------------------------------------------------------
/*
 * Window is a pointer to the structure that stores all information about a
 * 	single window.
 */
typedef struct window* Window;

/*
 * ActThresh is a pointer to the structure used while finding the number of
 * 	actual peaks at all the different thresholds.  It stores all information
 * 	about a threshold that differs from the other thresholds.
 */
typedef struct actThresh* ActThresh;


//----------------------------------------------------------------------------
// PUBLIC FUNCTION PROTOTYPES
//----------------------------------------------------------------------------
/*
 * This function allocates memory for and intializes a new double array of
 * 	window structures.  The first index in the array indicates chromosome
 * 	number and the second index indicates the window number on that
 * 	chromosome.
 * Outputs a pointer to the new double array.
 */
Window** newWA( void );

/*
 * This function computes and stores all mappability fractions to a double
 * 	array of window structures.
 * map: a pointer to the file with the data about the mapability fractions.
 * wa: a pointer to the double array of window structures in which the data
 * 	will be stored.
 */
void getMapFracts( FILE* map, Window** wa );

/*
 * This function computes and stores the number of reads in each window to the
 * 	single array of window structures for this chromosome.  It also stores all
 * 	the starting positions of reads into a starts structure.
 * eland: a pointer to the eland file with the sample data.
 * wa: a pointer to the single array of window structures for this chromosome.
 * starts: at the end of this function, this structure will contain an array
 * 	the starting positions of all the reads in this file.
 */
void getNreadsStarts( FILE* eland, Window* wa, Starts starts );

/*
 * This function allocates memory for a new actThresh array.
 * Outputs a pointer to the array.
 */
ActThresh* newActThreshArray( void );

/*
 * This function goes through an sgr file and counts the number of peaks in
 * 	each window at each threshold.  These counts are stored in the window
 * 	double array. It also stores the peak locations in the actThreshes
 * 	structure. Since I purposely neglected to allocate storage for the counts
 * 	array when the window array was initially created as this saves
 * 	unneccessary memory from being allocated, I initialize these fields within
 * 	this function.
 * sgr: a pointer to the file containing the data.
 * wa: a pointer to the single array of window structures for this chromosome.
 * ata: a pointer to the array containing all information that differs between
 * 	thresholds including peak locations but not including the count at each
 * 	threshold for each window (which is stored in the window array).
 */
void getActCounts( FILE* sgr, Window* wa, ActThresh* ata );

/*
 * This function returns the mappability fraction of a window.
 * w: a pointer to the window containing the requested data.
 * Outputs the mappability fraction.
 */
double getWMapFract( Window w );

/*
 * This function returns the number of reads in a window.
 * w: a pointer to the window containing the requested data.
 * Outputs the number of reads in the window.
 */
int getWNReads( Window w );

/*
 * This function returns the number of actual peaks at a threshold in a
 * 	window.
 * w: a pointer to the structure containing data about the window.
 * thresh: the threshold.
 * Outputs the count.
 */
int getWCount( Window w, const int thresh );

/*
 * This function finds the actual peaks in a window and outputs them to a
 * 	function that computes and prints statistics about these peaks.
 * out: a pointer to the file to which the data should be outputted.
 * at: a pointer to the actThresh structure containing the peaks to be
 * 	printed.
 * sample_starts: a pointer to the structure containing an array of start
 * 	positions of reads from the sample Eland file.
 * control_starts: a pointer to the structure containing an array of start
 * 	positions of reads from the control input file.
 * data: a pointer to the structure containing an array of all peaks outputted
 * 	to the OUTPUT file.
 * pCache: a cache array of all previously-computed powers of 0.5.
 * factCache: a cache array of all previously-computed factorials.
 * last_startp: a pointer to the field containing the start position of the
 * 	most recently found peak.
 * last_stopp: a pointer to the field containing the stop position of the most
 * 	recently found peak.
 * w_start: the first nucleotide position in the window.
 * w_stop: the last nucleotide position in the window.
 * scaling_factor: the scaling factor used for tag (filter) output.
 * max_fact: the maximum factorial that can be computed on this computer.
 * max_exp: the maximum exponent to which 0.5 can be raised to on this
 * 	computer.
 * cname: the string in the format chr1 or chrX.
 */
void findPeaks( FILE* out, ActThresh at, Starts sample_starts,
		Starts control_starts, Data data, long double* pCache,
		long double* factCache, int* last_startp, int* last_stopp,
		const int w_start, const int w_stop, const long double scaling_factor,
		const int max_fact, const int max_exp, const String cname );

/*
 * This function frees all memory associated with an array of actThresh
 * 	structures.
 * ata: a pointer to the array to be freed.
 */
void freeActThreshArray( ActThresh* ata );

/*
 * This function frees all the counts arrays for a single chromosome's window
 * 	structures.
 * wa: the single array of window structures for the chromosome whose counts
 * 	arrays are to be freed.
 */
void freeCounts( Window* wa );

/*
 * This function frees all memory in the double array of window structures.
 * wa: a pointer to the double array to be freed.
 */
void freeWA( Window** wa );


#endif	/* WINDOW_H */
