/*
 * PeakSeq
 * Version 1.01
 * Paper by Joel Rozowsky, et. al.
 * Coded in C by Theodore Gibson.
 * filter.h
 * This module deals with output to the OUTPUT file.
 */


#ifndef FILTER_H
#define FILTER_H


#include "analyze.h"
#include "io.h"


//----------------------------------------------------------------------------
// PUBLIC STRUCTURE DEFINITION
//----------------------------------------------------------------------------
/*
 * Starts is a pointer to the structure that stores a dynamic array of all the
 * 	starting positions of reads.
 */
typedef struct starts* Starts;


//----------------------------------------------------------------------------
// PUBLIC FUNCTION PROTOTYPES
//----------------------------------------------------------------------------
/*
 * This function creates a new array of num slots.  It is to store
 * 	intermediate results of factorial and power calculations so these
 * 	calculations need only be performed once.
 * num: the highest number for which the calculation might be needed.
 * Outputs a pointer to the new array.
 */
long double* newCache( const int num );

/*
 * This function finds the maximum factorial that can be computed by this
 * 	system.  No more than 500 MB will be allocated.
 * Outputs the highest number whose factorial can be computed.
 */
int getMaxFact( void );

/*
 * This function finds the maximum power of 0.5 (p) that can be computed by
 * 	this system.  No more than 500 MB will be allocated.
 * Outputs the highest exponent such that 0.5^e can be computed.
 */
int getMaxExp( void );

/*
 * This function allocates memory for a new starts structure.
 * Outputs a pointer to the new structure.
 */
Starts newStarts( void );

/*
 * This function inserts a position into a starts array.
 * starts: a pointer to the structure in which the position should be
 * 	inserted.
 * pos: the position.
 */
void insertStart( Starts starts, const int pos );

/*
 * This function sorts the positions in a starts structure by position.
 * starts: a pointer to the structure whose array is to be sorted.
 */
void sortStarts( Starts starts );

/*
 * This function stores the start positions of all reads in a file to a
 * 	dynamic array of positions.
 * input: a pointer to the file containing the data.
 * wa: a pointer to the single array of window structures for this chromosome.
 * starts: at the end of this function, this structure will contain an array
 * 	of all the starting positions of reads in this file.
 */
void getStarts( FILE* input, Starts starts );

/*
 * This function computes the scaling factor for a given set of data.
 * sample_starts: the sorted starting positions of the reads in the sample
 * 	Eland file.
 * control_starts: the sorted starting positions of the reads in the input
 * 	(control) file.
 * bin_size: the size of a single bin.
 * Outputs the scaling factor.
 */
long double getScalingFactor( Starts control_starts, Starts sample_starts,
		const int bin_size );

/*
 * This function prints information about the number of reads in a peak to the
 * 	output file indicated in filter_hits_PolII.pl.
 * out: the FINAL file to which this data should be outputted.
 * sample_starts: a pointer to the structure containing an array of all
 * 	positions from the sample Eland file.
 * control_starts: a pointer to the structure containing an array of all
 * 	positions from the control input file.
 * data: a pointer to the structure containing an array of all peaks outputted
 * 	to the OUTPUT file.
 * pCache: a cache array of all previously-computed powers of 0.5.
 * factCache: a cache array of all previously-computed factorials.
 * start: the start position of the peak.
 * stop: the stop position of the peak.
 * scaling_factor: a scaling factor.
 * max_fact: the maximum factorial that can be computed on this computer.
 * max_exp: the maximum exponent to which 0.5 can be raised to in order to
 * 	output a non-zero answer on this computer.
 * cname: the string representation of this chromosome (chr1 to chrM).
 */
void fprintfTagData( FILE* out, Starts sample_starts, Starts control_starts,
		Data data, long double* pCache, long double* factCache,
		const int start, const int stop, const long double scaling_factor,
		const int max_fact, const int max_exp, const String cname );

/*
 * This function frees all memory associated with a starts structure.
 * starts: a pointer to the structure to be freed.
 */
void freeStarts( Starts starts );


#endif	/* FILTER_H */
