/*
 * PeakSeq
 * Version 1.01
 * Paper by Joel Rozowsky, et. al.
 * Coded in C by Theodore Gibson.
 * analyze.h
 * This module deals with BH correction for multiple hypothesis testing.
 */


#ifndef ANALYZE_H
#define ANALYZE_H


#include "io.h"


//----------------------------------------------------------------------------
// PRIVATE STRUCTURE DEFINITION
//----------------------------------------------------------------------------
/*
 * Data is a pointer to the structure that stores a dynamic array of datum
 * 	structures, each of which contains information about a single peak.
 */
typedef struct data* Data;


//----------------------------------------------------------------------------
// PUBLIC FUNCTION PROTOTYPES
//----------------------------------------------------------------------------
/*
 * This function allocates memory for a new data structure.
 * Outputs a pointer to the new structure.
 */
Data newData( void );

/*
 * This function allocates memory for and initializes a new datum structure
 * 	and inserts it into the array.
 * data: a pointer to the structure containing the dynamic array of datum
 * 	structures.
 * cname: the string representation of the chromosome on which the peak is
 * 	located.
 * start: the start position of the peak.
 * stop: the stop position of the peak.
 * sample_tags: the number of reads from the Eland file located in the peak.
 * control_tags: the number of reads from the control input file located in
 * 	the peak.
 * enrich: the enrichment at this peak.
 * excess: the excess at this peak.
 * pval: the p-value of this peak.
 * zscore: the z-score of the peak for peaks with very small p-values.
 * 	Otherwise, 0.
 */
void insertData( Data data, const String cname, const int start,
		const int stop, const int sample_tags, const int control_tags,
		const long double enrich, const int excess, const long double pval,
		const long double zscore );

/*
 * This function sorts the data array in order of p-value.
 * data: a pointer to the structure containing the array to be sorted.
 */
void sortData( Data data );

/*
 * This function prints an already-sorted data structure to a file.  Before
 * 	doing so it sorts all 0 p-values by z-score.  It also frees all memory
 * 	associated with the data structure.
 * out: a pointer to the file to which the data should be outputted.
 * data: a pointer to the structure containing the peaks to be outputted.
 */
void fprintfData( FILE* out, Data data );

/*
 * This function frees all memory for a data structure.  It is only called if
 * 	the program exits prematurely before calling fprintfData().
 * data: a pointer to the structure to be freed.
 */
void freeData( Data data );


#endif	/* ANALYZE_H_ */
